function reorderMultipleGenes(rootDirectory, datasetName, treeFileName, recombinationType)
    
    delimiter = getDelimiter();
    rootToResults = [rootDirectory datasetName 'Results' delimiter];
    % 'rootToResults' is a root to a directory that contains multiple
    % sub-folders, whose names are the names of the genes that have been
    % analysed, each containing fastGEAR results for that gene.
    
    rootToSummaries = [rootDirectory datasetName 'Summaries' delimiter];
    % 'rootToSummaries' is a root to a directory to which various summaries
    % will be saved by this function.
    % 
    % 'treeFileName' is the name of file containing the phylogeny for the
    % strains in the data set. It is used here to extract the list of
    % strain names. Such a file much be available in the summary directory.
    
    if nargin == 3
        recombinationType = 'both';
    end
    
    delimiter = getDelimiter;
    treeFileName = [rootToSummaries delimiter treeFileName];
    
    % If there is a file named 'geneNames.mat' in the summary folder, then
    % load it, otherwise create the file by extracting the gene names from
    % the sub-folders of the result folder.
    if exist([rootToSummaries 'geneNames.mat'])
        load([rootToSummaries 'geneNames.mat']); % geneNames
        nGenes = length(geneNames);
    else
        fNames = dir(rootToResults);
        fNames = fNames(3:end); % Remove '.' and '..'
        nGenes = length(fNames);
        geneNames = cell(nGenes,1);
        for geneIndex = 1:nGenes
            geneNames{geneIndex} = fNames(geneIndex).name;
        end
        save([rootToSummaries 'geneNames.mat'],'geneNames');
    end
    
    
    % Get here the global names from the tree...
    globalNames = getGlobalNames(treeFileName);
    nStrains = length(globalNames);
    % These names are 'global' in the sense that names in any
    % individual alignment (i.e. 'locally') must be a subset of these.
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Combine results for core genes in a specific order %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if (1==1)
        
        collectedPopStructures = cell(1,nGenes);
        collectedSnpPositions = cell(1,nGenes);
        collectedGeneLengths = zeros(1,nGenes);
        collectedPartitions = zeros(length(globalNames), nGenes, 'uint8');
        
        % Collect population structures
        for geneIndex = 1:nGenes
            
            nameNow = geneNames{geneIndex};
            % Load fastGEAR results for this gene.
            load([rootToResults delimiter nameNow delimiter nameNow '_res.mat']); % uData
            
            localNames = uData.strainLabels;
            
            nSnps = length(uData.snpPositions);
            
            popStructure = zeros(length(globalNames), nSnps, 'uint8');
            [~,localToGlobalMapping] = ismember(localNames, globalNames);
            
            popStructureWithBothRecombinations = formulatePopStructureToPlot(uData, recombinationType);
            
            popStructure(localToGlobalMapping,:) = popStructureWithBothRecombinations;
            
            % compress by removing identical columns
            [~,pos1,pos2] = unique(popStructure', 'rows');
            changePoints = find(pos2(2:end) - pos2(1:end-1) ~= 0);
            positionsToKeep = unique([changePoints; changePoints+1; 1; nSnps]);

            collectedPopStructures{geneIndex} = popStructure(:, positionsToKeep);
            collectedSnpPositions{geneIndex} = uData.snpPositions(positionsToKeep);
            collectedGeneLengths(geneIndex) = uData.totalSequenceLength;
            collectedPartitions(localToGlobalMapping,geneIndex) = uData.groupedPartition;
            
        end
        
        % Reorder colors to minimize average entropy of color distributions
        collectedPopStructures = reorderColorsMinEntropy(collectedPopStructures, collectedPartitions, collectedGeneLengths, collectedSnpPositions);
        collectedMostCommonClusters = zeros(1,nGenes);
        for geneIndex = 1:nGenes
            % Find out the most common color for the gene
            collectedMostCommonClusters(geneIndex) = calculateMostCommonColor(collectedPopStructures{geneIndex}, collectedSnpPositions{geneIndex}, collectedGeneLengths(geneIndex));
        end
        save([rootToSummaries delimiter 'collectedPopStructures.mat'],'collectedPopStructures', 'collectedSnpPositions', 'collectedGeneLengths', 'collectedMostCommonClusters');
        save([rootToSummaries delimiter 'globalNames.mat'],'globalNames');
        
        snpCounts = cellfun(@(x)length(x), collectedSnpPositions);
        totalNSnps = sum(snpCounts);
        combinedPopStructure = zeros(length(globalNames), totalNSnps, 'uint8');
        combinedSnpPositions = zeros(1,totalNSnps);
        currentNSnps = 0;
        currentLength = 0;
        for geneIndex = 1:nGenes
            combinedPopStructure(:,currentNSnps+1 : currentNSnps+snpCounts(geneIndex)) = ...
                collectedPopStructures{geneIndex};

            combinedSnpPositions(currentNSnps+1 : currentNSnps+snpCounts(geneIndex)) = ...
                collectedSnpPositions{geneIndex} + currentLength;

            currentNSnps = currentNSnps + snpCounts(geneIndex);
            currentLength = currentLength + collectedGeneLengths(geneIndex);

        end
        totalCoreLength = currentLength;
        maxClusterIndex = max(collectedPartitions(:));
        save([rootToSummaries delimiter 'combinedPopStructure.mat'],'combinedPopStructure', 'combinedSnpPositions', 'totalCoreLength', 'maxClusterIndex', 'collectedGeneLengths');
    end
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Create segmentations of the strains, variable 'allSegments' %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if (1==1)
        load([rootToSummaries delimiter 'combinedPopStructure.mat']) % combinedPopStructure, combinedSnpPositions, totalCoreLength, globalNames, maxClusterIndex, collectedGeneLengths
        
        popStructureToPlot = combinedPopStructure;
        nLineages = double(maxClusterIndex);
        totalSequenceLength = totalCoreLength;
        snpPositions = combinedSnpPositions;
        
        popStructureToPlot(popStructureToPlot == 0) = nLineages + 2;
        
        allSegments = cell(1,nStrains);
        for strainIndex = 1:nStrains
            % Find segments of the strain at this position
            
            if ~isempty(strainIndex)
                % Identify segments
                segments = identifySegments(popStructureToPlot, strainIndex, snpPositions, totalSequenceLength);
                allSegments{strainIndex} = segments;
            end
        end
        save([rootToSummaries delimiter 'allSegments.mat'],'allSegments')
    end
    
    
    if (1==0)
        % This creates a 'finalPopStructure' which compresses the
        % population structure even further, by summarizing each block of
        % 1000 bp with a single element. (can aid plotting very many genes
        % side by size)
        load([rootToSummaries delimiter 'combinedPopStructure.mat']); % combinedPopStructure, combinedSnpPositions, totalCoreLength, globalNames, maxClusterIndex, collectedGeneLengths
        load([rootToSummaries delimiter 'allSegments.mat']); % allSegments
        
        maxLabel = max(combinedPopStructure(:));
        
        nStrains = length(allSegments);
        blockSize = 1000;
        nBlocks = ceil(totalCoreLength ./ blockSize);
        finalPopStructure = zeros(nStrains,nBlocks,'uint8');
        for strainIndex = 1:nStrains
            
            % Go through all strains, and find the block average
            wholeSequence = zeros(1,totalCoreLength);
            segmentsNow = allSegments{strainIndex};
            for segmentIndex = 1:size(segmentsNow,1)
                wholeSequence(segmentsNow(segmentIndex,1):segmentsNow(segmentIndex,2)) = ...
                    segmentsNow(segmentIndex,3);
            end
            
            for blockIndex = 1:nBlocks
                first = (blockIndex-1) .* blockSize + 1;
                last = min(blockIndex .* blockSize, totalCoreLength);
                [~,mostCommonLabel] = max(histc(wholeSequence(first:last),1:maxLabel));
                finalPopStructure(strainIndex,blockIndex) = mostCommonLabel;
            end
        end
        save([rootToSummaries delimiter 'finalPopStructure.mat'], 'finalPopStructure');
    end
    
end



function mostCommonCluster = calculateMostCommonColor(popStructure, snpPositions, totalSequenceLength)
    
    if any(snpPositions > totalSequenceLength)
        error('Inconsistent SNP positions');
    end
    
    nSnps = length(snpPositions);
    
    segments = identifySegments(1:nSnps, 1, snpPositions, totalSequenceLength);
    segmentLengths = segments(:,2) - segments(:,1) + 1;
    maxLabel = max(max(popStructure));
    
    colorAmounts = zeros(1,maxLabel);
    for colorIndex = 1:maxLabel
        colorAmounts(colorIndex) = sum((popStructure==colorIndex) * segmentLengths);
    end
    colorAmounts = colorAmounts ./ (totalSequenceLength .* size(popStructure,1));
    
    [~,mostCommonCluster] = max(colorAmounts);
end




function globalNames = getGlobalNames(treeFileName)
    % If treeFileName is a .tre, then read the names of the strains from
    % the treeleaves.
    %
    % Otherwise treeFileName must be a .txt. In this case, it contains the
    % names of the strains.
    
    if isequal(treeFileName(end-3:end), '.tre')
        tr = phytreeread(treeFileName);
        tr = reroot(tr);
        globalNames = get(tr,'LeafNames');
        
    elseif isequal(treeFileName(end-3:end), '.txt')
        globalNames = cell(30000,1);
        nNames = 0;
        fid = fopen(treeFileName);
        
        line = fgetl(fid);
        while ischar(line)
            nNames = nNames + 1;
            globalNames{nNames} = line;
            line = fgetl(fid);
        end
        globalNames = globalNames(nNames:-1:1);
        % If the names are given in the file in the order they are to
        % appear in the plot (from top to bottom), then globalNames must be
        % reversed. In this way this matches the globalNames that are
        % obtained from the tree, as also they appear eventually in the
        % reversed order in the plot.
    end
end
