function uData = clearNonSignificantRecentRecombinations(outputFile)

    load(outputFile); % uData
    load(uData.snpDataFileName); % snpData
    
    if length(uData.snpPositions) == 1
        uData.popStructureCleaned = uData.updatedPopStructure;
        uData.recentRecombinationSignificances = [];
        return
    end
    
    forLineages = true;
    ancestralSeqTable = createAncestralSequences(uData, forLineages); % Lineage ancestral sequences
    nLineages = max(uData.groupedPartition);


    nStrains = length(uData.partition);

    
    uData.popStructureCleaned = uData.updatedPopStructure;
    uData.recentRecombinationSignificances = cell(nStrains,1);
    
    for strainIndex = 1:nStrains
        
        segments = identifySegments(uData.updatedPopStructure, strainIndex, uData.snpPositions, uData.totalSequenceLength);
        nSegments = size(segments,1);
        
        homeLineage = uData.groupedPartition(strainIndex);
        homeSegmentIndices = find(segments(:,3) == homeLineage);
        homeSegments = segments(homeSegmentIndices,:);
        
        segments(homeSegmentIndices,:) = [];
        recSegments = segments; % Recombinant segments
        clear segments;
        
        if size(recSegments,1) > 0
            
            % Compute the SNP differences between the target strain and lineages.
            homeSnpDifferences = formulateSnpDifferenceBetweenStrainAndLineage(ancestralSeqTable, homeLineage, snpData, strainIndex, uData.totalSequenceLength, uData.snpPositions);
            
            % Compute home lineage SNP counts
            homeSegmentLengths = homeSegments(:,2) - homeSegments(:,1) + 1;
            
            % Compute home lineage SNP density in home segments
            nHomeSegments = size(homeSegments,1);
            nSnpsInHomeSegments = zeros(nHomeSegments,1);
            for segmentIndex = 1:nHomeSegments
                
                first = homeSegments(segmentIndex, 1);
                last = homeSegments(segmentIndex, 2);
                nSnpsInHomeSegments(segmentIndex) = length(find((homeSnpDifferences.snpPositions >= first) & (homeSnpDifferences.snpPositions <= last)));
                
            end
            homeCounts = [sum(nSnpsInHomeSegments) sum(homeSegmentLengths)-sum(nSnpsInHomeSegments)];
            
            
            recSegmentLengths = recSegments(:,2) - recSegments(:,1) + 1;
            nRecSegments = size(recSegments,1);
            nSnpsInRecSegments = zeros(size(recSegments,1),1);
            recCounts = zeros(nRecSegments, 2);
            for segmentIndex = 1:nRecSegments
                
                first = recSegments(segmentIndex,1);
                last = recSegments(segmentIndex,2);
                nSnpsInRecSegments(segmentIndex) = length(find((homeSnpDifferences.snpPositions >= first) & (homeSnpDifferences.snpPositions <= last)));
                
                recCounts(segmentIndex, :) = [nSnpsInRecSegments(segmentIndex) last-first+1 - nSnpsInRecSegments(segmentIndex)];
                
            end
            
            logBf = computeLogBfForRecombinations(recCounts, homeCounts, uData.priorCounts);
            isCombined = zeros(nRecSegments,1);
            
            while any(logBf <= 0) %uData.ancestralLogBfThreshold)
                %disp(num2str(logBf'));
                
                [~,minIndex] = min(logBf);
                
                % Combine recombination with minIndex with homeCluster.
                isCombined(minIndex) = 1;
                homeCounts = homeCounts + recCounts(minIndex,:);
                
                % Recompute logBf for all segments that are not combined
                logBf(isCombined == 0) = computeLogBfForRecombinations(recCounts(isCombined==0,:), homeCounts, uData.priorCounts);
                
                % Some large default value for all segments that are already combined
                logBf(isCombined == 1) = 1e5;
                
            end
            
        else
            
            % There were no recombinations to check
            isCombined = [];
            logBf = [];
            
        end
        
        
        % Remove all recombinations that have been combined with the home
        % lineage.
        cleanedStrainStructure = cleanStrainStructure(uData.updatedPopStructure(strainIndex,:), recSegments(isCombined==1,:), uData.snpPositions, homeLineage);
        uData.popStructureCleaned(strainIndex,:) = cleanedStrainStructure;
        uData.recentRecombinationSignificances{strainIndex} = logBf(isCombined == 0);
        
    end
    
    save(outputFile, 'uData');
    
end



function cleanedStrainStructure = cleanStrainStructure(origStructure, segmentsToClean, snpPositions, homeLineage)
    
    cleanedStrainStructure = origStructure;
    for i = 1:size(segmentsToClean,1)
        first = segmentsToClean(i,1);
        last = segmentsToClean(i,2);
        cleanedStrainStructure((snpPositions >= first) & (snpPositions <= last)) = homeLineage;
    end
    
end



function logBfForKeepingSeparate = computeLogBfForRecombinations(segmentCounts, homeCounts, priorCounts)
    
    nRecombinations = size(segmentCounts,1);
    logBfForKeepingSeparate = zeros(nRecombinations,1);
    
    for segmentIndex = 1:nRecombinations
        logBfForKeepingSeparate(segmentIndex) = computeBinomialLogml([segmentCounts(segmentIndex,:); homeCounts], priorCounts);
    end
    
end



function data = formulateSnpDifferenceBetweenStrainAndLineage(ancestralSeqTable, lineageIndex, snpData, strainIndex, totalSequenceLength, snpPositions)    
    
    % "data" has two fields: "totalSequenceLength" and
    % "snpPositions", where the "snpPositions" now tells
    % the locations of the SNPs between the lineages. 
    
    % "ancestralSeqTable" contains ancestral sequences. If some element is
    % equal to 9, then it means that at this column every strain in the
    % lineage has a recent recombination, and, so, nothing is known of the
    % ancestral sequences. A between lineage SNP will not be marked at this
    % columns.
    
    seqForComparison = [ancestralSeqTable(lineageIndex, :); snpData(strainIndex,:)];
    ancestralSeqAsCell = num2cell(seqForComparison, 1);
    nAlleles = cellfun(@(x) length(unique(x)), ancestralSeqAsCell);
    data.totalSequenceLength = totalSequenceLength;
    data.snpPositions = snpPositions;
    
    eitherIsUnknown = cellfun(@(x) any(x==9 | x==0), ancestralSeqAsCell);
    nAlleles(eitherIsUnknown) = 1; % Don't count these as SNPs either.
    
    data.snpPositions(nAlleles==1) = []; % Remove positions not SNPs between ancestral sequences. 
    
end



function logBfForKeepingSeparate = computeBinomialLogml(counts, priorCounts)
    % "counts" must be a 2*2 table, such that rows represent counts [n1,n0]
    % in two different clusters.
    % 
    % "priorCounts" is [alpha, beta], which are the parameters of the beta 
    % distribution for SNP frequency.
    
    counts(3,:) = sum(counts); % Last row is counts for the combined cluster
    
    priorCounts = priorCounts([1 1 1],:);
    
    logml = ...
        gammaln(sum(priorCounts,2)) - sum(gammaln(priorCounts),2) ...
        + sum(gammaln(priorCounts + counts),2) ...
        - gammaln(sum(priorCounts + counts,2));
    
    logBfForKeepingSeparate = logml(1) + logml(2) - logml(3);
    
end
