function [lengthPrivate, lengthInvestigated] = computeProportionPrivateAncestry(nameToRun, globalNames, globalPartition, type)
    % type=='private' or 'shared' or 'both'
    
    %codeDir = '/triton/ics/scratch/mi/pemartti/Brat4/source/sparc_analysis/';
    resultDir = '/triton/ics/scratch/mi/pemartti/Brat4/sparc_results/';
    
    outputFile = [resultDir 'results/' nameToRun '/' nameToRun '_res.mat'];
    load(outputFile); % uData
    
    regionLengths = computeRegionsSurroundingSnps(uData.snpPositions, uData.totalSequenceLength);
    
    popStructure = formulatePopStructureWithBothRecombinations(uData);
    
    localNames = extractShortLabels(uData.strainLabels);
    
    
    if isequal(type,'private') || isequal(type,'both')
        % Compute the proportion of private ancestry (a vector)
        [lengthPrivate, lengthInvestigated] = ...
            computeAmountPrivateAncestry(localNames, regionLengths, ...
            popStructure, globalNames, globalPartition);

        resultFileName = [resultDir 'results/' nameToRun '/' nameToRun '_private_ancestry.mat'];
        save(resultFileName, 'lengthPrivate', 'lengthInvestigated');
    end
    
    
    if isequal(type,'shared') || isequal(type,'both')
        % Compute the proportion of shared ancestry (a matrix)
        [lengthShared, lengthCompared] = computeAmountSharedAncestry(...
            localNames, regionLengths, popStructure, globalNames);
        resultFileName = [resultDir 'results/' nameToRun '/' nameToRun '_ancestry_shared.mat'];
        save(resultFileName, 'lengthShared', 'lengthCompared');
    end
    
end



function [lengthShared, lengthCompared] = computeAmountSharedAncestry(...
        localNames, regionLengths, popStructure, globalNames)
    
    nStrainsGlobal = length(globalNames);
    totalSequenceLength = sum(regionLengths);
    
    % Many large matrices will be saved. The following conversion saves
    % a little bit of space.
    if totalSequenceLength <= 65535
        lengthShared = zeros(nStrainsGlobal, 'uint16');
        lengthCompared = zeros(nStrainsGlobal, 'uint16');
    else
        lengthShared = zeros(nStrainsGlobal, 'uint32');
        lengthCompared = zeros(nStrainsGlobal, 'uint32');
    end
    
    [~,localToGlobalMapping] = ismember(localNames, globalNames);
    
    for globalIndex1 = 1 : nStrainsGlobal-1
        
        localIndices1 = find(localToGlobalMapping == globalIndex1)';
        
        if ~isempty(localIndices1)
        
            for globalIndex2 = globalIndex1+1 : nStrainsGlobal

                localIndices2 = find(localToGlobalMapping == globalIndex2)';
                
                if ~isempty(localIndices2)
                    
                    nComparisonsTotal = length(localIndices1) .* length(localIndices2);
                    
                    for localIndex1 = localIndices1
                        for localIndex2 = localIndices2
                            
                            equalPositions = popStructure(localIndex1,:) == popStructure(localIndex2,:);
                            
                            lengthShared(globalIndex1, globalIndex2) = ...
                                lengthShared(globalIndex1, globalIndex2) + ...
                                sum(regionLengths(equalPositions));
                            
                            lengthCompared(globalIndex1, globalIndex2) = ...
                                lengthCompared(globalIndex1, globalIndex2) + ...
                                totalSequenceLength;
                            
                        end
                    end
                    
                end
            end
        
        end
    end
    
end



function [lengthPrivate, lengthInvestigated] = computeAmountPrivateAncestry(localNames, ...
    regionLengths, popStructure, globalNames, globalPartition)
    
    nStrainsGlobal = length(globalNames);
    
    lengthPrivate = zeros(nStrainsGlobal,1);
    lengthInvestigated = zeros(nStrainsGlobal,1);
    
    [~,localToGlobalMapping] = ismember(localNames, globalNames);
    localPartition = globalPartition(localToGlobalMapping);
    
    nLineages = max(localPartition);
    lineageSizes = histc(localPartition, 1:nLineages);
    lineageComplementSizes = length(localPartition) - lineageSizes;
    % i:th element tells how many strains there are in all other lineages
    % except the i:th lineage.
    
    for globalIndex = 1 : nStrainsGlobal
        
        localIndices = find(localToGlobalMapping == globalIndex)';
        %globalNameNow = globalNames{globalIndex};
        %localIndicesAlt = find(cellfun(@(x)isequal(x,globalNameNow),localNames))';
        %if ~isequal(localIndices, localIndicesAlt)
        %    keyboard
        %end
        if globalPartition(globalIndex) ~= unique(localPartition(localIndices))
            error('Should not happen');
        end
            
        
        if ~isempty(localIndices)
            
            lengthPrivateReps = zeros(1, length(localIndices));
            % Go through each copy of the gene that this particular strains has
            % in this alignment. The proportion of private origin that is
            % reported is taken to be the average over all the copies.

            for localIndex = localIndices
                
                % Compute the proportion of ancestry that is found only within
                % the strains in the same lineage.
                originsNow = popStructure(localIndex,:);
                lineageNow = localPartition(localIndex);
                
                privateRegionIndices = find(all(popStructure(localPartition ~= lineageNow,:) ~= repmat(originsNow,[lineageComplementSizes(lineageNow), 1]),1));
                lengthPrivateReps(localIndex==localIndices) = sum(regionLengths(privateRegionIndices));
                
            end
            lengthPrivate(globalIndex) = sum(lengthPrivateReps);
            lengthInvestigated(globalIndex) = sum(regionLengths) .* length(localIndices);
        end
    end
    
end



function shortLabels = extractShortLabels(strainLabels)
    
    nStrains = length(strainLabels);
    shortLabels = cell(length(strainLabels),1);
    
    for strainIndex = 1:nStrains
        labelNow = strainLabels{strainIndex};
        underScorePos = find(labelNow == '_');
        shortLabels{strainIndex} = labelNow(1:underScorePos(1)-1);
    end
    
end



function regionLengths = computeRegionsSurroundingSnps(snpPositions, totalSequenceLength)
    
%     % TEST CASE 1
%     snpPositions = [1 4 6];
%     totalSequenceLength = 6;
%     correctLengths = [2 3 1]';
%     % TEST CASE 2
%     snpPositions = [10 20];
%     totalSequenceLength = 30;
%     correctLengths = [15 15]';
%     % TEST CASE 3
%     snpPositions = [1 2 3 4];
%     totalSequenceLength = 4;
%     correctLengths = [1 1 1 1]';
%     % TEST CASE 4
%     snpPositions = [2 3 4];
%     totalSequenceLength = 5;
%     correctLengths = [2 1 2]';
    
    
    nSnps = length(snpPositions);
    
    if nSnps == 1
        % The special case of only one SNP.
        regionLengths = totalSequenceLength;
        return;
    end
    
    distances = snpPositions(2:end) - snpPositions(1:end-1) - 1;
    % i:th distance is the number of bases between SNPs i and i+1;
    
    even = find(mod(distances,2) == 0);
    uneven = find(mod(distances,2) == 1);
    
    % Compute right end points of the regions
    rightEndPoints = zeros(nSnps,1);
    rightEndPoints(even) = snpPositions(even) + distances(even)/2;
    rightEndPoints(uneven) = snpPositions(uneven) + (distances(uneven)+1)/2;
    rightEndPoints(end) = totalSequenceLength;
    
    % Compute left end points of the regions
    leftEndPoints = zeros(nSnps,1);
    leftEndPoints(1+even) = snpPositions(1+even) - distances(even)/2;
    leftEndPoints(1+uneven) = snpPositions(1+uneven) - (distances(uneven)-1)/2;
    leftEndPoints(1) = 1;
    
    regionLengths = rightEndPoints - leftEndPoints + 1;
    
%     if isequal(regionLengths, correctLengths)
%         disp('joo');
%     else
%         disp('ei');
%     end
%     error('KESKEN')
    
end



function popStructure = formulatePopStructureWithBothRecombinations(uData)
    
    popStructure = derivePopStructureFromLineageStructure(uData, 2);
    % The 2nd argument indicates that a population structure based on
    % "lineageStructureCleaned" should be formulated.
    
    % Add all recent recombinations
    nSnps = length(uData.snpPositions);
    lineageStructureWithoutRecentRecombinations = uint8(repmat(uData.groupedPartition, [1 nSnps]));
    recentRecombinationInds = find(lineageStructureWithoutRecentRecombinations ~= uData.popStructureCleaned);
    
    popStructure(recentRecombinationInds) = uData.popStructureCleaned(recentRecombinationInds);
    
end

