function betaHats = hmmBetaHatRecursion(coefs, snpPositions, ...
    preCalculatedTransMatrices, emissionProbs, ...
    initTransitionMatrix, initBetaHat)
% The function performs beta-recursion. The function gets as input 
% parameters for the initial state, as well as the emission probabilities
% at every SNP. Unlike the emission probabilities, and in order to save 
% storage space, the transition probabilities are not pre-calculated but 
% they are calculated inside the function.
%
% "snpPositions" is a vector with positions of the SNPs in the sequence.
% "totalSequenceLength" is the total length of the sequence.
% "priorProbOfNoChangesInSequence" is the prior probability that there is
% not a single state change in the whole sequence.
% The previous three are needed when the transition probabilities are
% calculated. (note that the totalSequenceLength may be larger than the
% the number of SNPs analyzed here, because it is possible that not all 
% SNPs in a data set are analyzed at the same time).
%
% "emissionProbs" is an nStates*(nSnps+1) table with emission 
% probabilities.
% "coefs" is a 1*(nSnps+1) vector.
% In the previous two, the last element is used for specifying the values
% for the N+1:st element. (i.e. when the current last SNP is not the last
% SNP in the whole data set). If the last SNP analyzed here is the last SNP
% in the whole data set, the last column of "emissionProbs" can be set to 1
% and the least element of the "coefs" vector to 1.
%
% "initTransitionMatrix" is the transition matrix between the N:th and 
% N+1:st SNPs.
% "initBetaHat" is the betaHat for the N+1:st SNP. 
% If the last SNP in the current set is the overall last SNP in the 
% sequence, then both "initBetaHat" and "initTransitionMatrix" can be set 
% empty.

[nStates, nSnps] = size(emissionProbs);  % Note: nSnps=nSnps+1
nSnps = nSnps-1;

betaHats = zeros(nStates, nSnps);

if isempty(initBetaHat)
    betaHats(:,nSnps) = 1;
else
    betaHats(:,nSnps) = scaledBetaRecursionStep(initBetaHat, ...
        coefs(nSnps+1), emissionProbs(:,nSnps+1), initTransitionMatrix);
end

snpDistances = snpPositions(2:end)-snpPositions(1:end-1);
for i=nSnps-1:-1:1
    %transitionMatrix = calcTransitionProbs(snpDistances(i), ...
    %    preCalculatedTransMatrices);
    betaHats(:,i) = scaledBetaRecursionStep(betaHats(:,i+1), ...
        coefs(i+1), emissionProbs(:,i+1), preCalculatedTransMatrices, snpDistances(i));
end



function betaHat = scaledBetaRecursionStep(nextBetaHat, nextCoef, ...
    nextEmissionProbs, preCalculatedTransMatrices, snpDistanceNow)
% The function does one iteration of the scaled beta recursion procedure
% described on p. 627-628 of Bishop's book.
%
% "transitionMatrix" contains transition probabilities when moving from 
% time point n to n+1.
%
% "betaHat" corresponds to the beta hat variable at time point t. The
% pre-fix "next" in remaining variables refers to time point n+1.
%
% USAGE EXAMPLE:
% transitionMatrix = [0.8 0.1 0.1; 0.1 0.8 0.1; 0.1 0.1 0.8];
% nextBetaHat = [1.3 0.4 0.9]';
% nextCoef = 0.73;
% nextEmissionProbs = [0.75 0.7 0.1]';
% [betaHat] = scaledBetaRecursionStep(nextBetaHat, nextCoef, ...
% nextEmissionProbs, transitionMatrix);
% betaHat == [1.1192 0.4527 0.2705]';  % about!

%betaHat = transitionMatrix*(nextBetaHat.*nextEmissionProbs);
betaHat = preCalculatedTransMatrices{snpDistanceNow}*(nextBetaHat.*nextEmissionProbs);
betaHat = betaHat ./ nextCoef;