function [alphaHats coefs] = hmmAlphaHatRecursion(snpPositions, ...
    preCalculatedTransMatrices, emissionProbs, ...
    initTransitionMatrix, initAlphaHat)
% The function performs alpha-recursion. The function gets as input the
% distribution for the initial state, as well as the emission probabilities
% at every SNP. Unlike the emission probabilities, and in order to save 
% storage space, the transition probabilities are not pre-calculated but 
% they are calculated inside the function.
%
% "snpPositions" is a vector with positions of the SNPs in the sequence.
% "totalSequenceLength" is the total length of the sequence.
% "priorProbOfNoChangesInSequence" is the prior probability that there in
% not a single state change in the sequence.
% The previous three are needed when the transition probabilities are
% calculated. (note that the totalSequenceLength may be larger than the
% the number of SNPs analyzed here, because it is possible that not all 
% SNPs in a data set are analyzed at the same time).
%
% "emissionProbs" is an nStates*nSnps table with emission probabilities.
%
% "initTransitionMatrix" is the transition matrix between the first SNP in
% the current set of SNPs and the last SNP analyzed in the preceding set of
% SNPs. If the first SNP in the current set is the overall first SNP in the
% sequence, then "initTransitionMatrix" contains on each row the prior
% distribution for the first state.
%
% "initAlphaHat" is the alphaHat for the last SNP analyzed in the preceding 
% set of SNPs. If the first SNP in the current set is the overall first SNP 
% in the sequence, then "initAlphaHat" can be set empty.

[nStates, nSnps] = size(emissionProbs);

alphaHats = zeros(nStates, nSnps);
coefs = zeros(1, nSnps);

if isempty(initAlphaHat)
    initAlphaHat = 1/nStates.*ones(nStates,1);
end

alphaHats(:,1) = emissionProbs(:,1).*(initAlphaHat'*initTransitionMatrix)';
coefs(1) = sum(alphaHats(:,1));
alphaHats(:,1) = alphaHats(:,1)./coefs(1);

snpDistances = snpPositions(2:end)-snpPositions(1:end-1);
for i=2:nSnps
    %transitionMatrix = calcTransitionProbs(snpDistances(i-1), ...
    %    preCalculatedTransMatrices);
    %[alphaHats(:,i),coefs(i)] = scaledAlphaRecursionStep(...
    %    alphaHats(:,i-1), transitionMatrix, emissionProbs(:,i));
    [alphaHats(:,i),coefs(i)] = scaledAlphaRecursionStep(...
        alphaHats(:,i-1), emissionProbs(:,i), preCalculatedTransMatrices, snpDistances(i-1));
end



function [alphaHat coef] = scaledAlphaRecursionStep(...
    previousAlphaHat, emissionProbs, preCalculatedTransMatrices, snpDistanceNow)
% The function does one iteration of the scaled alpha recursion procedure
% described on p. 627-628 of Bishop's book.
%
% USAGE EXAMPLE:
% transitionMatrix = [0.8 0.1 0.1; 0.1 0.8 0.1; 0.1 0.1 0.8];
% previousAlphaHat = [0.5 0.2 0.3]';
% emissionProbs = [0.75 0.7 0.1]';
% [alphaHat coef] = scaledAlphaRecursionStep(previousAlphaHat, ...
% transitionMatrix, emissionProbs);
% % alphaHat == [0.6291 0.3131 0.0578]';
% % coef = 0.5365;

%alphaHat = (previousAlphaHat'*transitionMatrix)'.*emissionProbs;
alphaHat = (previousAlphaHat'*preCalculatedTransMatrices{snpDistanceNow})'.*emissionProbs;

coef = sum(alphaHat); % normalize
alphaHat = alphaHat ./ coef;
