function emissionProbs = calcEmissionProbs(counts, sumCounts, ...
    priorCounts, priorSumCounts, dataRow)
% The function calculates emission probabilities for the observed SNP 
% values.
%
% "counts" contains the counts of observed SNPs in different clusters.
%
% "sumCounts" contains the total count of observed SNPs in different
% positions in different clusters.
%
% "priorCounts" and "priorSumCounts" are analogous to the previous two, 
% except that these contain the prior counts (parameters of the Dirichlet
% distributions)
%
% "dataRow" contains the observed SNP values.
%
% USAGE EXAMPLE:
% counts(:,:,1) = [1 2 2; 4 4 2; 1 0 2];
% counts(:,:,2) = [0 0 1; 0 1 1; 2 0 0];
% counts(:,:,3) = [0 0 0; 0 0 0; 0 0 0];
% sumCounts = [6 6 6;2 1 2;0 0 0];
% priorCounts = [1/3 1/2 1/3;1/3 1/2 1/3;1/3 0 1/3];
% priorSumCounts = sum(priorCounts);
% dataRow = [2 1 -9];
% emissionProbs = calcEmissionProbs(counts, sumCounts, priorCounts,
% priorSumCounts, dataRow);
% % result: [0.6190 0.3571 1.000;0.1111 0.2500 1.000;0.3333 0.5000 1.000];

[maxNoalle, nloci, npops] = size(counts);
emissionProbs = zeros(npops,nloci);

%totalCounts = counts + repmat(priorCounts, [1 1 npops]);
%totalSumCounts = sumCounts + repmat(priorSumCounts, [npops, 1]);

totalCounts = counts + priorCounts;
totalSumCounts = sumCounts + priorSumCounts;

totalSumCounts = repmat(reshape(totalSumCounts', [1 nloci npops]), ...
    [maxNoalle 1 1]);

allFreqs = totalCounts ./ totalSumCounts;

nonMissing = find(dataRow>0);
missing = setdiff(1:nloci, nonMissing);

for k = 1:npops
    emissionProbs(k,nonMissing) = ...
        allFreqs(maxNoalle*nloci*(k-1) + ...
        sub2ind([maxNoalle nloci], dataRow(nonMissing), nonMissing));
    emissionProbs(k,missing) = 1;
end
