function uData = learnRecombinationsInClusters(inputFile, outputFile, outputDir, reducedOutput)
    
    load(inputFile);  % Variable uData
    load(uData.snpDataFileName); % Variable snpData
    
    % Initial population structure, where every strain is fully added to 
    % one of the original clusters.
    c = createBapsOutputStruct(uData.groupedPartition, snpData, ...
        uData.snpPositions, length(unique(uData.groupedPartition))+1, [], []);  % Add one empty cluster.
    clear snpData % Save space
    
    nSnps = length(uData.snpPositions);
    
    nStates = size(c.COUNTS,3);
    
    [nStrains nSnps] = size(c.data);
    
    % Go through each cluster and try to find recombinations in it.
    uData.updatedPopStructure = uint8(repmat(c.PARTITION,[1 nSnps]));
    
    nClusters = length(unique(uData.groupedPartition)); % No need to analyse the empty cluster
    
    for clusterIndex = 1:nClusters
        
        finalPopStructureForCluster = learnRecombinationsInCluster(clusterIndex, uData, nStates, c, outputDir, uData.delimiter, reducedOutput);
        % The function also saves at the last iteration the current
        % simulated population structure and the corresponding transition
        % parameters for the cluster: 'simulatedPopStructureForCluster', 
        % 'cluster_transPars'.
        % The 'finalPopStructureForCluster' is derived from marginal
        % cluster probabilities at the last iteration.
        %
        % Note: changes in 'c' that are done within the function are not
        % returned back to this function, so each cluster is started with a
        % 'clean' assignment of strains to their home clusters.
        
        strainsInCluster = find(uData.groupedPartition == clusterIndex);
        uData.updatedPopStructure(strainsInCluster,:) = finalPopStructureForCluster; % This is derived using marginal probabilities and probabilityThreshold.
        
    end
    
    %uData.outputDir = outputDir;
    uData.nIterCompleted = uData.nIterCompleted + uData.nIter;
    uData.state = 'recProfilesCalculated';
    save(outputFile, 'uData', '-v7.3');
    
end



function finalPopStructureForCluster = learnRecombinationsInCluster(clusterIndex, uData, nStates, c, outputDir, delimiter, reducedOutput)
    
    strainsInCluster = find(uData.groupedPartition==clusterIndex);
    nStrainsInCluster = length(strainsInCluster);
    
    nSnps = length(uData.snpPositions);
    
    if uData.nIterCompleted == 0
        
        cluster_transPars = uData.transPars;
        cluster_transPars.home = clusterIndex;
        simulatedPopStructureForCluster = clusterIndex .* ones(nStrainsInCluster, nSnps, 'uint8');
        
    else
        
        load([outputDir delimiter 'simuClusterInfo_' num2str(clusterIndex) '.mat']); % simulatedPopStructureForCluster cluster_transPars
        
        
        % Recombinations must be removed from counts etc. in 'c' for the cluster.
        oldStates = clusterIndex .* ones(1,nSnps); % This is where the observations are currently in the counts (i.e., all in their home cluster).
        for indexOfStrainInCluster = 1:nStrainsInCluster
            newStates = []; % Don't add anything to any cluster
            changeThisSite = (simulatedPopStructureForCluster(indexOfStrainInCluster,:) ~= clusterIndex); % Remove only the recombinant sites
            c = updateCountsWithStrain(c, strainsInCluster(indexOfStrainInCluster), newStates, oldStates, changeThisSite); 
        end
        
    end
    
    segmentsForStrains = cell(1,nStrainsInCluster);
    finalPopStructureForCluster = clusterIndex .* ones(nStrainsInCluster, nSnps, 'uint8');
    
    disp(['Processing lineage ' num2str(clusterIndex) ' with ' num2str(nStrainsInCluster) ' strains.']);
    
    disp([' Iteration: 0, 1-rho = ' num2str(1-cluster_transPars.rho) ', a = ' num2str(cluster_transPars.a)]);
    
    
    for iteration = uData.nIterCompleted+1 : uData.nIterCompleted+uData.nIter
        
        preCalculatedTransMatrices = preCalculateTransMatrices(uData.totalSequenceLength, cluster_transPars, nStates, uData.snpPositions);
        
        for indexOfStrainInCluster = 1:nStrainsInCluster
            % Check recombinations in all strains of the population
            
            strainIndex = strainsInCluster(indexOfStrainInCluster);
            
            % Remove observations for strain with "strainIndex" from counts and
            % sumcounts
            newStates = []; % Don't add anything to counts.
            oldStates = simulatedPopStructureForCluster(indexOfStrainInCluster, :); % Current labels for the observations
            changeThisSite = (oldStates == clusterIndex); % Remove the non-recombinant sites (because only they are there...)
            
            c = updateCountsWithStrain(c, strainIndex, newStates, oldStates, changeThisSite);
            
            
%             % CHECK THAT THE STRAIN WAS REALLY REMOVED
%             if 1==0
%                 structureToTest = simulatedPopStructureForCluster;
%                 if clusterIndex==1
%                     someOtherCluster = 2;
%                 else
%                     someOtherCluster = 1;
%                 end
%                 structureToTest(indexOfStrainInCluster,:) = someOtherCluster;
%                 %checkCountsClusterStructureCompatibility(c, structureToTest, clusterIndex)
%                 
%                 % CHECK ALSO THAT THE RECOMBINATIONS ARE NOT ADDED TO OTHER
%                 % CLUSTERS, SUFFICIENT TO CHECK JUST ONE CLUSTER (=3)...
%                 if clusterIndex ~= 3
%                     clusterToTest = 3;
%                     nStrainsInClusterToTest = length(find(c.PARTITION==clusterToTest));
%                     structureToTest = clusterToTest .* ones(nStrainsInClusterToTest,size(simulatedPopStructureForCluster,2));
%                     checkCountsClusterStructureCompatibility(c, structureToTest, clusterToTest);
%                 end
%             end
            
            
            
            % Detect recombinations in the strain
            [marginals, ~, simulatedStates] = hmmRecombinationAnalysis(c, preCalculatedTransMatrices, strainIndex);
            
            
            % Put back to counts the observations at non-recombinant sites.
            oldStates = [];
            newStates = simulatedStates;
            changeThisSite = (newStates == clusterIndex); % Put back the non-recombinant sites
            c = updateCountsWithStrain(c, strainIndex, newStates, oldStates, changeThisSite);
            
            
            simulatedPopStructureForCluster(indexOfStrainInCluster, :) = uint8(simulatedStates);
            % This represents a realization of the population structure simulated from the current posterior.
            
            
%             % CHECK THAT AFTER UPDATING THE STRAIN, THE DATA STRUCTURES ARE
%             % CORRECT.
%             if 0==1 % For debugging
%                 checkCountsClusterStructureCompatibility(c, simulatedPopStructureForCluster, clusterIndex)
%             end
            
            
            segmentsForStrains{indexOfStrainInCluster} = identifySegments(simulatedPopStructureForCluster, indexOfStrainInCluster, c.snpPositions, uData.totalSequenceLength);
            
            
            
            if iteration == uData.nIterCompleted + uData.nIter
                % On last iteration, save the final profiles, derived from the
                % marginals
                
                [~,maxOrigin] = max(marginals);
                
                finalPopStructureForCluster(indexOfStrainInCluster,:) = clusterIndex;
                
                recombinantSites = find(marginals(clusterIndex,:) < uData.probabilityThreshold);
                finalPopStructureForCluster(indexOfStrainInCluster,recombinantSites) = maxOrigin(recombinantSites);
                
                if reducedOutput == 0
                    % Save detailed results for a strain only if the
                    % reduced output was not requested.
                    save([outputDir delimiter 'strain_' num2str(strainIndex) '.mat'], 'marginals');
                end
                
            end
            
        end
        
        
        % After estimating profiles for all strains, update the transition
        % parameters.
        cluster_transPars = learnTransPars(segmentsForStrains, uData.totalSequenceLength, cluster_transPars);
        disp([' Iteration: ' num2str(iteration) ', 1-rho = ' num2str(1-cluster_transPars.rho) ', a = ' num2str(cluster_transPars.a)]);
        
    end
    
    % The following will be saved for use in additional iterations, if
    % needed.
    if reducedOutput == 0
        save([outputDir delimiter 'simuClusterInfo_' num2str(clusterIndex) '.mat'], 'simulatedPopStructureForCluster', 'cluster_transPars');
    end
    
end
