function c = getBapsInputBlockFromSnpData(blockIndex, blockWidth, ...
    snpData, snpPositions,totalSequenceLength, alpha)
% Creates BAPS input struct corresponding to a given block of SNPs. The
% genome is divided into blocks of size "blockWidth", and the SNPs inside
% the "blockIndex"th block are included into the struct.
%
% OTHER INPUTS:
% "snpData", nStrains*nSnps containing SNPs, (A=1,C=2,G=3,T=4,miss=-9).
%
% "snpPositions", 1*nSnps, the positions of the SNPs in the genome.
%
% "totalSequenceLength", total length of the genomes.
%
% OUTPUT:
% "c", BAPS input struct, fields: 'data', 'noalle', 'adjprior', 'priorTerm'.
% 
% TESTING EXAMPLE:
% [strainLabels, data, snpPositions, totalSequenceLength] = ...
%   readFastaData('..\data\testData\fastaTest.aln');
% c = getBapsInputBlockFromSnpData(2,5,data,snpPositions,totalSequenceLength);
% isequal(c.data,[1 1 1;1 1 2;1 1 -9;2 2 2])
% c = getBapsInputBlockFromSnpData(4,2,data,snpPositions,totalSequenceLength);
% isequal(c.data,[1 1;1 2;1 -9;2 2])

if nargin==5
    alpha=1;
end

firstPosition = (blockIndex-1)*blockWidth+1;
if firstPosition>totalSequenceLength
    error('Block index out of range.');
end
lastPosition = min(blockIndex*blockWidth,totalSequenceLength);

columnsToInclude = (snpPositions>=firstPosition & snpPositions<=lastPosition);

[c.data c.noalle] = relabelCodes(snpData(:,columnsToInclude));
if any(c.noalle==1)
    error('SNP with only a single value.');
end
[c.adjprior, c.priorTerm] = calculatePriorTerms(c, alpha);