#!/usr/bin/env zsh
setopt extendedglob

fail () {
  print -u 2 -- "$1"
  exit 1
}

zparseopts -D -E -a help random:=random modelsize:=modelsize formulasize:=formulasize timelimit:=timelimit memlimit:=memlimit logdir:=logdir help -help h

if [[ -n $help ]]; then
  cat <<help
Options:
-h, -help, --help:  show this help
-random <n>:        generate n random tests
-modelsize <n>:     size of random Kripke structures; required by -random
-formulasize <n>:   size of random PSL formulae; required by -random
-timelimit <n>:     time limit for a single run in seconds (default 60)
-memlimit <n>:      memory limit for a single run in MB (default 512)
-logdir <dir>:      log results for plotting, use <dir>
                    (created if necessary), always required
help
  exit 0
fi

timelimit=${timelimit[2]:-60}
memlimit=${memlimit[2]:-512}
memlimit=$((memlimit*1024))

[[ -z $logdir ]] && fail "You must specify -logdir"
[[ -e $logdir[2] && ! -d $logdir[2] ]] && fail "$logdir exists but isn't a directory"
if [[ ! -e $logdir[2] ]] ; then
  mkdir -p $logdir[2] || fail "Could not make new directory $logdir[2]"
fi

pushd ${0:h}
nusmv=${PWD%/nusmv/?#}/nusmv/NuSMV
popd

if [[ -n $random ]]; then
  pushd ${0:h}
  [[ ! -x randomKripke.pl ]] && fail "${0:h}/randomKripke.pl not found"
  [[ ! -x randomPSL.py ]] && fail "${0:h}/randomPSL.py not found"
  [[ ! -x /usr/bin/time ]] && fail "No /usr/bin/time"
  randomKripke=$PWD/randomKripke.pl
  randomPSL=$PWD/randomPSL.py
  popd
  random=${random[2]:-100}
  [[ -z $modelsize || -z $formulasize ]] && fail "-random requires -modelsize and -formulasize"
  modelsize=$modelsize[2]
  ((modelsize<=3)) && fail "Model size must be at least 4"
  formulasize=$formulasize[2]
  logdir=$logdir[2]
  for ((i=0;i<random;i++)) ; do
    mkdir $logdir/$i || fail "Could not make test dir in $logdir"
    pushd $logdir/$i
    $randomKripke -k $modelsize -l 3 -p 6 -o tmp.smv
    cat tmp.smv|grep -v "JUSTICE" > model.smv
    rm tmp.smv
    $randomPSL --length=$formulasize --atoms=6 >> model.smv
    /usr/bin/time -p zsh -c "ulimit -t $timelimit -v $memlimit ; $nusmv model.smv > out.log 2> err.log ; print \$? > exitstatus" 2> time
    popd
  done
else
  while ((#>0)); do
    mkdir $logdir/${1:t:r} || fail "Could not make test dir in $logdir"
    cp $1 $logdir/${1:t:r}/model.smv
    pushd $logdir/${1:t:r}
    /usr/bin/time -p zsh -c "ulimit -t $timelimit -v $memlimit ; $nusmv model.smv > out.log 2> err.log ; print \$? > exitstatus" 2> time
    popd
    shift
  done
fi