#!/usr/bin/perl
#--preamble, use declarations
use strict;
use warnings;

use diagnostics;
use IO::Handle;
#--end preamble
# 2004 Timo Latvala <timo.latvala@hut.fi>
#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation; either version 2 of the License, or
#(at your option) any later version.

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


##subroutine for generating an integer between min and max

sub random_int_in ($$) {
     my($min, $max) = @_;
      # Assumes that the two arguments are integers themselves!
     return $min if $min == $max;
     ($min, $max) = ($max, $min)  if  $min > $max;
     return $min + int rand(1 + $max - $min);
   }  

#print the usage of the program

sub usage() {
    print ("lassoKripke -p number -k number [-o filename] [-l number]\n");
    print ("p \t number of atomic proposition in a state \n");
    print ("k \t number of states in the model\n");
    print ("l \t expected number of edges from a state  (l<k must hold)\n");
}

#return TRUE when the arguement is 1 and FALSE when 
#the argument is 0

sub tf($) {
    if($_[0]==1) {
	return "";
    }
    elsif($_[0]==0) {
	return "!";
    }
    else {
	return "UNDEF";
    }
}
 

#output the NuSMV code for a connected Kripke structure
#to the file $_[0]
#which has $_[1]+1 unique states, 
#where the state $_[2] is the expected number of edges from a state 
#and each state has $_[3] atomic propositions with random truth values.

sub kripke($$$$) {
    my ($filename, $k, $l, $p)=@_;
    
    my $FILE=IO::Handle->new();

    if ($filename ne "") { 
	open($FILE,"> $filename") or die "Can't open file $filename: $!";    
    } else {
	$FILE=*STDOUT;
    }

    print $FILE "MODULE main\n";
    #variable declarations
    print $FILE "VAR\n";
    my $real_k=$k-1;
    print $FILE "state : 0..$real_k;\n";    
    for (my $i=0; $i < $p; $i++) {
	print $FILE "\t p$i : boolean;\n";
    }
    
    print $FILE "\n";
    print $FILE "INIT\n";
    #state variable intialisation
    print $FILE "((state = 0) & ("; 
    my @initvalue;
    #pi variable initialisations;
    for (my $i=0; $i < $p ; $i++) {
	my $value=random_int_in(0,1);
	print $FILE " & " if ($i>0);
	print $FILE tf($value);
	print $FILE "p$i";
	push @initvalue,$value;
    }
    print $FILE "))\n";
    print $FILE "\n";

    
    print $FILE "TRANS\n";
    my %Processed;
    my @ToProcess;
    push @ToProcess,0;
    $Processed{"0"}=0;
    my %Unreachable;
    for (my $i=1; $i<$k; $i++) {
	$Unreachable{"$i"}=$i;
    }
    #the transitions
    print $FILE "(\n";
    while(@ToProcess>0) {
	my $source=pop @ToProcess;
	print $FILE "\t ((state = $source) & (";
	my $edges=0;
	if (%Unreachable) {
	    my $node;
	    do {
		$node=random_int_in(1,$real_k);
	    } while (! exists $Unreachable{"$node"});
	    delete $Unreachable{"$node"};
	    print $FILE "(next(state) = $node)";
	    push @ToProcess,$node; 
	    $Processed{"$node"}=$node;
	    $edges++;
	}
	for(my $i=0; $i<$k; $i++) {
	    #coin toss to determine if there is an edge from source to i
	    if ($l >= random_int_in(1,$k)) {
		print $FILE " | " if ($edges>0);
		print $FILE "(next(state) = $i)";
		if (exists $Unreachable{"$i"}) { 
		    delete $Unreachable{"$i"};
		    if (! exists $Processed{"$i"}) {
			push @ToProcess,$i; 
			$Processed{"$i"}=$i;
		    }
		}
		$edges++;
	    }
	}
	#add selfloop to source if no other edges exist
	print $FILE "(next(state) = $source)" if($edges==0); 
	print $FILE "))\n";
	print $FILE " | " if (@ToProcess>0);
    } 
    print $FILE ")\n";
    
    print $FILE " & \n (\n";
    #APs for each state 
    for (my $i=0; $i < $k; $i++) {
 	print $FILE " | " if ($i>0);
 	print $FILE "(next(state) = $i &";
  	print $FILE " (";
 	for (my $j=0; $j < $p; $j++) {
	    print $FILE " & " if ($j>0);
	    if($i==0) {
		print $FILE tf($initvalue[$j]);
	    } else {	
		print $FILE tf(random_int_in(0,1)); 
	    }
	    print $FILE "next(p$j)";
 	}	
 	print $FILE "))\n";
    }
    print $FILE ")\n";
    print $FILE "JUSTICE (state=".random_int_in(0,$real_k)." | state=".random_int_in(0,$real_k).");\n";

    

    close $FILE;
}

#---------------------------------------------------------------------
#                      MAIN PROGRAM
#---------------------------------------------------------------------
# Check arguments, open files, and call generator code
#
#---------------------------------------------------------------------

if (@ARGV < 4) {
    print ("Too few arguments!\n");
    usage();
    exit -1;
}

#declare parameter variables
my $p=-1;
my $k=-1;
my $l=-1;
my $output_file="";

for (my $i=0; $i <= $#ARGV; $i++) {
    if ($ARGV[$i] eq "-p") {
	$p=$ARGV[$i+1];
	($p !~ /\D+/) or die "Argument for -p should be a number.\n";
	$i=$i+1;
    }  
    elsif($ARGV[$i] eq "-k") {
	$k=$ARGV[$i+1];
	($k !~ /\D+/) or die "Argument for -k should be a number.\n";
	$i=$i+1;
    }
    elsif($ARGV[$i] eq "-l") {
	$l=$ARGV[$i+1];
	($l !~ /\D+/) or die "Argument for -l should be a number.\n";
	$i=$i+1;	
    }
    elsif($ARGV[$i] eq "-o") {
	$output_file=$ARGV[$i+1];
	($output_file =~ /.+/) or die "Illegal file name specified using -o.\n";
	$i=$i+1;
    }
    else {
	print ("Unknown option: $ARGV[$i] .\n");
	exit -1;
    }
}


if($k < 1) {
    print ("Value of k undefined or less than zero!\n");
    usage();
    exit -1;
}

if($p < 1) {
    print ("Value of p undefined or less than zero!\n");
    usage();
    exit -1;
}

if($l != -1 && $l<0) {
    print ("Value of l less than zero!\n");
    usage();
    exit -1;
}
    
kripke($output_file, $k, $l, $p);
