/*
 This file is part of the PSL Observer modification for NuSMV.
 Copyright (C) 2008-2009 Tuomas Launiainen.
 
 This program is free software; you can redistribute it and/or 
 modify it under the terms of the GNU Lesser General Public 
 License as published by the Free Software Foundation; either 
 version 2 of the License, or (at your option) any later version.
 
 This program is distributed in the hope that it will be useful, 
 but WITHOUT ANY WARRANTY; without even the implied warranty of 
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public 
 License along with this library; if not, write to the Free Software 
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.
 
 For more information, see <http://www.tcs.hut.fi/~tlauniai/psl-observer>.
 */ 
#define DEBUG

#include "observer.h"
#include "observer_utils.h"
#include "fsm.h"
#include "utils/assoc.h"
#include "utils/ustring.h"

#define module_body(m) cdr(m)
#define module_set_body(m,b) setcdr(m,b)
#define module_name(m) node_get_lstring(car(car(m)))->text
#define get_modtype(m) car(m)

#define final_state "fs"

hash_ptr pnf_hash=NULL;
hash_ptr process_hash=NULL;

unsigned char not_safety=0;

#pragma mark Auxiliary functions

void init_construction() {
  pnf_hash=new_assoc();
}

void finish_construction() {
  free_assoc(pnf_hash);
}

/*void pn(node_ptr n) {
  print_node(stdout,n);
  printf("\n");
}*/

typedef struct {
  node_ptr trans;
  node_ptr invar;
  node_ptr assigns;
  node_ptr vars;
  node_ptr params;
  node_ptr toplevel;
  hash_ptr var_hash;
  hash_ptr y_hash;
  hash_ptr z_hash;
  hash_ptr param_hash;
  node_ptr fs;
} mod_stub;

typedef struct {
  mod_stub* stub;
  hash_ptr node_hash;
} proc_data;

mod_stub* mod_stub_create() {
  mod_stub* ret=malloc(sizeof(mod_stub));
  assert(ret);
  ret->trans=Nil;
  ret->invar=Nil;
  ret->assigns=Nil;
  ret->vars=Nil;
  ret->params=Nil;
  ret->toplevel=Nil;
  ret->var_hash=new_assoc();
  ret->fs=atom(final_state);
  assert(ret->fs);
  assert(ret->var_hash);
  ret->param_hash=new_assoc();
  assert(ret->param_hash);
  ret->y_hash=new_assoc();
  assert(ret->y_hash);
  ret->z_hash=new_assoc();
  assert(ret->z_hash);
  return ret;
}

void mod_stub_free(mod_stub* m) {
  node_ptr tmp;
  while (m->trans) {
    tmp=m->trans;
    m->trans=cdr(m->trans);
    free_node(tmp);
  }
  while (m->invar) {
    tmp=m->invar;
    m->invar=cdr(m->invar);
    free_node(tmp);
  }
  while (m->vars) {
    tmp=m->vars;
    m->vars=cdr(m->vars);
    free_node(tmp);
  }
  free_assoc(m->var_hash);
  free_assoc(m->param_hash);
  free_assoc(m->y_hash);
  free_assoc(m->z_hash);
  free(m);
}

void mod_stub_add_var(mod_stub* m, node_ptr a) {
  a=share_subformulas(a);
  if (!find_assoc(m->var_hash,a)) {
    m->vars=cons(var(cons(colon(a,leaf(BOOLEAN)),Nil)),m->vars);
    insert_assoc(m->var_hash,a,trueexp);
  }
}

#define mod_stub_add_yvar(m,a,p) mod_stub_add_hvar(m,a,p,falseexp)
#define mod_stub_add_zvar(m,a,p) mod_stub_add_hvar(m,a,p,trueexp)

void mod_stub_add_hvar(mod_stub* m, node_ptr a, node_ptr p, node_ptr i) {
  a=share_subformulas(a);
  if ((i==falseexp && !find_assoc(m->y_hash,a)) ||
      (i==trueexp && !find_assoc(m->z_hash,a))) {
    m->vars=cons(var(cons(colon(a,leaf(BOOLEAN)),Nil)),m->vars);
    insert_assoc(m->var_hash,a,trueexp);
    if (m->assigns) {
      m->assigns=and(m->assigns,eqdef(sinit(a),i));
    } else {
      m->assigns=eqdef(sinit(a),i);
    }
    m->assigns=and(m->assigns,eqdef(next(a),p));
  }
}

void mod_stub_add_trans(mod_stub* m, node_ptr e) {
  if (e==Nil) {
    fprintf(stderr,"Error: adding Nil TRANS\n");
    exit(1);
  }
  m->trans=cons(trans(e),m->trans);
}

void mod_stub_add_invar(mod_stub* m, node_ptr e) {
  m->invar=cons(invar(e),m->invar);
}

void mod_stub_add_param(mod_stub* m, node_ptr p) {
  p=share_subformulas(p);
  if (!find_assoc(m->param_hash,p)) {
    m->params=cons(p,m->params);
    insert_assoc(m->param_hash,p,trueexp);
  }
}

node_ptr mod_stub_to_module(mod_stub* m, node_ptr name) {
  node_ptr body=Nil, iter=Nil;
  body=cons(init(m->toplevel),body);
  iter=m->trans;
  while (iter) {
    body=cons(car(iter),body);
    iter=cdr(iter);
  }
  iter=m->invar;
  while (iter) {
    body=cons(car(iter),body);
    iter=cdr(iter);
  }
  body=cons(var(cons(colon(atom(final_state),leaf(BOOLEAN)),Nil)),body);
  iter=m->vars;
  while (iter) {
    body=cons(car(iter),body);
    iter=cdr(iter);
  }
  if (m->assigns) {
    body=cons(assign(m->assigns),body);
  }
  body=cons(invarspec(not(atom(final_state))),body);
  return module(modtype(name,m->params),body);
}

proc_data* proc_data_create() {
  proc_data* p=malloc(sizeof(proc_data));
  assert(p);
  p->stub=mod_stub_create();
  p->node_hash=new_assoc();
  assert(p->stub && p->node_hash);
  return p;
}

void proc_data_free(proc_data* p) {
  mod_stub_free(p->stub);
  free_assoc(p->node_hash);
  free(p);
}

node_ptr positive_normal_form(node_ptr e) {
  node_ptr ret, orig;
  unsigned char not=0, arity=0, invert=0;
  int inv_op=0;
  orig=e;
  ret=find_assoc(pnf_hash,e);
  if (ret)
    return ret;
  if (!e)
    return e;
  if (node_get_type(e)==NOT) {
    not=1;
    e=car(e);
  }
  if (node_get_type(e)==PSL_NEVER) {
    e=globally(not(car(e)));
  }
  switch (node_get_type(e)) {
    case PSL_SERE:
    case PSL_SERECONCAT:
    case PSL_SEREFUSION:
    case PSL_SERECOMPOUND:
    case PSL_SEREREPEATED:
      if (not) {
        ret=not(e);
      } else {
        ret=e;
      }
      break;
    case OR:
      arity=2;
      inv_op=AND;
      invert=1;
      break;
    case AND:
      arity=2;
      inv_op=OR;
      invert=1;
      break;
    case IMPLIES:
      if (not)
        ret=and(positive_normal_form(car(e)),positive_normal_form(not(cdr(e))));
      else
        arity=2;
      break;
    case IFF:
    case XNOR:
    case XOR:
      if ((not && node_get_type(e)==IFF) ||
          (not && node_get_type(e)==XNOR) ||
          (!not && node_get_type(e)==XOR)) {
        ret=or(and(positive_normal_form(car(e)),
                   positive_normal_form(not(cdr(e)))),
               and(positive_normal_form(not(car(e))),
                   positive_normal_form(cdr(e))));
      } else {
        ret=or(and(positive_normal_form(car(e)),
                   positive_normal_form(cdr(e))),
               and(positive_normal_form(not(car(e))),
                   positive_normal_form(not(cdr(e)))));
      }
      break;
    case EQUAL:
      arity=2;
      inv_op=NOTEQUAL;
      invert=0;
      break;
    case NOTEQUAL:
      arity=2;
      inv_op=EQUAL;
      invert=0;
      break;
    case LT:
      arity=2;
      inv_op=GE;
      invert=0;
      break;
    case GT:
      arity=2;
      inv_op=LE;
      invert=0;
      break;
    case LE:
      arity=2;
      inv_op=GT;
      invert=0;
      break;
    case GE:
      arity=2;
      inv_op=LT;
      invert=0;
      break;
    case PSL_PIPEMINUSGT:
      if (not) {
        ret=tailex(positive_normal_form(tail_car(e)),
                   positive_normal_form(not(tail_cdr(e))));
      } else {
        ret=tailimpl(positive_normal_form(tail_car(e)),positive_normal_form(tail_cdr(e)));
      }
      break;
    case UNTIL:
      arity=2;
      inv_op=RELEASES;
      invert=1;
      break;
    case RELEASES:
      arity=2;
      inv_op=UNTIL;
      invert=1;
      break;
    case PSL_X:
    case OP_NEXT:
      arity=1;
      inv_op=OP_NEXT;
      invert=1;
      break;
    case OP_GLOBAL:
    case PSL_ALWAYS:
      arity=1;
      inv_op=OP_FUTURE;
      invert=1;
      break;
    case OP_FUTURE:
      arity=1;
      inv_op=OP_GLOBAL;
      invert=1;
      break;
    case OP_PREC:
      arity=1;
      inv_op=OP_NOTPRECNOT;
      invert=1;
      break;
    case OP_NOTPRECNOT:
      arity=1;
      inv_op=OP_PREC;
      invert=1;
      break;
    case OP_ONCE:
      arity=1;
      inv_op=OP_HISTORICAL;
      invert=1;
      break;
    case OP_HISTORICAL:
      arity=1;
      inv_op=OP_ONCE;
      invert=1;
      break;
    case SINCE:
      arity=2;
      inv_op=TRIGGERED;
      invert=1;
      break;
    case TRIGGERED:
      arity=2;
      inv_op=SINCE;
      invert=1;
      break;
    case NUMBER:
    case DOT:
    case ARRAY:
    case UNION:
    case SETIN: 
    case ATOM:
    case TIMES:
    case DIVIDE:
    case PLUS :
    case MINUS:
    case MOD: 
    case LSHIFT:
    case RSHIFT:
    case LROTATE:
    case RROTATE:
      if (not)
        ret=not(e);
      else
        ret=e;
      break;
    case NOT:
      if (not)
        ret=positive_normal_form(car(e));
      else
        assert(0);
      break;
    case FALSEEXP:
      if (not)
        ret=trueexp;
      else
        ret=falseexp;
      break;
    case TRUEEXP:
      if (not)
        ret=falseexp;
      else
        ret=trueexp;
      break;
    default:
      fprintf(stderr,"Observer construction encountered an invalid node (%d), aborting\n",node_get_type(e));
      exit(1);
  }
  if (!ret) {
    switch (arity) {
      case 1:
        if (not)
          if (invert)
            ret=find_node(inv_op,positive_normal_form(not(car(e))),Nil);
          else
            ret=find_node(inv_op,positive_normal_form(car(e)),Nil);
          else
            ret=find_node(node_get_type(e),positive_normal_form(car(e)),Nil);
        break;
      case 2:
        if (not)
          if (invert)
            ret=find_node(inv_op,positive_normal_form(not(car(e))),positive_normal_form(not(cdr(e))));
          else
            ret=find_node(inv_op,positive_normal_form(car(e)),positive_normal_form(cdr(e)));
          else
            ret=find_node(node_get_type(e),positive_normal_form(car(e)),positive_normal_form(cdr(e)));
        break;
      default:
        assert(0);
    }
  }
  insert_assoc(pnf_hash,orig,ret);
//  printf("PNF for "); print_node(stdout, orig); printf(": ");
//  print_node(stdout, ret); printf("\n");
  if (node_get_type(ret) == RELEASES || node_get_type(ret) == OP_GLOBAL ||
      istailimpl(ret)) {
    not_safety=1;
  }
  return ret;
}

#pragma mark Spec processing

node_ptr process_expr(node_ptr,proc_data*);

node_ptr process_simple_expr(node_ptr expr,proc_data* p) {
  node_ptr ret=Nil;
  switch (node_get_type(expr)) {
    case OR:
    case AND:
    case IMPLIES:
    case IFF:
    case XNOR:
    case XOR:
    case UNION:
    case SETIN: 
    case EQUAL: 
    case NOTEQUAL:
    case LT:
    case GT:
    case LE:
    case GE: 
    case TIMES:
    case DIVIDE:
    case PLUS :
    case MINUS:
    case MOD: 
    case LSHIFT:
    case RSHIFT:
    case LROTATE:
    case RROTATE:
      ret=find_node(node_get_type(expr),
                    process_expr(car(expr),p),
                    process_expr(cdr(expr),p));
      break;
    case NOT:
      ret=find_node(node_get_type(expr),process_expr(car(expr),p),Nil);
      break;
    case ATOM:
      mod_stub_add_param(p->stub,expr);
      ret=expr;
      break;
    case DOT:
    case ARRAY:
      ret=car(expr);
      while (ret && (node_get_type(ret)==DOT || node_get_type(ret)==ARRAY))
        ret=car(ret);
      assert(node_get_type(ret)==ATOM);
      mod_stub_add_param(p->stub,ret);
      ret=expr;
      break;
    case FALSEEXP:
    case TRUEEXP:
    case NUMBER:
      ret=expr;
      break;
    default:
      fprintf(stderr,"Observer construction encountered an invalid node (%d), aborting\n",node_get_type(expr));
      exit(1);
  }
  insert_assoc(p->node_hash,expr,ret);
  return ret;
}

#define expr_add(expr,connective,node) (expr==Nil? node : find_node(connective,node,expr))

#define VAR_NAME_LEN 30

void mod_stub_add_fsm(mod_stub* m,fsm a) {
  node_ptr state_iter=fsm_states(a);
  node_ptr notfs=Nil;
  while (state_iter) {
    node_ptr state=car(state_iter);
    mod_stub_add_var(m, state);
    if (notfs!=Nil)
      notfs=or(state,notfs);
    else
      notfs=state;
    state_iter=cdr(state_iter);
  }
  mod_stub_add_invar(m, implies(notfs,not(m->fs)));
}

node_ptr process_expr(node_ptr expr,proc_data* p) {
  char var_name[VAR_NAME_LEN];
  unsigned char new_var=0;
  node_ptr var=Nil, left=Nil, right=Nil;
  node_ptr notfs=Nil;
  fsm automaton;
  node_ptr result=find_assoc(p->node_hash,expr);
  if (result!=Nil)
    return result;
  if (!p->stub->toplevel)
    p->stub->toplevel=expr;
  switch (node_get_type(expr)) {
    case PSL_PIPEMINUSGT:
    {
      node_ptr conjunction=Nil;
      automaton=fsm_from_sere(tail_car(expr));
      if (fsm_edges_from(automaton, fsm_initial(automaton))==Nil) {
        if (fsm_is_accepting(automaton, fsm_initial(automaton))) {
          result=process_expr(tail_cdr(expr),p);
          break;
        }
        printf("SERE "); print_node(stdout,expr);
        printf(" has no possible matches, formula ≡ ⊤\n");
        result=trueexp;
        break;
      }
//      printf("SERE "); print_node(stdout, tail_car(expr));
//      printf(" translates to:\n"); print_fsm(stdout, automaton);
//      printf("and implies: "); print_node(stdout, tail_cdr(expr));
//      printf("\n");
      right=process_expr(tail_cdr(expr),p);
      mod_stub_add_fsm(p->stub, automaton);
      node_ptr state_iter=fsm_states(automaton);
      while (state_iter) {
        node_ptr state=car(state_iter);
        node_ptr edge_iter=fsm_edges_from(automaton, state);
        while (edge_iter) {
          node_ptr edge=car(edge_iter);
          process_simple_expr(edge_label(edge), p);
          conjunction=expr_add(conjunction,AND,
                               implies(and(state,edge_label(edge)),
                                                       next(edge_to(edge))));
          edge_iter=cdr(edge_iter);
        }
        if (state!=fsm_initial(automaton) &&
            fsm_is_accepting(automaton, state)) {
          mod_stub_add_trans(p->stub, 
                             implies(next(state), right));
        }
        state_iter=cdr(state_iter);
      }
      if (conjunction==Nil) {
        printf("Error: nothing to add to trans in:\n");
        print_fsm(stdout, automaton); printf("(created from "); print_node(stdout, tail_car(expr)); printf(")\n");
        exit(1);
      }
      mod_stub_add_trans(p->stub, conjunction);
#ifdef MYSEMANTICS
      if (fsm_is_accepting(automaton, fsm_initial(automaton))) {
        result=and(right,fsm_initial(automaton));
      } else {
        result=fsm_initial(automaton);
      }
#else
      result=fsm_initial(automaton);
#endif
      free_fsm(automaton);
      break;
    }
    case NOT:
      if (istailex(expr)) {
        node_ptr disjunction=Nil;
        node_ptr conjunction=Nil;
        automaton=fsm_from_sere(tail_car(expr));
        if (fsm_edges_from(automaton, fsm_initial(automaton))==Nil) {
          if (fsm_is_accepting(automaton, fsm_initial(automaton))) {
            result=process_expr(tail_cdr(expr),p);
            break;
          }
          printf("SERE "); print_node(stdout,expr);
          printf(" has no possible matches, formula ≡ ⊥\n");
          result=falseexp;
          break;
        }
//        printf("SERE "); print_node(stdout, tail_car(expr));
//        printf(" translates to:\n"); print_fsm(stdout, automaton);
//        printf("and is followed by: "); print_node(stdout, tail_cdr(expr));
//        printf("\n");
        right=process_expr(tail_cdr(expr),p);
        mod_stub_add_fsm(p->stub, automaton);
        node_ptr state_iter=fsm_states(automaton);
        while (state_iter) {
          node_ptr state=car(state_iter);
          node_ptr edge_iter=fsm_edges_from(automaton, state);
          disjunction=Nil;
          while (edge_iter) {
            node_ptr edge=car(edge_iter);
            process_simple_expr(edge_label(edge), p);
            if (fsm_is_accepting(automaton, edge_to(edge)) &&
                fsm_edges_from(automaton, edge_to(edge))!=Nil) {
              disjunction=expr_add(disjunction,OR,
                                   and(edge_label(edge),
                                       or(next(edge_to(edge)),right)));
            } else if (fsm_is_accepting(automaton, edge_to(edge)) &&
                       fsm_edges_from(automaton, edge_to(edge))==Nil) {
              disjunction=expr_add(disjunction,OR,
                                   and(edge_label(edge),right));
            } else {
              disjunction=expr_add(disjunction,OR,
                                   and(edge_label(edge), next(edge_to(edge))));
            }
            edge_iter=cdr(edge_iter);
          }
          if (disjunction!=Nil) {
            conjunction=expr_add(conjunction,AND,implies(state,disjunction));
          }
          state_iter=cdr(state_iter);
        }
        if (conjunction==Nil) {
          printf("Error: nothing to add to trans in:\n");
          print_fsm(stdout, automaton); printf("(created from "); print_node(stdout, tail_car(expr)); printf(")\n");
          exit(1);
        }
        mod_stub_add_trans(p->stub, conjunction);
#ifdef MYSEMANTICS
        if (fsm_is_accepting(automaton, fsm_initial(automaton))) {
          result=or(right,fsm_initial(automaton));
        } else {
          result=fsm_initial(automaton);
        }
#else
        result=fsm_initial(automaton);
#endif
        free_fsm(automaton);
      } else if (node_get_type(car(expr))==ATOM ||
                 node_get_type(car(expr))==DOT ||
                 node_get_type(car(expr))==ARRAY) {
        result=process_simple_expr(expr,p);
      } else if (node_get_type(car(expr))==PSL_SERE ||
                 node_get_type(car(expr))==PSL_SERECONCAT ||
                 node_get_type(car(expr))==PSL_SEREFUSION ||
                 node_get_type(car(expr))==PSL_SERECOMPOUND ||
                 node_get_type(car(expr))==PSL_SEREREPEATED) {
        result=process_expr(tailimpl(car(expr),falseexp),p);
      } else {
        fprintf(stderr,"Uncaught negation in property that should be in positive normal form; this is probably a bug.\n");
        print_node(stderr,expr);
        fprintf(stderr,"\n");
        exit(1);
      }
      break;
    case PSL_SERE:
    case PSL_SERECONCAT:
    case PSL_SEREFUSION:
    case PSL_SERECOMPOUND:
    case PSL_SEREREPEATED:
      result=process_expr(tailex(expr,trueexp),p);
      break;
    case UNTIL:
      left=process_expr(car(expr),p);
      right=process_expr(cdr(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"until_%x",expr);
      var=atom(var_name);
      mod_stub_add_var(p->stub,var);
      mod_stub_add_trans(p->stub,iff(var,or(right,and(left,next(var)))));
      mod_stub_add_invar(p->stub,implies(var,or(right,and(left,not(p->stub->fs)))));
      mod_stub_add_invar(p->stub,implies(right,var));
      result=var;
      break;
    case RELEASES:
      left=process_expr(car(expr),p);
      right=process_expr(cdr(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"releases_%x",expr);
      var=atom(var_name);
      mod_stub_add_var(p->stub,var);
      mod_stub_add_trans(p->stub,iff(var,and(right,or(left,next(var)))));
      mod_stub_add_invar(p->stub,implies(var,and(right,or(left,not(p->stub->fs)))));
      mod_stub_add_invar(p->stub,implies(and(left,right),var));
      result=var;
      break;
    case PSL_X:
    case OP_NEXT:
      left=process_expr(car(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"next_%x",expr);
      var=atom(var_name);
      mod_stub_add_var(p->stub,var);
      mod_stub_add_trans(p->stub,iff(var,next(left)));
      mod_stub_add_invar(p->stub,implies(var,not(p->stub->fs)));
      result=var;
      break;
    case PSL_ALWAYS:
    case OP_GLOBAL:
      left=process_expr(car(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"global_%x",expr);
      var=atom(var_name);
      mod_stub_add_var(p->stub,var);
      mod_stub_add_trans(p->stub,iff(var,and(not(p->stub->fs),and(left,next(var)))));
      mod_stub_add_invar(p->stub,implies(var,and(left,not(p->stub->fs))));
      result=var;
      break;
    case OP_FUTURE:
      left=process_expr(car(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"future_%x",expr);
      var=atom(var_name);
      mod_stub_add_var(p->stub,var);
      mod_stub_add_trans(p->stub,iff(var,or(left,next(var))));
      mod_stub_add_invar(p->stub,implies(left,var));
      mod_stub_add_invar(p->stub,implies(var,or(left,not(p->stub->fs))));
      result=var;
      break;
    case SINCE:
      left=process_expr(car(expr),p);
      right=process_expr(cdr(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"since_%x",expr);
      var=atom(var_name);
      result=or(right,and(left,var));
      mod_stub_add_yvar(p->stub,var,result);
      break;
    case TRIGGERED:
      left=process_expr(car(expr),p);
      right=process_expr(cdr(expr),p);
      snprintf(var_name,VAR_NAME_LEN,"triggered_%x",expr);
      var=atom(var_name);
      result=and(right,or(left,var));
      mod_stub_add_zvar(p->stub,var,result);
      break;
    case OP_PREC:
      left=process_expr(car(expr),p);
      snprintf(var_name, VAR_NAME_LEN, "yesterday_%x",expr);
      var=atom(var_name);
      result=var;
      mod_stub_add_yvar(p->stub,var,left);
      break;
    case OP_NOTPRECNOT:
      left=process_expr(car(expr),p);
      snprintf(var_name, VAR_NAME_LEN, "z_%x",expr);
      var=atom(var_name);
      result=var;
      mod_stub_add_zvar(p->stub,var,left);
      break;
    case OP_ONCE:
      left=process_expr(car(expr),p);
      snprintf(var_name, VAR_NAME_LEN, "once_%x",expr);
      var=atom(var_name);
      result=or(left,var);
      mod_stub_add_yvar(p->stub,var,result);
      break;
    case OP_HISTORICAL:
      left=process_expr(car(expr),p);
      snprintf(var_name, VAR_NAME_LEN, "historically_%x",expr);
      var=atom(var_name);
      result=and(left,var);
      mod_stub_add_zvar(p->stub,var,result);
      break;
    case OR:
    case AND:
    case IMPLIES:
    case IFF:
    case XNOR:
    case XOR:
    case UNION:
    case SETIN: 
    case EQUAL:
    case NOTEQUAL:
    case LT:
    case GT:
    case LE:
    case GE: 
    case TIMES:
    case DIVIDE:
    case PLUS :
    case MINUS:
    case MOD: 
    case LSHIFT:
    case RSHIFT:
    case FALSEEXP:
    case TRUEEXP:
    case NUMBER:
    case ATOM:
    case LROTATE:
    case RROTATE:
    case DOT:
    case ARRAY:
      result=process_simple_expr(expr,p);
      break;
    default:
      fprintf(stderr,"Observer construction encountered an invalid node (%d), aborting\n",node_get_type(expr));
      exit(1);
  }
  insert_assoc(p->node_hash,expr,result);
  if (p->stub->toplevel == expr) {
    p->stub->toplevel=result;
  }
  return result;
}

node_ptr process_module(node_ptr mod) {
  node_ptr observers=Nil, obs_mods=Nil, tmp=Nil, fr=Nil;
  node_ptr iter=module_body(mod);
  char obs_name[VAR_NAME_LEN];
  while (iter) {
    switch (node_get_type(car(iter))) {
      case SPEC:
      case INVARSPEC:
      case COMPUTE:
        fprintf(stderr,"Cannot handle other specifications than PSL, aborting\n");
        exit(1);
      case VAR:
      case IVAR:
      case DEFINE:
      case CONSTANTS:
      case ASSIGN:
      case TRANS:
      case INIT:
      case INVAR:
      case FAIRNESS:
      case JUSTICE:
      case COMPASSION:
      case ISA:
        break;
      case PSLSPEC:
      case LTLSPEC:
      {
        proc_data* p=proc_data_create();
//        printf("pnf(¬spec):\n"); print_node(stdout, positive_normal_form(not(car(car(iter))))); printf("\n");
        process_expr(positive_normal_form(not(car(car(iter)))),p);
        snprintf(obs_name,VAR_NAME_LEN,"OBSERVER_%x",car(iter));
        tmp=mod_stub_to_module(p->stub,atom(obs_name));
        snprintf(obs_name,VAR_NAME_LEN,"obs_instance",car(iter));
        observers=cons(colon(atom(obs_name),get_modtype(tmp)),observers);
        obs_mods=cons(tmp,obs_mods);
        proc_data_free(p);
      }
        break;
      default:
        fprintf(stderr,"Unsupported node found in module %s body (%d), aborting\n",
                module_name(mod),node_get_type(car(iter)));
        exit(1);
    }
    iter=cdr(iter);
  }
  iter=module_body(mod);
  while (iter) {
    tmp=cdr(iter);
    while (tmp && (node_get_type(car(tmp)) == PSLSPEC ||
                   node_get_type(car(tmp)) == LTLSPEC)) {
      fr=tmp;
      tmp=cdr(tmp);
      free_node(fr);
    }
    if (cdr(iter) != tmp)
      setcdr(iter,tmp);
    iter=cdr(iter);
  }
  if (module_body(mod) && (node_get_type(car(module_body(mod))) == PSLSPEC ||
                           node_get_type(car(module_body(mod))) == LTLSPEC))
    module_set_body(mod,cdr(module_body(mod)));
  if (observers)
    module_set_body(mod,cons(var(observers),module_body(mod)));
  if (not_safety) {
    fprintf(stderr, "Warning: using an observer for checking a property that is not a syntactical safety property\n");
  }
  return obs_mods;
}

void insert_observers() {
  node_ptr iter=parsed_tree;
  node_ptr obs_mods=Nil;
  node_ptr tmp=Nil;
  init_construction();
  while (iter) {
    obs_mods=process_module(car(iter));
    while (obs_mods) {
      parsed_tree=cons(car(obs_mods),parsed_tree);
//      printf("Observer:\n"); print_node(stdout, car(obs_mods)); printf("\n");
      tmp=obs_mods;
      obs_mods=cdr(obs_mods);
      free_node(tmp);
    }
    iter=cdr(iter);
  }
//  printf("Model with observer:\n");
//  print_node(stdout,parsed_tree);
//  printf("\n");
  finish_construction();
}
