/*
 This file is part of the PSL Observer modification for NuSMV.
 Copyright (C) 2008-2009 Tuomas Launiainen.
 
 This program is free software; you can redistribute it and/or 
 modify it under the terms of the GNU Lesser General Public 
 License as published by the Free Software Foundation; either 
 version 2 of the License, or (at your option) any later version.
 
 This program is distributed in the hope that it will be useful, 
 but WITHOUT ANY WARRANTY; without even the implied warranty of 
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public 
 License along with this library; if not, write to the Free Software 
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.
 
 For more information, see <http://www.tcs.hut.fi/~tlauniai/psl-observer>.
 */ 
#include <stdio.h>
#include <string.h>
#include "observer/fsm.h"
#include "observer/observer_utils.h"
#include "utils/ustring.h"
#include "utils/assoc.h"
#include "utils/NodeList.h"

#define make_edge(a,b,c) cons(a,cons(b,c))
#define free_edge(e) free_node(cdr(e)); free_node(e)

// TODO: Use NuSMV utils for queue/stack

#pragma mark fsm operations

#define state_name_length 40

struct fsm {
  node_ptr initial;
  hash_ptr accepting;
  hash_ptr edges;
  hash_ptr nodes;
};

static int state_no=0;

fsm make_fsm() {
  struct fsm* ret=malloc(sizeof(struct fsm));
  char buf[state_name_length];
  snprintf(buf,state_name_length-1,"state_%d",state_no++);
  ret->initial=atom(buf);
  ret->edges=new_assoc();
  ret->accepting=new_assoc();
  ret->nodes=new_assoc();
  return (fsm)ret;
}

void free_fsm(fsm a) {
  node_ptr state_iter=fsm_states(a);
  while (state_iter) {
    node_ptr state=car(state_iter);
    node_ptr edge_iter=fsm_edges_from(a, state);
    while (edge_iter) {
      node_ptr edge=car(edge_iter);
      free_edge(edge);
      edge_iter=cdr(edge_iter);
    }
    state_iter=cdr(state_iter);
  }
  free_assoc(a->edges);
  free_assoc(a->accepting);
  free_assoc(a->nodes);
  free(a);
}

char* sprint_fsm(fsm a) {
  return NULL;
}

boolean fsm_is_accepting(fsm a, node_ptr s) {
  if (find_assoc(a->accepting, s)) {
    return 1;
  } else {
    return 0;
  }
}

node_ptr fsm_initial(fsm a) {
  return a->initial;
}

void fsm_add_accepting(fsm a, node_ptr s) {
  insert_assoc(a->nodes, s, trueexp);
  insert_assoc(a->accepting, s, trueexp);
}

node_ptr fsm_list_accepting(fsm a) {
  return assoc_get_keys(a->accepting, 0);
}

void fsm_set_edges(fsm a, node_ptr from, node_ptr list) {
  insert_assoc(a->edges, from, list);
  insert_assoc(a->nodes, from, trueexp);
  while (list) {
    if (edge_to(car(list))==fsm_initial(a)) {
      fprintf(stderr,"Error: adding an edge that leads to the initial state of the automaton. This should not happen.\n");
      exit(1);
    }
    insert_assoc(a->nodes, edge_to(car(list)), trueexp);
    list=cdr(list);
  }
}

void fsm_add_edge(fsm a,node_ptr from,node_ptr lbl,node_ptr to) {
  if (to==fsm_initial(a)) {
    fprintf(stderr,"Error: adding an edge that leads to the initial state of the automaton. This should not happen.\n");
    exit(1);
  }
  node_ptr edges=find_assoc(a->edges, from);
  edges=cons(make_edge(from,lbl,to),edges);
  insert_assoc(a->edges, from, edges);
  insert_assoc(a->nodes, from, trueexp);
  insert_assoc(a->nodes, to, trueexp);
}

node_ptr fsm_edges_from(fsm a, node_ptr from) {
  return find_assoc(a->edges, from);
}

void fsm_copy_edges(fsm a, fsm from) {
  node_ptr states=fsm_states(from);
  while (states) {
    node_ptr edges=fsm_edges_from(from, car(states));
    while (edges) {
      fsm_add_edge(a, edge_from(car(edges)), edge_label(car(edges)), edge_to(car(edges)));
      edges=cdr(edges);
    }
    states=cdr(states);
  }
}

void fsm_copy_accepting(fsm a, fsm from) {
  node_ptr accepting=fsm_list_accepting(from);
  while (accepting) {
    fsm_add_accepting(a, car(accepting));
    accepting=cdr(accepting);
  }
}

node_ptr edge_from(node_ptr edge) {
  assert(edge && node_get_type(edge)==CONS && cdr(edge) && node_get_type(cdr(edge))==CONS);
  return car(edge);
}

node_ptr edge_label(node_ptr edge) {
  assert(edge && node_get_type(edge)==CONS && cdr(edge) && node_get_type(cdr(edge))==CONS);
  return car(cdr(edge));
}

node_ptr edge_to(node_ptr edge) {
  assert(edge && node_get_type(edge)==CONS && cdr(edge) && node_get_type(cdr(edge))==CONS);
  return cdr(cdr(edge));
}

node_ptr fsm_states(fsm automaton) {
  return assoc_get_keys(automaton->nodes, 0);
}

unsigned int fsm_size(fsm automaton) {
  node_ptr n=fsm_states(automaton);
  unsigned int i=0;
  while (n) {
    n=cdr(n); i++;
  }
  return i;
}

#pragma mark Other functions

node_ptr nil_closure_edges(fsm a,node_ptr from) {
  stack process_stack=make_stack();
  hash_ptr seen=new_assoc();
  node_ptr closure=Nil;
  stack_push(process_stack, from);
  insert_assoc(seen, from, trueexp);
  while (!stack_empty(process_stack)) {
    node_ptr iter=fsm_edges_from(a, stack_pop(process_stack));
    while (iter) {
      if (edge_label(car(iter))==Nil) {
        if (fsm_is_accepting(a, edge_to(car(iter))))
          fsm_add_accepting(a, from);
        if (!find_assoc(seen, edge_to(car(iter)))) {
          stack_push(process_stack,edge_to(car(iter)));
          insert_assoc(seen, edge_to(car(iter)), trueexp);
        }
      } else {
        closure=cons(make_edge(from,edge_label(car(iter)),
                               edge_to(car(iter))),closure);
      }
      iter=cdr(iter);
    }
  }
  free_stack(process_stack);
  free_assoc(seen);
  return closure;
}

node_ptr nil_closure_states(fsm a,node_ptr from) {
  stack process_stack=make_stack();
  hash_ptr seen=new_assoc();
  node_ptr closure=cons(from,Nil);
  stack_push(process_stack, from);
  insert_assoc(seen, from, trueexp);
  while (!stack_empty(process_stack)) {
    node_ptr iter=fsm_edges_from(a, stack_pop(process_stack));
    while (iter) {
      node_ptr edge=car(iter);
      if (edge_label(edge)==Nil && !find_assoc(seen, edge_to(edge))) {
        stack_push(process_stack,edge_to(edge));
        insert_assoc(seen, edge_to(edge), trueexp);
        closure=cons(edge_to(edge),closure);
      }
      iter=cdr(iter);
    }
  }
  free_stack(process_stack);
  free_assoc(seen);
  return closure;
}

void remove_nil_edges(fsm a) {
  fsm ret=make_fsm();
  ret->initial=a->initial;
  stack reachables=make_stack();
  hash_ptr seen=new_assoc();
  node_ptr n, closure;
  stack_push(reachables, fsm_initial(a));
  insert_assoc(seen, fsm_initial(a), trueexp);
  while (!stack_empty(reachables)) {
    n=stack_pop(reachables);
    closure=nil_closure_edges(a, n);
    fsm_set_edges(ret, n, closure);
    while (closure) {
      if (!find_assoc(seen, edge_to(car(closure)))) {
        stack_push(reachables, edge_to(car(closure)));
        insert_assoc(seen, edge_to(car(closure)), trueexp);
      }
      closure=cdr(closure);
    }
  }
  node_ptr i=fsm_states(ret);
  while (i) {
    if (fsm_is_accepting(a, car(i)))
      fsm_add_accepting(ret, car(i));
    i=cdr(i);
  }  
  free_stack(reachables);
  free_assoc(seen);
  free_assoc(a->accepting);
  free_assoc(a->edges); // TODO: free the edges? it would require deep copying of the hash here.
  free_assoc(a->nodes);
  a->accepting=ret->accepting;
  a->edges=ret->edges;
  a->nodes=ret->nodes;
  free(ret);
}

void remove_rejecting_states(fsm a) {
  hash_ptr edges_to=new_assoc();
  hash_ptr seen=new_assoc();
  stack process_stack=make_stack();
  fsm ret=make_fsm();
  ret->initial=a->initial;
  node_ptr states=fsm_states(a);
  while (states) {
    node_ptr state=car(states);
    node_ptr edges=fsm_edges_from(a, state);
    while (edges) {
      node_ptr edge=car(edges);
      insert_assoc(edges_to, edge_to(edge),
                   cons(edge, find_assoc(edges_to,edge_to(edge))));
      edges=cdr(edges);
    }
    states=cdr(states);
  }
  node_ptr accepting_states=fsm_list_accepting(a);
  while (accepting_states) {
    stack_push(process_stack, car(accepting_states));
    insert_assoc(seen, car(accepting_states), trueexp);
    accepting_states=cdr(accepting_states);
  }
  while (!stack_empty(process_stack)) {
    node_ptr state=stack_pop(process_stack);
    node_ptr edges=find_assoc(edges_to, state);
    while (edges) {
      node_ptr edge=car(edges);
      fsm_add_edge(ret, edge_from(edge), edge_label(edge), edge_to(edge));
      if (!find_assoc(seen, edge_from(edge))) {
        insert_assoc(seen, edge_from(edge), trueexp);
        stack_push(process_stack, edge_from(edge));
      }
      edges=cdr(edges);
    }
  }
  accepting_states=fsm_list_accepting(a);
  while (accepting_states) {
    fsm_add_accepting(ret, car(accepting_states));
    accepting_states=cdr(accepting_states);
  }
  free_assoc(edges_to);
  free_assoc(seen);
  free_stack(process_stack);
  free_assoc(a->accepting);
  free_assoc(a->edges); // TODO: free the edges? it would require deep copying of the hash here.
  free_assoc(a->nodes);
  a->accepting=ret->accepting;
  a->edges=ret->edges;
  a->nodes=ret->nodes;
  free(ret);
}

void print_fsm(FILE* f, struct fsm* a) {
  fprintf(f,"digraph \"fsm\" {\n\t");
  print_node(f,fsm_initial(a));
  fprintf(f," [shape = diamond];\n");
  node_ptr i=fsm_list_accepting(a);
  while (i) {
    fprintf(f,"\t");
    print_node(f,car(i));
    fprintf(f," [shape = doublecircle];\n");
    i=cdr(i);
  }
  i=fsm_states(a);
  while (i) {
    node_ptr j=fsm_edges_from(a, car(i));
    while (j) {
      fprintf(f,"\t");
      print_node(f,car(i));
      fprintf(f," -> ");
      print_node(f,edge_to(car(j)));
      fprintf(f," [label = \"");
      if (edge_label(car(j)))
        print_node(f,edge_label(car(j)));
      else
        fprintf(f,"ε");
      fprintf(f,"\"];\n");
      j=cdr(j);
    }
    i=cdr(i);
  }
  fprintf(f,"}\n");
}

node_ptr compose_state(node_ptr n) {
  assert(node_get_type(n)==CONS && node_get_type(car(n))==ATOM && node_get_type(cdr(n))==ATOM);
  char buf[state_name_length];
  snprintf(buf,state_name_length-1,"states_%x_%x",car(n),cdr(n));
  return atom(buf);
}

fsm process_sere(node_ptr sere) {
  static int q=0;
  char buf[state_name_length];
  fsm automaton=make_fsm();
//  printf("processing sere ");
//  print_node(stdout,sere);
//  printf("\n");
  while (node_get_type(sere) == PSL_SERE ||
         (node_get_type(sere) == PSL_SERECOMPOUND && cdr(sere) == Nil))
    sere=car(sere);
  switch (node_get_type(sere)) {
    case NOT:
      if (node_get_type(car(sere))!=ATOM &&
		  node_get_type(car(sere))!=DOT) {
        fprintf(stderr, "Negation in front of node ");
        print_node(stderr, car(sere));
        fprintf(stderr, "\n");
        exit(1);
      }
  case DOT:
    case ATOM:
    case TRUEEXP:
      snprintf(buf,state_name_length-1,"state_%d",state_no++);
      node_ptr to=atom(buf);
      fsm_add_edge(automaton, fsm_initial(automaton), sere, to);
      fsm_add_accepting(automaton, to);
      break;
    case OR:
    {
      fsm left=process_sere(car(sere));
      fsm right=process_sere(cdr(sere));
      fsm_copy_edges(automaton, left);
      fsm_copy_edges(automaton, right);
      fsm_add_edge(automaton, fsm_initial(automaton), Nil, fsm_initial(left));
      fsm_add_edge(automaton, fsm_initial(automaton), Nil, fsm_initial(right));
      fsm_copy_accepting(automaton, left);
      fsm_copy_accepting(automaton, right);
      free_fsm(left);
      free_fsm(right);
    }
      break;
    case AND:
      free_fsm(automaton);
      return process_sere(or(intersect(concat(car(sere),kleene(trueexp)), cdr(sere)),
                             intersect(concat(cdr(sere),kleene(trueexp)), car(sere))));
    case PSL_AMPERSANDAMPERSAND:
    {
//      printf("(");
//      print_node(stdout, car(sere));
//      printf("\nand\n");
//      print_node(stdout, cdr(sere));
//      printf("\n");
      stack process_stack=make_stack();
      hash_ptr seen=new_assoc();
      fsm left=process_sere(car(sere));
      fsm right=process_sere(cdr(sere));
      remove_nil_edges(left);
      remove_nil_edges(right);
//      print_fsm(stdout, automaton);
//      printf("\n");
      node_ptr current_pair=cons(fsm_initial(left),fsm_initial(right));
      node_ptr composed=compose_state(current_pair);
      fsm_add_edge(automaton, fsm_initial(automaton), Nil, composed);
      stack_push(process_stack,current_pair);
      insert_assoc(seen, composed, trueexp);
      while (!stack_empty(process_stack)) {
        current_pair=stack_pop(process_stack);
        if (fsm_is_accepting(left, car(current_pair)) &&
            fsm_is_accepting(right, cdr(current_pair))) {
          fsm_add_accepting(automaton, compose_state(current_pair));
        }
        node_ptr edge_iter=fsm_edges_from(left, car(current_pair));
        while(edge_iter) {
          node_ptr edge=car(edge_iter);
          node_ptr other_edge_iter=fsm_edges_from(right, cdr(current_pair));
          while (other_edge_iter) {
            node_ptr other_edge=car(other_edge_iter);
            node_ptr new_pair=cons(edge_to(edge),edge_to(other_edge));
            fsm_add_edge(automaton, compose_state(current_pair),
                         and(edge_label(edge),edge_label(other_edge)),
                         compose_state(new_pair));
            if (!find_assoc(seen, compose_state(new_pair))) {
              insert_assoc(seen, compose_state(new_pair), trueexp);
              stack_push(process_stack, new_pair);
            }
            other_edge_iter=cdr(other_edge_iter);
          }
          edge_iter=cdr(edge_iter);
        }
        edge_iter=fsm_edges_from(automaton, cdr(current_pair));
      }
      free_assoc(seen);
      free_stack(process_stack);
      free_fsm(left);
      free_fsm(right);
    }
      break;
    case PSL_SERECONCAT:
    {
      fsm left=process_sere(car(sere));
      fsm right=process_sere(cdr(sere));
      fsm_copy_edges(automaton, left);
      fsm_copy_edges(automaton, right);
      fsm_copy_accepting(automaton, right);
      fsm_add_edge(automaton, fsm_initial(automaton), Nil, fsm_initial(left));
      node_ptr accepting=fsm_list_accepting(left);
      while (accepting) {
        fsm_add_edge(automaton, car(accepting), Nil, fsm_initial(right));
        accepting=cdr(accepting);
      }
      free_fsm(left);
      free_fsm(right);
    }
      break;
    case PSL_SEREFUSION:
    {
      fsm left=process_sere(car(sere));
      fsm right=process_sere(cdr(sere));
      remove_nil_edges(left);
      remove_nil_edges(right);
//      printf("Fusing "); print_node(stdout, sere); printf("\n");
//      print_fsm(stdout, left); printf("\n");
//      print_fsm(stdout, right); printf("\n");
      fsm_copy_edges(automaton, left);
      fsm_copy_edges(automaton, right);
      fsm_add_edge(automaton, fsm_initial(automaton), Nil, fsm_initial(left));
      fsm_copy_accepting(automaton, right); // TODO: does this take care of an accepting initial state of right? (see semantics)
//      printf("Let's begin with:\n");
//      print_fsm(stdout, automaton); printf("\n");
      node_ptr states=fsm_states(left);
      while (states) {
        node_ptr edges=fsm_edges_from(left, car(states));
        while (edges) {
          if (fsm_is_accepting(left, edge_to(car(edges)))) {
            node_ptr redges=fsm_edges_from(right, fsm_initial(right));
            while (redges) {
              fsm_add_edge(automaton, edge_from(car(edges)),
                           and(edge_label(car(edges)), edge_label(car(redges))),
                           edge_to(car(redges)));
              redges=cdr(redges);
            }
          }
          edges=cdr(edges);
        }
        states=cdr(states);
      }
      free_fsm(left);
      free_fsm(right);
    }
      break;
    case PSL_SEREREPEATED: // NOTE: I'm assuming only plain * or +
    {
      if ((!is_kleene(sere) && !is_plusrepeat(sere)) ||
          repeat_amout(sere)!=Nil) {
        fprintf(stderr,"Only [+] and [*] repeats are supported\n");
        exit(1);
      }
      fsm body=process_sere(repeat_body(sere));
      fsm_copy_edges(automaton, body);
      fsm_copy_accepting(automaton, body);
      fsm_add_edge(automaton, fsm_initial(automaton), Nil, fsm_initial(body));
      node_ptr accepting=fsm_list_accepting(automaton);
      while (accepting) {
        fsm_add_edge(automaton, car(accepting), Nil, fsm_initial(body));
        accepting=cdr(accepting);
      }
      if (is_kleene(sere)) {
        fsm_add_accepting(automaton, fsm_initial(automaton));
      }
      free_fsm(body);
    }
      break;
    default:
		fprintf(stderr, "While processing node "); print_node(stderr, sere); fprintf(stderr, ":\n");
		fatal("Invalid node in PSL expression: %d\n",node_get_type(sere));
		break;
  }
  remove_nil_edges(automaton);
  remove_rejecting_states(automaton);
//  print_node(stdout, sere); printf(":\n");
//  print_fsm(stdout, automaton); printf("\n");
//  printf("Returning automaton with %d states\n",fsm_size(automaton));
  return automaton;
}

fsm fsm_from_sere(node_ptr sere) {
  fsm result=process_sere(sere);
  remove_nil_edges(result);
  remove_rejecting_states(result);
  return result;
}
