/*
 * Partitions and partition refinement things by Tommi Junttila
 */

#include <assert.h>
#include "userconfig.H"
#include "TJ_partition.H"

#if defined(SYMMETRY) && (SYMMINTEGRATION == 7)


/* 
 * A stack with fixed capacity
 */
class CellStack {
public:
  CellStack() {entries = 0; cursor = 0; last = 0;}
  ~CellStack() {if(entries) free(entries);}
  init(int k) {
    TJ_DEBUG_ASSERT(k > 0);
    if(entries) free(entries);
    entries = (CellEntry**)malloc(k * sizeof(CellEntry*));
    cursor = entries;
    last = entries + k - 1;
  }
  bool empty() const {return(cursor == entries); }
  CellEntry *pop() {
    TJ_DEBUG_ASSERT(cursor > entries);
    cursor--;
    return *cursor;
  }
  void push(CellEntry *obj) {
    TJ_DEBUG_ASSERT(cursor <= last);
    *cursor = obj;
    cursor++;
  }
private:
  CellEntry **entries, **cursor, **last;
};

static CellStack neighbourhood_stack;




void TJ_Partition::init()
{
  unsigned int i;
  N = Places[0]->cnt + Transitions[0]->cnt;
  assert(N > 0);

  elements = (Node **)malloc(N * sizeof(Node *));
  for(i = 0; i < Places[0]->cnt; i++)
    elements[i] = Places[i];
  for(i = 0; i < Transitions[0]->cnt; i++)
    elements[Places[0]->cnt + i] = Transitions[i];

  for(i = 0; i < N; i++)
    elements[i]->TJ_ival = 0;

  cells = (CellEntry*)malloc(N * sizeof(CellEntry));

  cells[0].first = 0;
  cells[0].length = N;
  cells[0].max_ival = 0;
  cells[0].max_ival_count = 0;
  cells[0].splitted = false;
  cells[0].in_neighbourhood_stack = false;
  cells[0].initial = true;
  cells[0].next = 0;
  first_cell = &cells[0];

  for(i = 1; i < N; i++) {
    cells[i].first = 0;
    cells[i].length = 0;
    cells[i].max_ival = 0;
    cells[i].max_ival_count = 0;
    cells[i].splitted = false;
    cells[i].in_neighbourhood_stack = false;
    cells[i].initial = false;
    cells[i].next = &cells[i+1];
  }
  if(N > 1) {
    cells[N-1].next = 0;
    free_cells = &cells[1];
  } else {
    free_cells = 0;
  }

  for(i = 0; i < N; i++)
    elements[i]->TJ_in_cell = first_cell;

  split_set.init(N);

  neighbourhood_stack.init(N);
}



bool TJ_Partition::refine(unsigned int (*inv)(unsigned int))
{
  bool refined = false;
  unsigned int i;
  CellEntry *cell;

  for(i = 0; i < Places[0]->cnt; i++)
    Places[i]->TJ_ival = inv(i);
  for(i = 0; i < Transitions[0]->cnt; i++)
    Transitions[i]->TJ_ival = inv(i + Places[0]->cnt);

  for(cell = first_cell; cell; ) {
    CellEntry * const next_cell = cell->next;
    if(cell->length > 1 && shellsort_cell(cell)) {
      split_cell_plain(cell);
      refined = true;
    }
    cell = next_cell;
  }
  
  for(i = 0; i < Places[0]->cnt; i++)
    Places[i]->TJ_ival = 0;
  for(i = 0; i < Transitions[0]->cnt; i++)
    Transitions[i]->TJ_ival = 0;

  return refined;
}


/*
 * Assumes that the starting partition is equitable!
 */
void TJ_Partition::refine_according_to_marking()
{
  unsigned int i;
  CellEntry *cell;

#if defined(TJ_DEBUG)
  for(cell = first_cell; cell; cell = cell->next) {
    assert(cell->splitted == false);
    assert(cell->in_neighbourhood_stack == false);
    Node **np = elements + cell->first;
    for(i = cell->length; i > 0; i--) {
      assert((*np)->TJ_ival == 0);
      assert((*np)->TJ_in_cell == cell);
      np++;
    }
  }
#endif

  cell = first_cell;
  while(cell->first < Places[0]->cnt) {
    if(cell->length == 1) {
      cell = cell->next;
      continue;
    }
    unsigned int max_ival = 0;
    unsigned int max_ival_count = 0;
    Node **np = elements + cell->first;
    for(i = cell->length; i > 0; i--) {
      const unsigned int marking = ((Place*)(*np))->current_marking;
      (*np)->TJ_ival = marking;
      np++;
      if(marking > max_ival) {
	max_ival = marking;
	max_ival_count = 1;
      }
      else if(marking == max_ival)
	max_ival_count++;
    }
    if(max_ival_count == cell->length) {
      clear_ivs(cell);
      cell = cell->next;
      continue;
    }
    CellEntry *next_cell = cell->next;
    if(max_ival == 1) {
      /* specialized splitting for cells with binary invariant values */
      CellEntry *cell2 = sortandsplit_cell1(cell);
      TJ_DEBUG_ASSERT(cell != cell2);
    }
    else if(max_ival <= 255) {
      /* specialized splitting for cells with invariant values < 256 */
      CellEntry *cell2 = sortandsplit_cell255(cell);
      TJ_DEBUG_ASSERT(cell != cell2);
    }
    else {
      /* generic sorting and splitting */
      bool many = shellsort_cell(cell);
      TJ_DEBUG_ASSERT(many);
      CellEntry *cell2 = split_cell(cell);
      TJ_DEBUG_ASSERT(cell != cell2);
    }
    cell = next_cell;
    do_refine_to_equitable2();
  }

#if defined(TJ_DEBUG)
  for(i = 0; i < Places[0]->cnt; i++)
    assert(Places[i]->TJ_ival == 0);
  for(i = 0; i < Transitions[0]->cnt; i++)
    assert(Transitions[i]->TJ_ival == 0);
  for(cell = first_cell; cell; cell = cell->next) {
    assert(cell->splitted == false);
    assert(cell->in_neighbourhood_stack == false);
  }
#endif
}




void TJ_Partition::print(FILE *fp)
{
  unsigned int e, i;
  CellEntry *ce;

  ce = first_cell;
  assert(ce);
  
  e = 0;
  fprintf(fp, "[");
  while(ce) {
    assert(e == ce->first);
    fprintf(fp, "{");
    for(i = ce->length; i > 0; i--) {
      fprintf(fp, "%s", elements[e]->name);
      if(i > 1)
	fprintf(fp, ",");
      e++;
    }
    fprintf(fp, "}");
    ce = ce->next;
  }
  fprintf(fp, "]");
}

void TJ_Partition::print_signature(FILE *fp)
{
  CellEntry *ce;

  ce = first_cell;
  assert(ce);
  
  fprintf(fp, "[");
  while(ce) {
    fprintf(fp, "%u", ce->length);
    ce = ce->next;
    if(ce) fprintf(fp, ",");
  }
  fprintf(fp, "]");
}





/*
 * Assumes that the invariant values are NOT the same
 * and that the cell contains more than one element
 */
CellEntry *TJ_Partition::sortandsplit_cell1(CellEntry *cell) {
  Node **np0, **np1;

#if defined(TJ_DEBUG)
  assert(cell->first + cell->length <= N);
  bool found0 = false, found1 = false;
  for(unsigned int i = 0; i < cell->length; i++) {
    if(elements[cell->first + i]->TJ_ival == 0)
      found0 = true;
    else if(elements[cell->first + i]->TJ_ival == 1)
      found1 = true;
    else
      assert(false);
  }
  assert(found0);
  assert(found1);
#endif

  TJ_DEBUG_ASSERT(cell->length > 1);

  /* allocate new cell */
  CellEntry *new_cell = free_cells;
  TJ_DEBUG_ASSERT(new_cell);
  free_cells = new_cell->next;

  /* sort vertices in cell according to the invariant values */
  np0 = elements + cell->first;
  np1 = np0 + cell->length;
  while(np1 > np0) {
    Node * const node = *np0;
    const unsigned int ival = node->TJ_ival;
    node->TJ_ival = 0;
    TJ_DEBUG_ASSERT(ival <= 1);
    TJ_DEBUG_ASSERT(node->TJ_in_cell == cell);
    if(ival == 0) {
      np0++;
    } else {
      np1--;
      *np0 = *np1;
      *np1 = node;
      node->TJ_in_cell = new_cell;
    }
  }

  TJ_DEBUG_ASSERT(np1 != elements + cell->first);
  TJ_DEBUG_ASSERT(np0 != elements + cell->first + cell->length);

  /* update cell parameters */
  new_cell->first = np1 - elements;
  new_cell->length = cell->length - (new_cell->first - cell->first);
  new_cell->next = cell->next;
  cell->length = new_cell->first - cell->first;
  cell->next = new_cell;

  if(true) {
    if(cell->splitted == true) {
      TJ_DEBUG_ASSERT(new_cell->splitted == false);
      new_cell->splitted = true;      
      split_set.insert(new_cell->first);
    } else {
      if(new_cell->length < cell->length) {
	TJ_DEBUG_ASSERT(new_cell->splitted == false);
	new_cell->splitted = true;      
	split_set.insert(new_cell->first);
      } else {
	TJ_DEBUG_ASSERT(cell->splitted == false);
	cell->splitted = true;      
	split_set.insert(cell->first);
      }
    }
  }
  else {
    assert(new_cell->splitted == false);
    split_set.insert(new_cell->first);
    new_cell->splitted = true;
  }
  return new_cell;
}



/*
 * Tables and a subroutine for distribution count sorting
 */
static unsigned int count[256] = {
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};
static unsigned int start[256];

static void cumulate_count() {
  unsigned int i, new_start, *sp, *cp;
  sp = start;
  cp = count;
  new_start = 0;
  for(i = 256; i > 0; i--) {
    *sp = new_start;
    new_start += *cp;
    sp++;
    cp++;
  }
}


/*
 * Distribution count sorting of cells with invariant values less than 256 
 */
CellEntry *TJ_Partition::sortandsplit_cell255(CellEntry *cell) {
  Node *node, **np;
  unsigned int ival, ival2;
  unsigned int i, j;

  TJ_DEBUG_ASSERT(cell->first + cell->length <= N);

  if(cell->length == 1) {
    elements[cell->first]->TJ_ival = 0;
    return cell;
  }
  
#if defined(TJ_DEBUG)
  for(i = 0; i < 256; i++)
    assert(count[i] == 0);
#endif

  np = elements + cell->first;
  ival = (*np)->TJ_ival;
  TJ_DEBUG_ASSERT(ival <= 255);
  count[ival]++;
  TJ_DEBUG_ASSERT((*np)->TJ_in_cell == cell);
  np++;
  bool same = true;
  for(i = cell->length - 1; i > 0; i--) {
    TJ_DEBUG_ASSERT((*np)->TJ_in_cell == cell);
    ival2 = (*np)->TJ_ival;
    TJ_DEBUG_ASSERT(ival2 <= 255);
    count[ival2]++;
    if(ival2 != ival) {
      same = false;
    }
    np++;
  }
  TJ_DEBUG_ASSERT(!same);
  if(same) {
    TJ_DEBUG_ASSERT(count[ival] == cell->length);
    count[ival] = 0;
    clear_ivs(cell);
    return cell;
  }

  cumulate_count();

  TJ_DEBUG_ASSERT(start[255] + count[255] == cell->length);

  /* do the sorting magic here!!! */
  for(i = 0; i < 256; i++) {
    np = elements + cell->first + start[i];
    for(j = count[i]; j > 0; j--) {
      while(true) {
	node = *np;
	ival = node->TJ_ival;
	if(ival == i)
	  break;
	TJ_DEBUG_ASSERT(ival > i);
	TJ_DEBUG_ASSERT(count[ival] > 0);
	*np = elements[cell->first + start[ival]];
	elements[cell->first + start[ival]] = node;
	start[ival]++;
	count[ival]--;
      }
      np++;
    }
    count[i] = 0;
  }

#if defined(TJ_DEBUG) 
  for(i = 0; i < 256; i++)
    assert(count[i] == 0);
#endif

#ifdef VERBOSEDEBUG
  ep = elements + cell->first;
  //fprintf(stderr, "\n");
  for(i = cell->length; i > 0; i--, ep++)
    fprintf(stderr, "%u ", invariant_values[*ep]);
  fprintf(stderr, "\n");
#endif

  /* split cell */
  CellEntry *cell2 = split_cell(cell);
  TJ_DEBUG_ASSERT(cell2 != cell);
  return cell2;
}



bool TJ_Partition::shellsort_cell(CellEntry *cell) {
  unsigned int i, h, j;
  unsigned int iv;
  Node **np, *node;

  TJ_DEBUG_ASSERT(cell->first + cell->length <= N);

  if(cell->length == 1)
    return false;

  np = elements + cell->first;
  iv = (*np)->TJ_ival;
  TJ_DEBUG_ASSERT((*np)->TJ_in_cell == cell);
  np++;
  bool same = true;
  for(i = cell->length - 1; i > 0; i--) {
    TJ_DEBUG_ASSERT((*np)->TJ_in_cell == cell);
    if((*np)->TJ_ival != iv) {
      same = false;
      break;
    }
    np++;
  }
  if(same)
    return false;

  np = elements + cell->first;
  for(h = 1; h <= cell->length/9; h = 3*h + 1)
    ;
  for( ; h > 0; h = h/3) {
    for(i = h; i < cell->length; i++) {
      node = np[i];
      iv = node->TJ_ival;
      j = i;
      while(j >= h && np[j-h]->TJ_ival > iv) {
        np[j] = np[j-h];
        j -= h;
      }
      np[j] = node;
    }
  }
  return true;
}


void TJ_Partition::clear_ivs(CellEntry *cell) {
  unsigned int i;
  Node **np;

  np = elements + cell->first;
  for(i = cell->length; i > 0; ) {
    (*np)->TJ_ival = 0;
    i--;
    np++;
  }
}

CellEntry *TJ_Partition::split_cell(CellEntry *cell) {
  Node **fnp, **lnp;
  unsigned int ival;

  TJ_DEBUG_ASSERT(cell->length > 1);

  while(true) {
    fnp = elements + cell->first;
    lnp = fnp + cell->length;
    ival = (*fnp)->TJ_ival;
    (*fnp)->TJ_ival = 0;
    (*fnp)->TJ_in_cell = cell;
    fnp++;
    while(fnp < lnp) {
      if((*fnp)->TJ_ival != ival)
	break;
      (*fnp)->TJ_ival = 0;
      (*fnp)->TJ_in_cell = cell;
      fnp++;
    }
    if(fnp == lnp)
      break;
    CellEntry *new_cell = free_cells;
    TJ_DEBUG_ASSERT(new_cell);
    free_cells = new_cell->next;
    new_cell->first = fnp - elements;
    new_cell->length = cell->length - (new_cell->first - cell->first);
    assert(new_cell->splitted == false);
    split_set.insert(new_cell->first);
    new_cell->splitted = true;
    new_cell->next = cell->next;
    cell->length = new_cell->first - cell->first;
    cell->next = new_cell;
    cell = new_cell;
  }
  return cell;
}


void TJ_Partition::split_cell_plain(CellEntry *cell) {
  Node **fnp, **lnp;
  unsigned int ival;

  TJ_DEBUG_ASSERT(cell->length > 1);

  while(true) {
    fnp = elements + cell->first;
    lnp = fnp + cell->length;
    ival = (*fnp)->TJ_ival;
    (*fnp)->TJ_ival = 0;
    (*fnp)->TJ_in_cell = cell;
    fnp++;
    while(fnp < lnp) {
      if((*fnp)->TJ_ival != ival)
	break;
      (*fnp)->TJ_ival = 0;
      (*fnp)->TJ_in_cell = cell;
      fnp++;
    }
    if(fnp == lnp)
      break;
    CellEntry *new_cell = free_cells;
    TJ_DEBUG_ASSERT(new_cell);
    free_cells = new_cell->next;
    new_cell->first = fnp - elements;
    new_cell->length = cell->length - (new_cell->first - cell->first);
    new_cell->next = cell->next;
    cell->length = new_cell->first - cell->first;
    cell->next = new_cell;
    cell = new_cell;
  }
  return;
}


void TJ_Partition::refine_to_equitable2() {
#if defined(TJ_DEBUG)
  for(CellEntry *cell = first_cell; cell; cell = cell->next) {
    assert(cell->splitted == false);
    assert(cell->in_neighbourhood_stack == false);
  }
#endif

  split_set.clear();
  for(CellEntry *cell = first_cell; cell; cell = cell->next) {
    assert(cell->splitted == false);
    cell->splitted = true;
    split_set.insert(cell->first);
  }

  do_refine_to_equitable2();
}

void TJ_Partition::refine_to_equitable2(CellEntry *new_singleton_cell)
{
#if defined(TJ_DEBUG)
  assert(new_singleton_cell->length == 1);
  for(CellEntry *cell = first_cell; cell; cell = cell->next) {
    assert(cell->splitted == false);
    assert(cell->in_neighbourhood_stack == false);
  }
#endif
  
  split_set.clear();
  assert(new_singleton_cell->splitted == false);
  split_set.insert(new_singleton_cell->first);
  new_singleton_cell->splitted = true;

  do_refine_to_equitable2();
}


void TJ_Partition::split_neighbourhood() {
  while(!neighbourhood_stack.empty()) {
    CellEntry * const cell = neighbourhood_stack.pop();
    TJ_DEBUG_ASSERT(cell->in_neighbourhood_stack);
    cell->in_neighbourhood_stack = false;
    TJ_DEBUG_ASSERT(cell->length != 1);
    /*
    if(cell->length == 1) {
      invariant_values[elements[cell->first]] = 0;
      cell->max_ival = 0;
      cell->max_ival_count = 0;
      continue;
    }
    */
    if(cell->max_ival_count == cell->length) {
      clear_ivs(cell);
      cell->max_ival = 0;
      cell->max_ival_count = 0;
      continue;
    }
    //fprintf(stderr, "s");
    if(cell->max_ival == 1) {
      /* specialized splitting for cells with binary invariant values */
      CellEntry *cell2 = sortandsplit_cell1(cell);
      TJ_DEBUG_ASSERT(cell != cell2);
    }
    else if(cell->max_ival <= 255) {
      /* specialized splitting for cells with invariant values < 256 */
      CellEntry *cell2 = sortandsplit_cell255(cell);
      TJ_DEBUG_ASSERT(cell != cell2);
    }
    else {
      /* generic sorting and splitting */
      bool many = shellsort_cell(cell);
      TJ_DEBUG_ASSERT(many);
      CellEntry *cell2 = split_cell(cell);
      TJ_DEBUG_ASSERT(cell != cell2);
    }
    cell->max_ival = 0;
    cell->max_ival_count = 0;
  }
}

void TJ_Partition::do_refine_to_equitable2()
{
  CellEntry *cell;
  Arc **arcp;
  TJ_DEBUG_ASSERT(neighbourhood_stack.empty());

  TJ_DEBUG_ASSERT(!split_set.is_empty());

  while(!split_set.is_empty()) {
    unsigned int start = split_set.remove();
    cell = elements[start]->TJ_in_cell;
    TJ_DEBUG_ASSERT(cell->first == start);
    TJ_DEBUG_ASSERT(cell->splitted == true);
    cell->splitted = false;

#if defined(TJ_DEBUG)
  for(CellEntry *dcell = first_cell; dcell; dcell = dcell->next) {
    assert(dcell->max_ival == 0);
    assert(dcell->max_ival_count == 0);
    Node **np = elements + dcell->first;
    for(unsigned int i = dcell->length; i > 0; i--) {
      assert((*np)->TJ_ival == 0);
      assert((*np)->TJ_in_cell == dcell);
      np++;
    }
  }
#endif
    
    Node * const first_node = elements[cell->first];
    { /* preset */
      unsigned short int first_arc = 0, last_arc;
      unsigned int multiplicity;
      while(first_arc < first_node->NrOfArriving) {
	{ /* find last arc with the same multiplicity */
	  arcp = first_node->ArrivingArcs + first_arc;
	  multiplicity = (*arcp)->Multiplicity;
	  Arc ** const end_arcp =
	    first_node->ArrivingArcs + first_node->NrOfArriving;
	  while(arcp < end_arcp && (*arcp)->Multiplicity == multiplicity)
	    arcp++;
	  arcp--;
	  last_arc = arcp - first_node->ArrivingArcs;
	}

	Node **np = elements + cell->first;
	for(unsigned int i = cell->length; i > 0; i--) {
	  Node * const node = *np;
	  np++;
	  TJ_DEBUG_ASSERT(node->NrOfArriving == first_node->NrOfArriving);
	  
	  arcp = node->ArrivingArcs + first_arc;
	  for(int j = last_arc - first_arc; j >= 0; j--) {
	    TJ_DEBUG_ASSERT(multiplicity == (*arcp)->Multiplicity);
	    Node * const source = (*arcp)->Source;
	    arcp++;
	    if(source->TJ_in_cell->length == 1)
	      continue;
	    const unsigned int ival = source->TJ_ival + 1;
	    source->TJ_ival = ival;
	    CellEntry * const in_cell = source->TJ_in_cell;
	    if(ival > in_cell->max_ival) {
	      in_cell->max_ival = ival;
	      in_cell->max_ival_count = 1;
	    } else if(ival == in_cell->max_ival) {
	      in_cell->max_ival_count++;
	    }
	    if(!in_cell->in_neighbourhood_stack) {
	      in_cell->in_neighbourhood_stack = true;
	      neighbourhood_stack.push(in_cell);
	    }
	  }
	}
	split_neighbourhood();
	first_arc = last_arc + 1;
      }
    } /* preset */
    { /* postset */
      TJ_DEBUG_ASSERT(!cell->splitted);
      unsigned short int first_arc = 0, last_arc;
      unsigned int multiplicity;
      while(first_arc < first_node->NrOfLeaving) {
	{ /* find last with the same multiplicity */
	  arcp = first_node->LeavingArcs + first_arc;
	  Arc ** const end_arcp =
	    first_node->LeavingArcs + first_node->NrOfLeaving;
	  multiplicity = (*arcp)->Multiplicity;
	  while(arcp < end_arcp && (*arcp)->Multiplicity == multiplicity)
	    arcp++;
	  arcp--;
	  last_arc = arcp - first_node->LeavingArcs;
	}
	
	Node **np = elements + cell->first;
	for(unsigned int i = cell->length; i > 0; i--) {
	  Node * const node = *np;
	  np++;
	  TJ_DEBUG_ASSERT(node->NrOfLeaving == first_node->NrOfLeaving);
	  
	  arcp = node->LeavingArcs + first_arc;
	  for(int j = last_arc - first_arc; j >= 0; j--) {
	    TJ_DEBUG_ASSERT(multiplicity == (*arcp)->Multiplicity);
	    Node * const destination = (*arcp)->Destination;
	    arcp++;
	    if(destination->TJ_in_cell->length == 1)
	      continue;
	    const unsigned int ival = destination->TJ_ival + 1;
	    destination->TJ_ival = ival;
	    CellEntry * const in_cell = destination->TJ_in_cell;
	    if(ival > in_cell->max_ival) {
	      in_cell->max_ival = ival;
	      in_cell->max_ival_count = 1;
	    } else if(ival == in_cell->max_ival) {
	      in_cell->max_ival_count++;
	    }
	    if(!in_cell->in_neighbourhood_stack) {
	      in_cell->in_neighbourhood_stack = true;
	      neighbourhood_stack.push(in_cell);
	    }
	  }
	}
	split_neighbourhood();
	first_arc = last_arc + 1;
      }
    } /* postset */

    TJ_DEBUG_ASSERT(!cell->splitted);
  }
}






#endif
