#include <math.h>
#include <string.h>
#include <cassert>
#include "str.hh"

namespace str {

  bool
  read_line(std::string *str, FILE *file, bool do_chomp)
  {
    char *ptr;
    char buf[4096];

    str->erase();
    while (1) {
      ptr = fgets(buf, 4096, file);
      if (!ptr)
	break;

      str->append(buf);
      if ((*str)[str->length() - 1] == '\n') {
	break;
      }
    }

    if (ferror(file) || str->length() == 0)
      return false;

    if (do_chomp)
      chomp(str);

    return true;
  }

  bool
  read_string(std::string *str, size_t length, FILE *file)
  {
    assert(length >= 0);

    str->erase();
    if (length == 0)
      return true;
    str->reserve(length);

    // Read the string
    char buf[4096];
    size_t buf_size = 4096;
    while (length > 0) {
      if (length < buf_size)
	buf_size = length;
      size_t ret = fread(buf, buf_size, 1, file);
      if (ret != 1)
	return false;
    
      str->append(buf, buf_size);
      length -= buf_size;
    }

    return true;
  }

  void
  chomp(std::string *str)
  {
    if (str->empty())
      return;

    if ((*str)[str->length() - 1] == '\n')
      str->resize(str->length() - 1);
  }

  void
  clean(std::string *str, const char *chars)
  {
    int i;

    // Clean trailing chars
    for (i = str->length(); i > 0; i--)
      if (!strchr(chars, (*str)[i - 1]))
	break;
    str->erase(i);

    // Clean leading chars
    for (i = 0; i < (int)str->length(); i++)
      if (!strchr(chars, (*str)[i]))
	break;
    str->erase(0, i);
  }

  void
  split(const std::string *str, const char *delims, bool group,
	std::vector<std::string> *strings, int fields)
  {
    int begin = 0;
    int end = 0;

    strings->clear();
    while (begin < (int)str->length()) {

      // If 'fields' fields was requested and, this is the last field,
      // include the rest.
      if (fields > 0 && (int)strings->size() == fields - 1) {
	strings->push_back(str->substr(begin));
	break;
      }

      // Find the string before the next delim
      while (end < (int)str->length() && !strchr(delims, (*str)[end]))
	end++;
      strings->push_back(str->substr(begin, end - begin));

      // Eat the delim or group of delims
      end++;
      if (group)
	while (end < (int)str->length() && strchr(delims, (*str)[end]))
	  end++;

      begin = end;
    }
  }

  long
  str2long(const char *str, bool *ok)
  {
    char *endptr;

    long value = strtol(str, &endptr, 10);
    if (value == LONG_MIN || value == LONG_MAX) {
      fprintf(stderr, "str2long(): value out of range\n");
      exit(1);
    }

    if (*str == '\0' || *endptr != '\0')
      *ok = false;

    return value;
  }

  double
  str2float(const char *str, bool *ok)
  {
    char *endptr;

    float value = strtod(str, &endptr);
#ifdef HUGE_VALF
    if (value == HUGE_VALF || value == -HUGE_VALF) {
      fprintf(stderr, "str2float(): value out of range\n");
      exit(1);
    }
#else
    if (value == HUGE_VAL || value == -HUGE_VAL) {
      fprintf(stderr, "str2float(): value out of range\n");
      exit(1);
    }
#endif

    if (*str == '\0' || *endptr != '\0')
      *ok = false;

    return value;
  }

  long 
  str2long(std::string *str, bool *ok) 
  { 
    return str2long(str->c_str(), ok); 
  }

  double 
  str2float(std::string *str, bool *ok) 
  { 
    return str2float(str->c_str(), ok); 
  }

}
