#ifndef CONF_HH
#define CONF_HH

#include <vector>
#include <string>
#include <map>

/** Configuration options, command line parameters and config files. 
 *
 * The command line options are parsed with the following features:
 * \li "-" is not an option \li The first "--" is skipped, and all
 * parameters after that are not options \li A single hyphen starts a
 * short option name.  Short options can be grouped even if they
 * require parameters.  \li Two hyphens start a long option name.
 *
 */
namespace conf {
  
  /** An option */
  struct Option {
    Option() : short_name(0), required(false), needs_argument(false), 
	       specified(false) { }
    unsigned char short_name; //!< The short name used for command line
    std::string long_name; //!< The long name used also in configuration files
    std::string value; //!< The value parsed from command line or file
    bool required; //!< Is the option required
    bool needs_argument; //!< Does the option need an argument
    bool specified; //!< Has the user specified the option
    std::string help; //!< The help string of the option
  };

  /** A structure for defining, storing and querying options. 
   * \bug Reading options from a file is not implemented yet.
   */
  struct Config {
    
    /** Create an empty config. */
    Config();

    /** Add a new option. 
     * \param short_name = the short name of the option used after a hyphen
     * \param long_name = the long name of the option used after two hyphens
     * \param type = a space seperated list of type specifiers: 
     *   \li "arg" = argument must follow the option
     *   \li "must" = the argument must be specified by the user
     * \param default_value = the default value of the option
     * \param help = the help string of the option
     */
    Config& operator()(unsigned char short_name,
		       std::string long_name,
		       std::string type = "",
		       std::string default_value = "",
		       std::string help = "");

    /** Parse command line arguments. */
    void parse(int argc, char *argv[]);

    /** Read a config file 
     * \param file = file to read from
     */
    void read(FILE *file);
    
    /** Check if all required options are specified. */
    void check_required() const;

    /** Returns the usage information */
    std::string help_string() const;

    /** Get a value of an option. */
    const Option& operator[](unsigned char short_name) const;

    /** Get a value of an option. */
    const Option& operator[](std::string long_name) const;

    /** A type for mapping short option names to option indices. */
    typedef std::map<unsigned char, int> ShortMap;

    /** A type for mapping long option names to option indices. */
    typedef std::map<std::string, int> LongMap;

    std::vector<Option> options; //!< Options added to the configuration
    ShortMap short_map; //!< Mapping short names to options
    LongMap long_map; //!< Mapping long names to options
    std::vector<std::string> arguments; //!< Rest of the non-option arguments
    int longest_name_length; //!< The length of the longest long name
  };

};

#endif /* CONF_HH */
