/******************************************************************************\
   Go81 Plays the game of go on PalmOS devices.
   Copyright (C) 2002-2004 Tapani Raiko

    this program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    this program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS for A PARTICULAR PURPOSE.  See the
    GNU General public License for more details.

    You should have received a copy of the GNU General public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
\******************************************************************************/

// for temporary use only:
static int stack[GO_BOARDTABLESIZE*GO_BOARDTABLESIZE];

#ifdef _CONSOLE
  static float worth2[GO_BOARDTABLESIZE*GO_BOARDTABLESIZE];
  //  float worth3[GO_BOARDTABLESIZE*GO_BOARDTABLESIZE];
  static float weight2[GO_BOARDTABLESIZE*GO_BOARDTABLESIZE];
  //  float weight3[GO_BOARDTABLESIZE*GO_BOARDTABLESIZE];
  static float w1orw2 = 0.0;
#endif
//static float randomness = 1/(float)2048;
static float randomness = 1/(float)4000;


// ------------------------------  AGENT ---------------------------------



Agent::Agent()
{
  lint i;

//  re_evaluate_groups = NULL;
  removed_groups = NULL;
//  unsettled_groups = NULL;
  for (i = 0; i < GO_BOARDTABLESIZE * GO_BOARDTABLESIZE; i++)
  {
    contents[i] = GO_BORDER;
  }
}

Agent::Agent(lint _size, lint _color, lint _onoff)
{
//  re_evaluate_groups = NULL;
  removed_groups = NULL;
//  unsettled_groups = NULL;
  InitAgent(_size, _color, _onoff);
}

void Agent::InitAgent(lint _size, lint _color, lint _onoff)
{
  lint i,q;

  CleanUpAgent();
  for (i = 0; i < GO_BOARDTABLESIZE * GO_BOARDTABLESIZE; i++)
  {
    contents[i] = GO_BORDER;
    movetype[i] = GO_MOVE_NORMAL;
  }
  size = _size;
  color = _color;
  onoff = _onoff;

  this->AgentInitBoard();
}

void Agent::CleanUpAgent()
{
  lint q;
  
//  IntSetDelete(re_evaluate_groups);  
  IntSetDelete(removed_groups);
//  IntSetDelete(unsettled_groups);

  for (q = 0; q <= ngroups; q++)
    group[q].Clean();

//  re_evaluate_groups = NULL;
  removed_groups = NULL;
//  unsettled_groups = NULL;
}


void 
Agent::Copy(const Agent &a)
{
  lint q;
  
  size = a.size;
  temp_ko = a.temp_ko;
  current_ko_point = a.current_ko_point;
  ngroups = a.ngroups;
  onoff = a.onoff;
  color = a.color;
  removed_groups = IntSetCopy(a.removed_groups);
  my_memcpy((char *)&contents,(char *)a.contents,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(char));
  my_memcpy((char *)&movetype,(char *)a.movetype,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(char));
  my_memcpy((char *)&hoodtype,(char *)a.hoodtype,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(char));
  my_memcpy((char *)&worth1,(char *)a.worth1,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(float));
#ifdef _CONSOLE
  //  my_memcpy((char *)&worth2,(char *)a.worth2,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(float));
  //  my_memcpy((char *)&weight2,(char *)a.weight2,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(float));
  //  my_memcpy((char *)&worth3,(char *)a.worth3,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(float));
  //  my_memcpy((char *)&weight3,(char *)a.weight3,GO_BOARDTABLESIZE*GO_BOARDTABLESIZE*sizeof(float));
#endif
  my_memcpy((char *)&captured,(char *)a.captured,(GO_WHITE+1)*sizeof(lint));
  for (q=0;q<=a.ngroups;q++) group[q] = a.group[q];
  for (q=a.ngroups+1;q<GO_MAX_N_GROUPS;q++) {
    group[q].stones = NULL;
    group[q].liberties = NULL;
    group[q].neighbours = NULL;
  }
}


lint Agent::GetCont(lint x, lint y)
{
  return group[contents[XY2POINT(x, y)]].color;
}

void Agent::SetCaptured(int bla, int whi)
{
  captured[GO_BLACK]=bla;
  captured[GO_WHITE]=whi;
}

void Agent::GetCaptured(int *bla, int *whi)
{
  *bla = captured[GO_BLACK];
  *whi = captured[GO_WHITE];
}

lint Agent::AgentInitBoard()
{
  lint x, y, q, w;

  for (x = 0; x < size; x++)
    for (y = 0; y < size; y++)
    {
      contents[XY2POINT(x, y)] = GO_EMPTY;
      hoodtype[XY2POINT(x, y)] = HOOD_OUTDATED;
      movetype[XY2POINT(x, y)] = GO_MOVE_NORMAL;
    }
  captured[GO_BLACK] = 0;
  captured[GO_WHITE] = 0;

  for (q=0;q<GO_MAX_N_GROUPS;q++) 
    group[q].Clean();

  ngroups = GO_BORDER; // max(go_empty, go_border)
  group[GO_EMPTY].color = GO_EMPTY;
  group[GO_EMPTY].size = 99999;
  group[GO_EMPTY].libe = 99999;
  group[GO_EMPTY].liberties = NULL;
  group[GO_EMPTY].stones = NULL;
  
  group[GO_BORDER].color = GO_BORDER;
  group[GO_BORDER].size = 99999;
  group[GO_BORDER].libe = 99999;
  group[GO_BORDER].liberties = NULL;
  group[GO_BORDER].stones = NULL;

//  re_evaluate_groups = NULL;
  removed_groups = NULL;
//  unsettled_groups = NULL;

  return 0;
}         


// returns -1 error, 0 normal, 1 capture, 2 suicide
lint Agent::SetCont(lint x, lint y, lint cont)
{
  lint s = XY2POINT(x, y), di, gro, iscapture = 0;
  lint x1,y1;

  if (x < 0 || y < 0 || x >= size || y >= size) return -1;
  if (cont == GO_EMPTY)
    contents[s] = 0;
  else
  {
    // new stonegroup
    if (removed_groups == NULL)
    {
      ngroups++;
//      if (ngroups>=GO_MAX_N_GROUPS)
//        Panic();
      gro = ngroups;
    }
    else
    {
      // take a recycled index for the group
      gro = removed_groups->i;
      IntSetRemove(&removed_groups, gro);
    }
    contents[s] = gro;
    group[gro].Clean();
    group[gro].x = x;
    group[gro].y = y;
    group[gro].color = cont;
    group[gro].libe = 0;
    group[gro].size = 1;
    group[gro].liberties = NULL;
    group[gro].neighbours = NULL;
    group[gro].stones = NULL;
    IntSetAdd(&group[gro].stones, s);
  
    lint nei[4] = {s, s, s, s};
    if (x > 0) nei[0] = nei[0] - 1;
    if (x < size - 1) nei[1] = nei[1] + 1;
    if (y > 0) nei[2] = nei[2] - GO_BOARDTABLESIZE;
    if (y < size - 1) nei[3] = nei[3] + GO_BOARDTABLESIZE;

    // check neighbours
    for (di = 0; di < 4; di++)
    {
      if (group[contents[nei[di]]].color == GO_EMPTY) 
      {
        group[contents[s]].AddLiberty(nei[di]);
      }
      else if (group[contents[nei[di]]].color == cont)
      {
        if (contents[nei[di]] == contents[s])
        {
          //own group neighbour twice, don't do anything
        }
        else
        {
          // connect groups "contents[s]" with "contents[nei[di]]"
          lint g2 = contents[s], g1 = contents[nei[di]];
          
          UpdateCont(g1, g2); //new, old
          
          // outdate liberties of an unsafe friendly group
          if (group[g1].libe < LIBERTIES_FOR_TACTICAL_SAFETY) {
            for (IntSet *cursor = group[g1].liberties; cursor != INTSET_END; cursor = cursor->next) {
              hoodtype[cursor->i] = HOOD_OUTDATED;
            }
          }
              
          group[g1].AddLiberties(group[g2].liberties);
          group[g1].RemoveLiberty(s);
          group[g1].size = group[g1].size + group[g2].size;
          if (group[g2].status == GS_BENSON_ALIVE)
            group[g1].status = GS_BENSON_ALIVE;
          IntSetUnion(&group[g1].stones, group[g2].stones);
          //remind g2:s neighbours of change
          for (IntSet *cursor = group[g2].neighbours; cursor != INTSET_END; cursor = cursor->next)
          {
            IntSetRemove(&group[cursor->i].neighbours, g2);
            IntSetAdd(&group[cursor->i].neighbours, g1);
          }
          IntSetUnion(&group[g1].neighbours, group[g2].neighbours);
          group[g2].Clean();

          IntSetAdd(&removed_groups, g2);
        }
      }
    }
    // check neighbours again considering enemy groups
    for (di = 0; di < 4; di++)
    {
      if (group[contents[nei[di]]].color == COMP_COL(cont))
      {
        // mark groups as neighbours
        IntSetAdd(&group[contents[s]].neighbours, contents[nei[di]]);
        IntSetAdd(&group[contents[nei[di]]].neighbours, contents[s]);

        // reduce enemy's liberties
        //group[contents[nei[di]]].libe--;
        group[contents[nei[di]]].RemoveLiberty(s);
        // capture?
        if (group[contents[nei[di]]].libe == 0)
        {
          CaptureGroup(contents[nei[di]]);
          iscapture = 1;
        } else
          // outdate liberties of an unsafe enemy group
          if (group[contents[nei[di]]].libe < LIBERTIES_FOR_TACTICAL_SAFETY) {
            for (IntSet *cursor = group[contents[nei[di]]].liberties; cursor != NULL; cursor = cursor->next) {
              hoodtype[cursor->i] = HOOD_OUTDATED;
            }
          }
      }
    }
    // suicide?
    if (group[contents[s]].libe == 0)
    {
      CaptureGroup(contents[s]);
//      FrmAlert(alertID_debug);
      iscapture = 2;
    }
  }
  // outdate neighourhood
  for (x1 = x-1; x1<=x+1; x1++)
    for (y1 = y-1; y1<=y+1; y1++) {
      hoodtype[XY2POINT(x1, y1)] = HOOD_OUTDATED;
    }
  // outdate liberties of the unsafe new group
  if (group[contents[s]].libe < LIBERTIES_FOR_TACTICAL_SAFETY) {
    for (IntSet *cursor = group[contents[s]].liberties; cursor != NULL; cursor = cursor->next) {
      hoodtype[cursor->i] = HOOD_OUTDATED;
    }
  }

  return iscapture;
}


lint Agent::MoveType(lint x, lint y, lint cont)
{
  lint ret, q, s, connects = false;
  lint nei[4], gro[4], lib[4];

  // is pass?
  if (x==GO_PASS) 
    return GO_MOVE_PASS;
    
  s = XY2POINT(x, y);

  // is point empty?
  ret = contents[s];
  if (group[ret].color != GO_EMPTY) 
    return GO_MOVE_UNEMPTY;

  // life and death init
  if (x > 0)        nei[0] = s - 1;    else nei[0] = s;
  if (x < size - 1) nei[1] = s + 1;    else nei[1] = s;
  if (y > 0)        nei[2] = s - GO_BOARDTABLESIZE; else nei[2] = s;
  if (y < size - 1) nei[3] = s + GO_BOARDTABLESIZE; else nei[3] = s;
  for (q = 0; q < 4; q++)
  {
    gro[q] = contents[nei[q]];
    lib[q] = group[gro[q]].libe;
  }
  // captures stone(s)?
  for (q = 0; q < 4; q++)
    if (group[gro[q]].color == COMP_COL(cont) && lib[q] == 1)
    {
      // ko
      if (group[gro[q]].size == 1 && s == current_ko_point)
        return GO_MOVE_KO;

      return GO_MOVE_CAPTURE;
    }
  // has a liberty so is not suicide
  for (q = 0; q < 4; q++)
    if (nei[q] != s && group[gro[q]].color == GO_EMPTY)
      return GO_MOVE_NORMAL;
  // connects to a group with enough liberties?
  for (q = 0; q < 4; q++)
    if (group[gro[q]].color == cont)
    { 
      if (lib[q] > 1)
        return GO_MOVE_NORMAL;
      connects = true;
    }
  // it is a suicide
  if (connects) 
    return GO_MOVE_SUICIDE;
  return GO_MOVE_SINGLESTONESUICIDE;
}

bool legal(lint movetype)
{
  return (movetype==GO_MOVE_NORMAL ||
          movetype==GO_MOVE_BENSON ||
          movetype==GO_MOVE_CAPTURE ||
          (movetype==GO_MOVE_SUICIDE && GO_SUICIDE_ALLOWED) ||
          (movetype==GO_MOVE_PASS && GO_PASS_ALLOWED));
}

bool Agent::IsLegal(lint x, lint y, lint cont)
{
  return legal(MoveType(x,y,cont));
}

// returns movetype
lint Agent::PlaceStone(lint x, lint y, lint cont)
{
  lint movetype;

  temp_ko = POINT_UNDECIDED;
  movetype = MoveType(x, y, cont);
  if (legal(movetype)) 
  {
    SetCont(x, y, cont);
    current_ko_point = temp_ko;
  }

  return movetype;
}


lint Agent::CaptureGroup(lint pris)
{
  lint nstack, di, q, s, x, y, found, nei[4];
  lint x1,y1;
  if (pris == 0) 
    return 0;

  nstack = 0;
  stack[0] = XY2POINT(group[pris].x, group[pris].y);
  contents[stack[0]] = 0;

  // update neighbours of the neighbours and outdate their liberties
  // must be done before liberties are increased.
  for (IntSet *cursor = group[pris].neighbours; cursor != INTSET_END; cursor = cursor->next) {
    IntSetRemove(&group[cursor->i].neighbours, pris);
    if (group[cursor->i].libe < LIBERTIES_FOR_TACTICAL_SAFETY) {
      for (IntSet *cursor2 = group[cursor->i].liberties; cursor2 != INTSET_END; cursor2 = cursor2->next) {
        hoodtype[cursor2->i] = HOOD_OUTDATED;
      }
    }
  }

  //if one stone captured, mark place as ko.
  if (group[pris].size == 1) 
  {
    if (temp_ko == POINT_UNDECIDED)
    {
      temp_ko = stack[0];
    }
    else 
    {
      temp_ko = POINT_NO_KO;
    }
  }
  captured[group[pris].color] += group[pris].size;
  
  while (nstack > -1)
  {
    s = stack[nstack];
    x = POINT2X(s);
    y = POINT2Y(s);
    // outdate neighbourhood
    for (x1 = x-1; x1<=x+1; x1++)
      for (y1 = y-1; y1<=y+1; y1++) {
        hoodtype[XY2POINT(x1, y1)] = HOOD_OUTDATED;
      }

    if (x > 0)        nei[0] = s - 1;    else nei[0] = s;
    if (x < size - 1) nei[1] = s + 1;    else nei[1] = s;
    if (y > 0)        nei[2] = s - GO_BOARDTABLESIZE; else nei[2] = s;
    if (y < size - 1) nei[3] = s + GO_BOARDTABLESIZE; else nei[3] = s;
    for (di = 0; di < 4; di++)
    {
      if (nei[di] == s) {}  //debug
      else 
      if (contents[nei[di]] == pris)
      {
        // remove captured neighbour and add to stack for search of its neigbours
        contents[nei[di]] = 0;
        stack[nstack] = nei[di];
        nstack++;
      } 
      else
      {
        if (group[contents[nei[di]]].color > GO_BORDER)
        {
          // neighbour gets a liberty
          group[contents[nei[di]]].AddLiberty(s);
//          IntSetAdd(&re_evaluate_groups, contents[nei[di]]);
        }
      }
    }
    nstack--;
  }

  group[pris].Clean();
  IntSetAdd(&removed_groups, pris);
  
  return 0;
}

lint Agent::UpdateCont(lint newg, lint oldg)
{
  lint nstack, di, q, s, x, y, found, nei[4];
  
  if (newg == oldg) return 0;
  nstack = 0;
  stack[0] = XY2POINT(group[oldg].x, group[oldg].y);
  contents[stack[0]] = newg;

  while (nstack > -1)
  {
    s = stack[nstack];
    x = POINT2X(s);
    y = POINT2Y(s);
    if (x > 0)        nei[0] = s - 1;    else nei[0] = s;
    if (x < size - 1) nei[1] = s + 1;    else nei[1] = s;
    if (y > 0)        nei[2] = s - GO_BOARDTABLESIZE;   else nei[2] = s;
    if (y < size - 1) nei[3] = s + GO_BOARDTABLESIZE;   else nei[3] = s;
    for (di = 0; di < 4; di++)
    {
      if (contents[nei[di]] == oldg)
      {
        // update neighbour and add to stack
        stack[nstack] = nei[di];
        contents[nei[di]] = newg;
        nstack++;
      } 
    }
    nstack--;
  }
  return 0;
}

int HoodType(Agent *agent, lint x, lint y, lint own, lint *points)
{
  lint di, q, neig[4], diag[4], other = COMP_COL(own);
  lint hoodtype = HOOD_NORMAL, distb1, distb2, temp;
  char around[8], history;
  *points = 0;
  if (agent->GetCont(x,y) != GO_EMPTY) 
    return HOOD_ILLEGAL;
  for (q=0;q<4;q++) {
    neig[q]=0;
    diag[q]=0;
  }
  around[0] = agent->GetCont(x-1,y-1);
  around[1] = agent->GetCont(x,y-1);
  around[2] = agent->GetCont(x+1,y-1);
  around[3] = agent->GetCont(x+1,y);
  around[4] = agent->GetCont(x+1,y+1);
  around[5] = agent->GetCont(x,y+1);
  around[6] = agent->GetCont(x-1,y+1);
  around[7] = agent->GetCont(x-1,y);
  for (di=0;di<8;di+=2)
  {
    diag[around[di]]++;
    neig[around[di+1]]++;
  }
  distb2 = MIN(x+1,datab.boardsize-x);
  temp = MIN(y+1,datab.boardsize-y);
  distb1 = MIN(distb2,temp);
  distb2 = MAX(distb2,temp);
  // selecting hoodtype
  if (neig[own] + neig[GO_BORDER] == 4 && 
      diag[own] + neig[GO_BORDER] >= 3)
    hoodtype = HOOD_OWNEYE;
  else if (neig[other] + neig[GO_BORDER] == 4 && 
      diag[other] + neig[GO_BORDER] >= 3)
    hoodtype = HOOD_OTHEREYE;
  else if (neig[GO_BLACK] + diag[GO_BLACK] + neig[GO_WHITE] + diag[GO_WHITE] == 0)
  {
    if (distb1==1) hoodtype = HOOD_EMPTY1;
    else if (distb1==3 || distb1==4) {
      if (distb2==3 || distb2==4) hoodtype = HOOD_FUSEKI;
      else hoodtype = HOOD_EMPTY3OR4;
    }
    else hoodtype = HOOD_EMPTY2OR5PLUS;
  }
  else if (neig[GO_EMPTY] == 4 && diag[GO_EMPTY] == 3 &&
           distb1 == 3 && 
           distb2 == 3)
  {
    hoodtype = HOOD_33AFTER44;
  }
  else if (neig[own] + neig[GO_BORDER] >= 3 &&
      2*neig[other] + diag[other] + (diag[GO_BORDER]>0) <= 1)
    hoodtype = HOOD_OWNTIGER;
  else if (neig[other] + neig[GO_BORDER] >= 3 &&
      2*neig[own] + diag[own] + (diag[GO_BORDER]>0) <= 1)
    hoodtype = HOOD_OTHERTIGER;  
   
  // now using around. The state for the machine before coming to 0:
  history = HISTORY_OTHER;
  if (around[7]==GO_BLACK) history = HISTORY_BLACK;
  else if (around[7]==GO_WHITE) history = HISTORY_WHITE;
  else if (around[7]==GO_EMPTY)
  {
    if (around[6]==GO_BLACK) history = HISTORY_EMPTY_BLACK;
    else if (around[6]==GO_WHITE) history = HISTORY_EMPTY_WHITE;
  }
  // going around
  for (di=0;di<8;di++)
  {
    if (around[di]==GO_EMPTY)
    {
      if (history == HISTORY_BLACK) history = HISTORY_EMPTY_BLACK;
      else if (history == HISTORY_WHITE) history = HISTORY_EMPTY_WHITE;
      else history = HISTORY_OTHER;
    }
    else if (around[di]==GO_BLACK)
    {
      // blackblack -2, whiteblack +4, whiteemptyblack +3, blackcornerblack -2
      if (history == HISTORY_BLACK) *points -= 20;
      else if (history == HISTORY_WHITE) *points += 40;
      else if (history == HISTORY_EMPTY_WHITE) *points += 30;
      else if (history == HISTORY_EMPTY_BLACK) *points -= 5 + 8*(di & 1);
      history = HISTORY_BLACK;
    }
    else if (around[di]==GO_WHITE)
    {
      if (history == HISTORY_WHITE) *points -= 20;
      else if (history == HISTORY_BLACK) *points += 40;
      else if (history == HISTORY_EMPTY_BLACK) *points += 30;
      else if (history == HISTORY_EMPTY_WHITE) *points -= 5 + 8*(di & 1);
      history = HISTORY_WHITE;
    }
    else history = HISTORY_OTHER;
  }
  // special case with two same colours in the opposite corners and others empty
  if (neig[GO_EMPTY]==4 && *points==0 && (diag[GO_BLACK]==2 || diag[GO_WHITE]==2))
    *points = 30;
  // special case of toothpaste
  else if (*points==80 && neig[GO_EMPTY]+neig[GO_BORDER]==3 && diag[GO_EMPTY]+neig[GO_BORDER]==2)
    *points = 120;
  // special case of cutting at the second row
  else if (*points==80 && distb1==2 && neig[GO_EMPTY]==2 && diag[GO_EMPTY]==3)
    hoodtype = HOOD_CUT_2ND_ROW;
  // special case of pushing to stop the tiger form (or forming it if push possible)
  else if (*points>30 && neig[GO_EMPTY]==3 && diag[GO_EMPTY]==2 && 
    ((neig[GO_BLACK]==0 && diag[GO_WHITE]==0) || (neig[GO_WHITE]==0 && diag[GO_BLACK]==0)))
    *points += 30;
  // lots of same neighbours is a bit bad
  if (neig[own]>1) *points-=16*(neig[own]-1);
  if (neig[other]>1) *points-=16*(neig[other]-1);
  // border and corner points
  if (distb1==1) {
    *points-=8;
    if (distb2==1) *points-=8;
  } else if (distb2==2) {
    *points-=8;
  } else if (distb1==3) {
    *points+=10;
  }
   
  return hoodtype;
}


void Agent::UpdateWorth1(lint turn, lint x, lint y)
{
  lint s = XY2POINT(x,y);
  lint libe_own = 0, libe_other = 0, size_own = 0, size_other = 0, points;
  float worth_libe;

  if (movetype[s]!=GO_MOVE_BENSON)
    movetype[s] = MoveType(x,y,turn);
  // illegal? (or in Benson's region)
  if (movetype[s] >= GO_MOVE_BENSON) {
    hoodtype[s] = HOOD_ILLEGAL;
    worth1[s] = -9999999.9; // worse than passing
    // if illegal because of a ko, outdate for the future
    if (movetype[s] == GO_MOVE_KO)
      hoodtype[s] = HOOD_OUTDATED;
    return;
  }
  hoodtype[s] = HoodType(this, x, y, turn, &points);
  // forbidden neigbourhoods
  if (hoodtype[s] == HOOD_OWNEYE || 
      hoodtype[s] == HOOD_OWNTIGER) {
    worth1[s] = -9999999.9;
    return;
  }
  worth1[s] = 0.025 * points;
  if (hoodtype[s] == HOOD_EMPTY3OR4) 
    worth1[s] += 1.0;
  else if (hoodtype[s] == HOOD_EMPTY2OR5PLUS) 
    worth1[s] += 0.5;
  else if (hoodtype[s] == HOOD_33AFTER44)
    worth1[s] += 0.7;
  else if (hoodtype[s] == HOOD_FUSEKI)
    worth1[s] += 2.0;
  // how will liberties change
  WhatIfHere(x, y, turn, &worth_libe, &libe_own, &size_own, &libe_other);
  worth1[s] += worth_libe * WORTHLIBE_TO_WORTH;
  // allow playing in opponents tiger only if gives (almost) atari or captures
  // restrict also big autoataris
  if ((hoodtype[s] == HOOD_OTHERTIGER && libe_other>2) ||
      (libe_own < 2 && size_own >3)) {  //better: nakade shape instead of size_own
    worth1[s] = -9999999.9;
    return;
  }
  // special case of cut on the second row
  if (hoodtype[s] == HOOD_CUT_2ND_ROW && libe_other<9999) {
    if (libe_other == 2)
      worth1[s] -= 0.5;
    else if (libe_other > 2)
      worth1[s] -= 1.0;
  }
  // autoatari (or 2 liberties on the border)
  if (libe_own <= 1) {
    worth1[s] -= 0.5;
    if (points>0) worth1[s] -= 0.02 * points;
  } else if (libe_own == 2 && (x==0 || x==datab.boardsize-1 || y==0 || y==datab.boardsize-1)) 
  {
    worth1[s] -= 0.2;
    if (points>0) worth1[s] -= 0.018 * points;
  }   
}

// Vital points in the border should have at least 4 stones of the same colour in their vicinity
void Agent::BorderVitalCheck(lint x,lint y)
{
  int nei[4],q;
  static int dx[16]={-1, 0,+1,-2,-1,+1,+2,-2,+2,-2,-1,+1,+2,-1, 0,+1},
             dy[16]={-2,-2,-2,-1,-1,-1,-1, 0, 0,+1,+1,+1,+1,+2,+2,+2};
  
  nei[GO_BLACK]=0;
  nei[GO_WHITE]=0;
  for (q=0;q<16;q++) 
  {
    if (x+dx[q]>=0 && x+dx[q]<size && y+dy[q]>=0 && y+dy[q]<size)
      nei[GetCont(x+dx[q],y+dy[q])]++;
  }
  if (nei[GO_BLACK]<4 && nei[GO_WHITE]<4) 
    vital[XY2POINT(x,y)]=0;
}

// counts and marks the sizes of the continuous neutral territories
// 0 border case --> half of the max of neighbours.
// -1 unbrowsed neutral
// -2 currently browsed
// -3 fully black or white (or out-of-board) territory.
void Agent::CountNeutralSizes()
{
  lint nstack, cursor, di, q, s, x, y, nei[4], x1, y1, nsize;

  for (y1=-1; y1<=size; y1++)
    for (x1=-1; x1<=size; x1++)
      neutralsize[XY2POINT(x1,y1)] = -3+2*(owners[XY2POINT(x1,y1)]==GO_EMPTY);
  for (y1=0; y1<size; y1++) {
    for (x1=0; x1<size; x1++) {
      if (neutralsize[XY2POINT(x1,y1)] == -1) {
        nstack = 1;
        cursor = 0;
        stack[0] = XY2POINT(x1, y1);
        neutralsize[XY2POINT(x1,y1)] = -2;
        if (group[contents[XY2POINT(x1,y1)]].color==GO_EMPTY)
          nsize = 1;
        else
          nsize = 2;

        while (cursor < nstack) {
          s = stack[cursor];
          x = POINT2X(s);
          y = POINT2Y(s);
          if (x > 0)        nei[0] = s - 1;                   else nei[0] = s;
          if (x < size - 1) nei[1] = s + 1;                   else nei[1] = s;
          if (y > 0)        nei[2] = s - GO_BOARDTABLESIZE;   else nei[2] = s;
          if (y < size - 1) nei[3] = s + GO_BOARDTABLESIZE;   else nei[3] = s;
          for (di = 0; di < 4; di++) {
            if (neutralsize[nei[di]] == -1) {
              if (group[contents[nei[di]]].color==GO_EMPTY)
                nsize++;
              else
                nsize += 2;
              stack[nstack] = nei[di];
              neutralsize[nei[di]] = -2;
              nstack++;
            }
          }
          cursor++;
        }
        // mark size to the whole neutral territory
        for (cursor=0;cursor < nstack; cursor++) {
          neutralsize[stack[cursor]] = nsize;
        }
      }
    }
  }
  //smoothing:
  for (y1=0; y1<size; y1++)
    for (x1=0; x1<size; x1++)
      if (neutralsize[XY2POINT(x1,y1)]==-3 &&
          (neutralsize[XY2POINT(x1-1,y1)]>0 || neutralsize[XY2POINT(x1+1,y1)]>0 ||
           neutralsize[XY2POINT(x1,y1-1)]>0 || neutralsize[XY2POINT(x1,y1+1)]>0))
        neutralsize[XY2POINT(x1,y1)]=0;
  for (y1=0; y1<size; y1++)
    for (x1=0; x1<size; x1++)
      if (neutralsize[XY2POINT(x1,y1)]==0
#ifdef _VISU
         && GetCont(x1,y1)==GO_EMPTY
#endif
         )
        neutralsize[XY2POINT(x1,y1)] = MAX(
           MAX(neutralsize[XY2POINT(x1-1,y1)],neutralsize[XY2POINT(x1+1,y1)]),
           MAX(neutralsize[XY2POINT(x1,y1-1)],neutralsize[XY2POINT(x1,y1+1)])
           ) >> 1;
}

#ifdef _VISU
static void DrawUpdateDebugMark(lint x, lint y, lint bitmapID)
{
  if (bitmapID==bitmapID_mark_liberties)
    DrawBitmap(bitmapID, XOFFSET_MARK+BITMAPSIZE*x-3,YOFFSET_MARK+BITMAPSIZE*y-3);
  else if (bitmapID==bitmapID_mark_vital)
    DrawBitmap(bitmapID, XOFFSET_MARK+BITMAPSIZE*x-3,YOFFSET_MARK+BITMAPSIZE*y+3);
  else if (bitmapID==bitmapID_mark_shape)
    DrawBitmap(bitmapID, XOFFSET_MARK+BITMAPSIZE*x+3,YOFFSET_MARK+BITMAPSIZE*y-3);
  else if (bitmapID==bitmapID_mark_neutral)
    DrawBitmap(bitmapID, XOFFSET_MARK+BITMAPSIZE*x+3,YOFFSET_MARK+BITMAPSIZE*y+3);
}
#endif

lint Agent::SelectPoint(lint turn, char _slow, lint latest)
{
  float bestworth = -9999.9, worth, weight;
  lint bestpoint = POINT_PASS, s,x,y,q,w;
  lint worthfuture;
  char slow;
  lint bs=0,ws=0;
  lint dummy, libe_own;
  float worth_libe;

  slow = _slow && (onoff & ONOFF_SLOW) && (datab.nmoves > NMOVES_SLOW);
 
#ifdef _CONSOLE
  #define NSCENARIOS 50
  #define NPASSSCENARIOS 20
  float worthpass = 0.0;
  // level 3:
  if (slow && (onoff & ONOFF_FUTUREFORALL)) {
    for (y=0;y<size;y++) {
      for (x=0;x<size;x++) {
        s = XY2POINT(x,y);
        weight2[s] = 1.0; 
        worth2[s] = -0.05*NSCENARIOS;
        //        weight3[s] = 1.0;
        //        worth3[s] = -100.0;
      }
    }
    if (datab.nmoves > NMOVES_NOPASSING) {
      for (q=0;q<NPASSSCENARIOS;q++) {
        randomness = 1/(float)(400.0+q);
        worthpass += ScoreMoveFuture(turn, POINT_PASS);
      }
    }
    for (q=0;q<NSCENARIOS;q++) {
      randomness = 1/(float)(400.0+q);
      worthfuture = ScoreMoveFuture(turn, POINT_UNDECIDED);
      weight = 1.0;
      for (w=0;w<NFUTMOVES;w+=2) {
        if (futmoves[w]==POINT_PASS) 
          break;
        weight2[futmoves[w]] += weight;
        worth2[futmoves[w]] += weight * worthfuture;
        weight *= 0.85;
      }
    }
    for (y=0;y<size;y++) {
      for (x=0;x<size;x++) {
        s = XY2POINT(x,y);
        if (hoodtype[s]==HOOD_OUTDATED)
          UpdateWorth1(turn,x,y);
        if (worth1[s] > -9999.9) {
	  //          printf("%7.2f", 0.1*worth2[s]/weight2[s]);
          if (weight2[s] > NSCENARIOS/10.0 && worth2[s] / weight2[s] + 0.01*weight2[s] > bestworth) {
            bestworth = worth2[s] / weight2[s] + 0.01*weight2[s];
            bestpoint = s;
	    //            printf("Best point (%d,%d)",POINT2X(s),POINT2Y(s));
          }
        } else {
	  //	  printf("    ...");
        }
      }
      //      printf("\n");
    }
    //    printf("\n");
    /*    for (y=0;y<size;y++) {
      for (x=0;x<size;x++) {
        s = XY2POINT(x,y);
        if (worth1[s] > -9999.9) {
	  printf("%7.2f", worth2[s]/(float)weight2[s]-worthpass/(float)NPASSSCENARIOS);
        } else {
	  printf("    ...");
        }
      }
      printf("\n");
    }
    printf("\n");
    printf("Passing was estimated %7.2f\n", worthpass/(float)NPASSSCENARIOS); 
    */
    if (datab.nmoves > NMOVES_NOPASSING && 
        bestpoint != POINT_PASS &&
        worth2[bestpoint]/(float)weight2[bestpoint]-worthpass/(float)NPASSSCENARIOS<0.20)
      return POINT_PASS;
    else
      return bestpoint;
  } 
  // level 3 with future for each point as next:
  /*  if (slow && (onoff & ONOFF_FUTUREFORALL)) {
    // score pass as an initial case
    bestworth = ScoreMoveFuture(turn, bestpoint);
    for (y=0;y<size;y++) {
      for (x=0;x<size;x++) {
        s = XY2POINT(x,y);
        if (hoodtype[s]==HOOD_OUTDATED)
          UpdateWorth1(turn,x,y);
        if (worth1[s] > -9999.9) {
          worthfuture = ScoreMoveFuture(turn, s);
//          printf("%4d", worthfuture / 10);
          if (worthfuture>bestworth) {
            bestworth = worthfuture;
            bestpoint = s;
          }
        }
      }
    }
    return bestpoint;
    } */
#endif
 
  // Level 2:
  if (slow) {
    ScoreGame(COUNTING_IGNORE,&bs,&ws);
    AllowPlayingInSurviversLastLiberty();
//FrmAlert(alertID_debug);
    CountNeutralSizes();
//FrmAlert(alertID_debug);
    if (datab.nmoves > NMOVES_VITAL) {
      // stone in atari might be a vital point -> mark liberty as vital.
      for (y=0;y<size;y++) {
        for (x=0;x<size;x++) {
          if (vital[XY2POINT(x,y)]>0 && group[contents[XY2POINT(x,y)]].libe==1)
            vital[group[contents[XY2POINT(x,y)]].liberties->i] += vital[XY2POINT(x,y)];
        }
      }
      // vital points in the border require enough stones around
      for (y = 0; y < datab.boardsize; y++) {
        if (vital[XY2POINT(y,0)]) BorderVitalCheck(y,0);
        if (vital[XY2POINT(0,y)]) BorderVitalCheck(0,y);
        if (vital[XY2POINT(y,size-1)]) BorderVitalCheck(y,size-1);
        if (vital[XY2POINT(size-1,y)]) BorderVitalCheck(size-1,y);
      }
    } 

  }
#ifdef _VISU
  if (_slow) {
    DrawBoard();
    if (slow) {
      for (y=0;y<size;y++) {
        for (x=0;x<size;x++) {
          if (GetCont(x,y)!=GO_EMPTY && GetCont(x,y)!=owners[XY2POINT(x,y)])
            DrawUpdateMark(x, y, owners[XY2POINT(x,y)]);
        }
      }
    }
  }
#endif
 
  // Levels 1 and 2:
  // go through all the points on the board
  for (y=0;y<size;y++) {
    for (x=0;x<size;x++) {
      s = XY2POINT(x,y);
      if (hoodtype[s]==HOOD_OUTDATED)
        UpdateWorth1(turn,x,y);
      worth = worth1[s];
#ifdef _VISU
      if (slow) {
        if (neutralsize[s] > 0)
          DrawUpdateDebugMark(x, y, bitmapID_mark_neutral);
        if (vital[s] > 2 && GetCont(x,y)==GO_EMPTY)
          DrawUpdateDebugMark(x, y, bitmapID_mark_vital);
      }
#endif
      if (worth < -999)
        continue;
#ifdef _CONSOLE
      if (w1orw2>0.0)
        worth = worth * (1 - w1orw2) 
          + 0.1 * w1orw2 * worth2[s] / weight2[s];
#endif
#ifdef _VISU
      if (_slow) {
        WhatIfHere(x, y, turn, &worth_libe, &dummy, &dummy, &dummy);
        if (worth-WORTHLIBE_TO_WORTH*worth_libe > 0.4)
          DrawUpdateDebugMark(x, y, bitmapID_mark_shape);
        if (WORTHLIBE_TO_WORTH*worth_libe > 0.4)
          DrawUpdateDebugMark(x, y, bitmapID_mark_liberties);
      }
#endif
      if (slow) {
         // inside b/w territory and not vital and not very good move according to level 1?
        if (neutralsize[s] == -3 && vital[s] == 0 && worth < 2.5) {
          continue;
        }
        worth += neutralsize[s] * 0.08;
        // in the meeting point of black, white, and neutral territory.
        worth += 0.5 *((owners[XY2POINT(x-1,y-1)]==GO_BLACK || owners[XY2POINT(x+1,y-1)]==GO_BLACK || 
                        owners[XY2POINT(x-1,y+1)]==GO_BLACK || owners[XY2POINT(x+1,y+1)]==GO_BLACK || 
                        owners[XY2POINT(x,y-1)]==GO_BLACK || owners[XY2POINT(x+1,y)]==GO_BLACK || 
                        owners[XY2POINT(x,y+1)]==GO_BLACK || owners[XY2POINT(x-1,y)]==GO_BLACK) && 
                       (owners[XY2POINT(x-1,y-1)]==GO_WHITE || owners[XY2POINT(x+1,y-1)]==GO_WHITE || 
                        owners[XY2POINT(x-1,y+1)]==GO_WHITE || owners[XY2POINT(x+1,y+1)]==GO_WHITE ||
                        owners[XY2POINT(x,y-1)]==GO_WHITE || owners[XY2POINT(x+1,y)]==GO_WHITE || 
                        owners[XY2POINT(x,y+1)]==GO_WHITE || owners[XY2POINT(x-1,y)]==GO_WHITE));
// next 'if' is not necessary anymore because of bordervitalcheck
//        if (hoodtype[s] != HOOD_EMPTY1 && hoodtype[s] != HOOD_EMPTY2OR5PLUS)
        worth += vital[s] * 0.3;
      }
      worth += (rand() & 2047) * (float)randomness;      
      if (worth>bestworth) {
        bestworth = worth;
        bestpoint = s;
      }
    }
  }
  // do not pass too early on
  if (slow && bestpoint == POINT_PASS) {
    // if the opponent passed or defended, agent can pass or resign.
    if (latest == POINT_PASS || owners[latest] == COMP_COL(turn)) {
      if (2*(bs-ws)*(1-2*(turn==GO_BLACK))>size*size)
        return POINT_RESIGN;
      else
        return POINT_PASS;
    }
    for (y=0;y<size;y++) {
      for (x=0;x<size;x++) {
        s = XY2POINT(x,y);
        // allow playing inside own territory
        if (owners[s] == turn) {
          worth = worth1[s] + (rand() & 2047)* (float)randomness;
          if (worth>bestworth) {
            bestworth = worth;
            bestpoint = s;
          }
        }
      }
    }
  }
  // postprocessing: if removing ones own second liberty and filling the other liberty would 
  //                 not be autoatari, choose that instead.
  // note: this might not be a good idea if there is a ko fight going on.
  if (bestpoint != POINT_PASS) {
    s = bestpoint;
    x=POINT2X(s);
    y=POINT2Y(s);
    WhatIfHere(x, y, turn, &worth_libe, &libe_own, &dummy, &dummy);
    if (libe_own==1)
    {
      lint nei[4];
      nei[0]=s-1;
      nei[1]=s+1;
      nei[2]=s-GO_BOARDTABLESIZE;
      nei[3]=s-GO_BOARDTABLESIZE;
      for (int di=0;di<4;di++) {
        if (group[contents[nei[di]]].color == turn) {
          for (IntSet *cursor = group[contents[nei[di]]].liberties; cursor!=INTSET_END; cursor=cursor->next) {
            if (cursor->i!=s) {
              WhatIfHere(POINT2X(cursor->i), POINT2Y(cursor->i), turn, &worth_libe, &libe_own, &dummy, &dummy);   
              if (libe_own>1) {
                bestpoint = cursor->i;
//                FrmAlert(alertID_debug);
              }
            }
          }
        }
      }
    }
  }

  return bestpoint;
}

/*
// doesn't take into account interdepending eyes:
// returns size of eye (1..2) or zero, if not an eye.
lint Agent::IsSimpleEye(lint x, lint y, lint cont)
{
  lint s, di, nborders, nstack, ret, count, gro = GO_EMPTY, temp;
  lint stack[10], border[10], borderdi[10], q, w, bbcount, eyesize = 0;
  IntSet *eye = NULL;

  nstack = 0;
  stack[0] = XY2POINT(x, y);
  count = 1;
  nborders = 0;
  if (group[contents[stack[0]]].color == cont || group[contents[stack[0]]].color == GO_BORDER) 
    return 0;
  eyesize = eyesize + IntSetAdd(&eye, stack[0]);

  while (nstack > -1)
  {
    s = stack[nstack];
//    x = POINT2X(s);
//    y = POINT2Y(s);
    for (di = 0; di < 4; di++)
    {
      temp = false;
      if (group[contents[s + Dir[di]]].color == cont)
      {
        // own
        if (group[contents[s + Dir[di]]].size == 1)
        {
          // put size 1 groups in eye
          temp = true;
        }
        else if (gro == GO_EMPTY)
        {
          // group of edge is set
          gro = contents[s + Dir[di]];
        }
        else
        {
          // wrong group as edge of eye?
          if (gro != contents[s + Dir[di]])
            return false;
        }
      }
      if (temp 
        || group[contents[s + Dir[di]]].color == GO_EMPTY
        || group[contents[s + Dir[di]]].color == COMP_COL(cont))
      {
        ret = IntSetAdd(&eye, s + Dir[di]);
        eyesize = eyesize + ret;
        if (ret > 0)
        {
          if (count >= 2)
          {
            // 6 (2) is max size of eye
            return false;
          }
          stack[nstack] = s + Dir[di];
          nstack++;
          count++;
        }
      } 
      else if (group[contents[s + Dir[di]]].color == GO_BORDER)
      {
        // save border-info for later use
        border[nborders] = s;
        borderdi[nborders] = di;
        nborders++;
      }
    }
    nstack--;
  }
  // can another eye fit inside this?
  for (q = 0; q < nborders; q++)
  {
    bbcount = 0;
    for (w = 0; w < nborders; w++)
    {
      for (di = 0; di < 4; di++)
      {
        if (border[q] + Dir[di] == border[w]
          && contents[border[q] - Dir[borderdi[q]]] != gro
          && contents[border[w] - Dir[borderdi[w]]] != gro)
        {
          bbcount++;
        }
      }
    }
    if (bbcount >= 2) 
      return 0;
  }

  return eyesize;
}*/




// The player who is in turn _next_ evaluates things. Bigger is better.
static float LibertiesToCost( bool own, lint liber, lint size)
{
  float cost;

  if (liber >= LIBERTIES_FOR_TACTICAL_SAFETY) 
    return 0.0;
 
  size++; // size increased by one.   
  if (own) {
    cost=-0.7 * size;
    if (liber==0) return -1.4 * size;
  }
  else cost=1.0 * size;
  while (liber>0) 
  {
    cost *= 0.5;
    liber--;
  }

  return cost;
}


// This function is from AtariGo
// The player who is in turn evaluates things after one move. Bigger is better.
// Returns only the change of value because of this move.
// point, turn, value
// No checking for legalness
lint Agent::WhatIfHere(lint x, lint y, lint turn, float *worth, lint *liber, lint *nstones,
                       lint *_libe_other)
{
  lint di, aboutliber = 0, willbealive = false;
  lint s=XY2POINT(x,y);
  IntSet *liberties = NULL;
  IntSet *ownneigh = NULL, *enemyneigh = NULL, *cursor, *cur2;
  lint libe_other = 9999; // used elsewhere as well!
  
  lint nei[4] = {s-1, s+1, s-GO_BOARDTABLESIZE, s+ GO_BOARDTABLESIZE};

  // check neighbours
  for (di = 0; di < 4; di++)
  {
    if (group[contents[nei[di]]].color == GO_EMPTY) 
    {
      // add regular liberty
//      IntSetAdd(&liberties,nei[di]); //
      aboutliber++;
    }
    else if (group[contents[nei[di]]].color == turn)
    {
      // remember own neighbour and its liberties
      IntSetAdd(&ownneigh,contents[nei[di]]);
//      IntSetUnion(&liberties,group[contents[nei[di]]].liberties); //
    }
    else if (group[contents[nei[di]]].color == COMP_COL(turn))
    {
      // remember other neighbour
      IntSetAdd(&enemyneigh,contents[nei[di]]);
    }
  }

  // remove the filled point from the union of liberties
//  IntSetRemove(&liberties,s);
  // turn changes! (use false for own and negative values)
  *worth = 0.0;
  // enemy neighbours
  for (cursor=enemyneigh; cursor!=INTSET_END; cursor=cursor->next)
  {
    // remove old values and add new values with one liberty less (enemy neighbours)
    *worth += LibertiesToCost(true,group[cursor->i].libe, group[cursor->i].size);
    *worth -= LibertiesToCost(true,group[cursor->i].libe-1, group[cursor->i].size);
    // remember the smallest number of enemy liberties
    if (group[cursor->i].libe-1 < libe_other) libe_other = group[cursor->i].libe-1;
    // is it a capture? add liberties optimistically
    if (group[cursor->i].libe == 1) 
    {
//      IntSetUnion(&liberties, group[cursor->i].stones); //
      aboutliber +=group[cursor->i].size;
      // capturing helps its neighbours
      for (cur2 = group[cursor->i].neighbours; cur2 != INTSET_END; cur2 = cur2->next) {
        if (!IntSetContains(ownneigh, cur2->i) && group[cur2->i].status != GS_BENSON_ALIVE) {
          *worth += LibertiesToCost(false, group[cur2->i].libe, group[cur2->i].size);
          *worth -= LibertiesToCost(false, group[cur2->i].libe + group[cursor->i].size, group[cur2->i].size);
        }
      }
    }
  }
  *nstones = 1;
  // own neighbours
  for (cursor=ownneigh; cursor!=NULL; cursor=cursor->next)
  {
    // one of own neighbours is unconditionally alive
    if (group[cursor->i].status == GS_BENSON_ALIVE)
      willbealive = true;
    else {
      // remove old values of own neighbours
      *worth += LibertiesToCost(false,group[cursor->i].libe, group[cursor->i].size);
      // count number of stones in the future string
      *nstones += group[cursor->i].size;
      // add (optimistically) all neighbour's liberties, except the one which is filled
      aboutliber += group[cursor->i].libe - 1;
    }
  }
  // add only one value. (the connected group)
//  *liber = IntSetCount(liberties);
  *liber = aboutliber;
  if (!willbealive) {
    if (*liber < 5) {
      *liber = LibertiesIfHereMoreAccurate(s, turn);
    }
    *worth-=LibertiesToCost(false,*liber, *nstones);
  }
  *_libe_other = libe_other;
  IntSetDelete(ownneigh);
  IntSetDelete(enemyneigh);
//  IntSetDelete(liberties);  

  return 0;
}


// Same as above but more accurate
lint Agent::LibertiesIfHereMoreAccurate(lint s, lint turn)
{
  lint di, liber;
  IntSet *liberties = NULL;
  lint nei[4] = {s-1, s+1, s-GO_BOARDTABLESIZE, s+ GO_BOARDTABLESIZE};

  // check neighbours
  for (di = 0; di < 4; di++)
  {
    if (group[contents[nei[di]]].color == GO_EMPTY) 
    {
      // add a regular liberty
      IntSetAdd(&liberties,nei[di]);
    }
    else if (group[contents[nei[di]]].color == turn)
    {
      // own neighbour's liberties
      IntSetUnion(&liberties,group[contents[nei[di]]].liberties);
    }
    else if (group[contents[nei[di]]].color == COMP_COL(turn))
    {
      // is it a capture? add liberties optimistically
      if (group[contents[nei[di]]].libe == 1) 
        IntSetUnion(&liberties, group[contents[nei[di]]].stones);
    }
  }

  // remove the filled point from the union of liberties
  IntSetRemove(&liberties,s);
  liber = IntSetCount(liberties);
  IntSetDelete(liberties);  

  return liber;
}




// O O O X X   The empty point here looks like definite territory for X.
// O X X . X   Still, one would like to allow either player to play there.
// O O O X X   
// The following marks the last liberty of a surviving group as neutral.
// But if it has a dying neighbour, it should not be done.
void Agent::AllowPlayingInSurviversLastLiberty()
{
  lint gro, s, should=true;
  IntSet *cur;
  
  for (gro = 0; gro<=ngroups; gro++) {
    if (group[gro].libe==1 && group[gro].liberties != NULL) {
      s = group[gro].liberties->i;
      if (owners[s] == group[gro].color) {
        should = true;
        for (cur = group[gro].neighbours; cur != INTSET_END; cur = cur->next)
          if (owners[XY2POINT(group[cur->i].x,group[cur->i].y)] == group[gro].color)
            should = false;
        if (should)
          owners[s] = GO_EMPTY;
      }
    }
  }
}


/*
Hood2Index(Agent *agent, int x, int y)
{
  int di, num = 0, s = XY2POINT(x,y);
  
  for (di=0;di<8;di++)
  {
    num = num * 4 + agent->GetCont(x+Dir_x[di],y+Dir_y[di]);
  }    
     
}
*/

// -1 unbrowsed
// -2 current army
lint CountArmyEyes()
{
  lint nstack, cursor, di, q, s, x, y, nei[4], x1, y1;
  lint nhalfeyes, owner;//, bensonalive;

  for (y1=0; y1<datab.boardsize; y1++)
    for (x1=0; x1<datab.boardsize; x1++)
      armyeyes[XY2POINT(x1,y1)] = -1;
  for (y1=0; y1<datab.boardsize; y1++) {
    for (x1=0; x1<datab.boardsize; x1++) {
      if (armyeyes[XY2POINT(x1,y1)] == -1) {
        nstack = 1;
        cursor = 0;
        stack[0] = XY2POINT(x1, y1);
        armyeyes[XY2POINT(x1,y1)] = -2;
        owner = owner1[XY2POINT(x1,y1)];
        nhalfeyes = halfeyes[XY2POINT(x1,y1)];
//        bensonalive = (group[contents[XY2POINT(x1,y1)]].status == GS_BENSON_ALIVE);

        // find army and count its eyes
        while (cursor < nstack) {
          s = stack[cursor];
          x = POINT2X(s);
          y = POINT2Y(s);
          if (x > 0)                   nei[0] = s - 1;    else nei[0] = s;
          if (x < datab.boardsize - 1) nei[1] = s + 1;    else nei[1] = s;
          if (y > 0)                   nei[2] = s - GO_BOARDTABLESIZE;   else nei[2] = s;
          if (y < datab.boardsize - 1) nei[3] = s + GO_BOARDTABLESIZE;   else nei[3] = s;
          for (di = 0; di < 4; di++) {
            if (owner1[nei[di]] == owner && armyeyes[nei[di]] == -1) {
              stack[nstack] = nei[di];
              armyeyes[nei[di]] = -2;
              nstack++;
              nhalfeyes += halfeyes[nei[di]];
//              bensonalive |= (group[contents[s]].status == GS_BENSON_ALIVE);
            } 
          }
          cursor++;
        }
        // mark eyes to the whole army
        for (cursor=0;cursor < nstack; cursor++) {
          armyeyes[stack[cursor]] = nhalfeyes;
        }
      }
    }
  }
  return 0;
}

static int ScoreGame(char counting, lint *bpoints, lint *wpoints)
{
  lint ret, point, q, w, npasses;
  lint x,y,count,con,s, owner, neig[4], diag[4], score[4], narmies = 0;
  char turn;
  IntSet *common=INTSET_END, *cursor;
#ifdef _CONSOLE
  w1orw2 = 0.0;
#endif

  // init tables  
  for (y = 0; y < datab.boardsize; y++) {
    for (x = 0; x < datab.boardsize; x++) {
      owners[XY2POINT(x,y)] = 0;
      vital[XY2POINT(x,y)] = 0;
    }
    owners[XY2POINT(-1,y)] = GO_BORDER;
    owners[XY2POINT(datab.boardsize,y)] = GO_BORDER;
    owners[XY2POINT(y,-1)] = GO_BORDER;
    owners[XY2POINT(y,datab.boardsize)] = GO_BORDER;
  }

  // play twice to the end
  for (w=0;w<2;w++) 
  {
    for (q=0;q<GO_BOARDTABLESIZE*GO_BOARDTABLESIZE;q++) {
      owner1[q] = GO_BORDER;
      halfeyes[q] = 0;
      armyeyes[q] = -3;
    }
    if (w==0) turn=GO_BLACK; else turn=GO_WHITE;
    fagentb->Copy(*player[0].agent);
    fagentw->Copy(*player[1].agent);
    npasses = 0;
    // predict future
    for (q=0;q<NFUTMOVES;q++) {
      if (turn==GO_BLACK)
        point = fagentb->SelectPoint(turn, false, POINT_NO_KO);
      else
        point = fagentw->SelectPoint(turn, false, POINT_NO_KO);
      if (point == POINT_PASS) {
        npasses++;
        if (npasses>1) break;
      } else 
        npasses = 0;
      fagentb->PlaceStone(POINT2X(point), POINT2Y(point), turn);
      fagentw->PlaceStone(POINT2X(point), POINT2Y(point), turn);
      turn = COMP_COL(turn);
    }
#ifdef JAPANESE 
    // japanese scoring -> must determine seki as a special case
    //             -> run Benson for the really finished game
    if (counting == COUNTING_JAPANESE) 
    {
      fagentb->Benson(GO_BLACK, &benson_end_b);
      fagentb->Benson(GO_WHITE, &benson_end_w);
    }
#endif

    count=0;
    for (y = 0; y < datab.boardsize; y++)
    {
      for (x = 0; x < datab.boardsize; x++)
      {
        // originally a benson area?
        if (benson_b && benson_b[XY2POINT(x,y)] > POINT_PASS) {
          owner = GO_BLACK;
          halfeyes[XY2POINT(x,y)] = 100;
        } else if (benson_w && benson_w[XY2POINT(x,y)] > POINT_PASS) {
          owner = GO_WHITE;
          halfeyes[XY2POINT(x,y)] = 100;
        } else 
          owner = fagentb->GetCont(x,y);
        // empty area might be an eye for someone
        if (owner == GO_EMPTY)
        {
          neig[GO_BLACK]=0;
          neig[GO_WHITE]=0;
          neig[GO_EMPTY]=0;
          neig[GO_BORDER]=0;
          neig[fagentb->GetCont(x,y-1)]++;
          neig[fagentb->GetCont(x,y+1)]++;
          neig[fagentb->GetCont(x-1,y)]++;
          neig[fagentb->GetCont(x+1,y)]++;
          if (neig[GO_BLACK]>0 && neig[GO_WHITE]==0) owner = GO_BLACK;
          if (neig[GO_WHITE]>0 && neig[GO_BLACK]==0) owner = GO_WHITE;
          // eyes?
          if (owner != GO_EMPTY) {
            diag[GO_BLACK]=0;
            diag[GO_WHITE]=0;
            diag[GO_EMPTY]=0;
            diag[GO_BORDER]=0;
            diag[fagentb->GetCont(x-1,y-1)]++;
            diag[fagentb->GetCont(x-1,y+1)]++;
            diag[fagentb->GetCont(x+1,y-1)]++;
            diag[fagentb->GetCont(x+1,y+1)]++;
            // should not be a false eye
            if (diag[COMP_COL(owner)] - (diag[GO_BORDER]==0) <= 0) {
              if (neig[GO_EMPTY] == 0) 
                halfeyes[XY2POINT(x,y)] = 2;
              else 
                halfeyes[XY2POINT(x,y)] = 1;
            }
          }
        }
#ifdef JAPANESE 
        // seki is all neutral with Japanese scoring
        if (counting == COUNTING_JAPANESE)
        {
          if ((player[0].agent->GetCont(x,y) != GO_BLACK && 
                  owner == GO_BLACK && benson_end_b[XY2POINT(x,y)]<POINT_MIN_ONBOARD)
              || (player[0].agent->GetCont(x,y) != GO_WHITE && 
                  owner == GO_WHITE && benson_end_w[XY2POINT(x,y)]<POINT_MIN_ONBOARD))
            owner = GO_EMPTY;
        }
#endif
        owner1[XY2POINT(x,y)] = owner;
        if (owner==GO_BLACK)
          owners[XY2POINT(x,y)]++;
        else if (owner==GO_WHITE) 
          owners[XY2POINT(x,y)]--;
        count++;
      }
    }
    fagentb->CleanUpAgent(); 
    fagentw->CleanUpAgent();
    if (datab.nmoves > NMOVES_VITAL) {
      // count army eyes
      CountArmyEyes();
      // find vital points
      for (y = 0; y < datab.boardsize; y++) {
        for (x = 0; x < datab.boardsize; x++) {
          if (armyeyes[XY2POINT(x,y)]<=6) {
            vital[XY2POINT(x,y)] += (halfeyes[XY2POINT(x-1,y)] + halfeyes[XY2POINT(x+1,y)]
                     + halfeyes[XY2POINT(x,y-1)] + halfeyes[XY2POINT(x,y+1)]
                     + (halfeyes[XY2POINT(x-1,y-1)]>>1) + (halfeyes[XY2POINT(x-1,y+1)]>>1)
                     + (halfeyes[XY2POINT(x+1,y-1)]>>1) + (halfeyes[XY2POINT(x+1,y+1)]>>1))
              * (1 + 2 * (armyeyes[XY2POINT(x,y)]<=4));
          }
        }
      }
    } 
  }

  // some postprosessing:
  score[GO_BLACK]=0;
  score[GO_WHITE]=0;
  for (y = 0; y < datab.boardsize; y++)
  {
    for (x = 0; x < datab.boardsize; x++)
    {
      if (owners[XY2POINT(x,y)] == 2)
        owners[XY2POINT(x,y)] = GO_BLACK;
      else if (owners[XY2POINT(x,y)] == -2)
        owners[XY2POINT(x,y)] = GO_WHITE;
      else owners[XY2POINT(x,y)] = GO_EMPTY;
    }
  }
//FrmAlert(alertID_debug);
  // the border has always stones on it -> otherwise mark as common
  count=0;
  for (y = 0; y < datab.boardsize; y++)
    for (x = 0; x < datab.boardsize; x++) {
      if (owners[XY2POINT(x,y)] != GO_EMPTY && datab.board[count]==GO_EMPTY)
      {
        if (owners[XY2POINT(x-1,y)]==COMP_COL(owners[XY2POINT(x,y)]) ||
            owners[XY2POINT(x+1,y)]==COMP_COL(owners[XY2POINT(x,y)]) ||
            owners[XY2POINT(x,y-1)]==COMP_COL(owners[XY2POINT(x,y)]) ||
            owners[XY2POINT(x,y+1)]==COMP_COL(owners[XY2POINT(x,y)])) {
          IntSetAdd(&common,-XY2POINT(x,y));
        }
      }
      count++;
    }
  for (cursor=common;cursor!=INTSET_END;cursor=cursor->next)
    owners[-(cursor->i)] = GO_EMPTY;
  IntSetDelete(common);
//FrmAlert(alertID_debug);
  for (y = 0; y < datab.boardsize; y++)
  {
    for (x = 0; x < datab.boardsize; x++)
    {
#ifdef JAPANESE 
      // Japanese counting
      if (counting==COUNTING_JAPANESE) {
        if (owners[XY2POINT(x,y)]==GO_BLACK) {
          if (player[0].agent->GetCont(x,y) == GO_EMPTY)
            score[GO_BLACK]++;
          else if (player[0].agent->GetCont(x,y) == GO_WHITE)
            score[GO_BLACK]+=2;
        } else if (owners[XY2POINT(x,y)]==GO_WHITE) {
          if (player[0].agent->GetCont(x,y) == GO_EMPTY)
            score[GO_WHITE]++;
          else if (player[0].agent->GetCont(x,y) == GO_BLACK)
            score[GO_WHITE]+=2;
        } else {
          if (player[0].agent->GetCont(x,y) == GO_BLACK)
            score[GO_BLACK]--;
          else if (player[0].agent->GetCont(x,y) == GO_WHITE)
            score[GO_WHITE]--;
        }
      } else
#endif
      // chinese counting (also when ignoring...)
        score[owners[XY2POINT(x,y)]]++;
    }
  }
#ifdef JAPANESE 
  if (counting==COUNTING_JAPANESE) {
    score[GO_BLACK] -= datab.prisoners_black;
    score[GO_WHITE] -= datab.prisoners_white;
  }
#endif
 
  *bpoints = score[GO_BLACK];
  *wpoints = score[GO_WHITE];
 
  return 0;
}

 
#ifdef _CONSOLE
static lint ScoreMoveFuture(char origturn, lint origpoint)
{
  lint ret, q,w, npasses;
  lint x,y,count,con,s, owner, neig[4], score[4], sumscores = 0, turn, point;
  float tempw1orw2;

  if (futmoves == NULL)
    futmoves = (int *)PNEW_SIZE(sizeof(int)*NFUTMOVES);

  for (w = 0; w < 1; w++) {
    // copy agents
    fagentb->Copy(*player[0].agent);
    fagentw->Copy(*player[1].agent);
    // place given stone
    if (origpoint!=POINT_UNDECIDED) {
      fagentb->PlaceStone(POINT2X(origpoint), POINT2Y(origpoint), origturn);
      fagentw->PlaceStone(POINT2X(origpoint), POINT2Y(origpoint), origturn);
      turn = COMP_COL(origturn);
    } else turn = origturn;
    npasses = 0;
    tempw1orw2 = 0.8;
    // predict future
    for (q=0;q<NFUTMOVES;q++) {
      //      if (turn==origturn) {
        w1orw2 = tempw1orw2;
        tempw1orw2 *= 0.9;
        //      } else
        //        w1orw2 = 0.0;
      if (turn==GO_BLACK)
        point = fagentb->SelectPoint(turn, false, POINT_NO_KO);
      else
        point = fagentw->SelectPoint(turn, false, POINT_NO_KO);
      futmoves[q] = point;
      if (point == POINT_PASS) {
        npasses++;
        if (npasses>1) break;
      }
      else npasses = 0;
      fagentb->PlaceStone(POINT2X(point), POINT2Y(point), turn);
      fagentw->PlaceStone(POINT2X(point), POINT2Y(point), turn);
      turn = COMP_COL(turn);
    }
    
    count=0;
    score[GO_BLACK] = 0;
    score[GO_WHITE] = 0;
    for (y = 0; y < datab.boardsize; y++) {
      for (x = 0; x < datab.boardsize; x++) {
        owner = fagentb->GetCont(x,y);
        if (owner == GO_EMPTY) {
          neig[GO_BLACK]=0;
          neig[GO_WHITE]=0;
          neig[fagentb->GetCont(x,y-1)]++;
          neig[fagentb->GetCont(x,y+1)]++;
          neig[fagentb->GetCont(x-1,y)]++;
          neig[fagentb->GetCont(x+1,y)]++;
          if (neig[GO_BLACK]>0 && neig[GO_WHITE]==0) owner = GO_BLACK;
          if (neig[GO_WHITE]>0 && neig[GO_BLACK]==0) owner = GO_WHITE;
        }
        score[owner]++;
      }
    }
    fagentb->CleanUpAgent(); 
    fagentw->CleanUpAgent();
    if (origturn == GO_BLACK) 
      sumscores += score[GO_BLACK] - score[GO_WHITE];
    else
      sumscores += score[GO_WHITE] - score[GO_BLACK];
  }
  
  return sumscores;
}
#endif

// for Benson's algorithm. Is a region at x,y healthy for color?
bool Agent::IsHealthy(lint x, lint y, lint color, int *benson)
{
  lint s, s1, di, nstack, cont, ret = true, fill, borderok;
  IntSet *empty = INTSET_END, *border = INTSET_END, *enemy = INTSET_END, *cursor;
  int Dir[4] = {+GO_BOARDTABLESIZE, +1, -1, -GO_BOARDTABLESIZE};

  nstack = 0;
  stack[0] = XY2POINT(x, y);
  cont = group[contents[stack[0]]].color;
  if (cont != GO_EMPTY) 
    return false;
  IntSetAdd(&empty, stack[0]);

  //  printf("IsHealthy %c%d? ",X2CHAR(x),Y2INT(y),color);

  // map the region
  while (nstack > -1) {
    // s1 is needed since nstack can change
    s1 = stack[nstack];
    // border needs to be found for every empty point in region
    borderok = (group[contents[s1]].color != GO_EMPTY);
    for (di = 0; di < 4; di++) {
      s = s1 + Dir[di];
      cont = group[contents[s]].color;
      if (cont == color) {
        // own --> border
        IntSetAdd(&border, contents[s]);
        borderok = true;
      } else if (cont == GO_EMPTY) {
        // empty
        if (IntSetAdd(&empty, s))
          stack[nstack++] = s;
      } else if (cont == COMP_COL(color)) {
        // enemy
        if (IntSetAdd(&enemy, s))
          stack[nstack++] = s;
      }
    }
    nstack--;
    ret &= borderok;
  }

  //  if (ret) printf("Yes.\n"); else printf("No.\n");

  // mark region as either healthy (by index) or not healthy
  if (ret) 
    fill = XY2POINT(x,y);
  else
    fill = POINT_NOT_HEALTHY;
  for (cursor = empty; cursor != INTSET_END; cursor = cursor->next)
    benson[cursor->i] = fill;
  for (cursor = enemy; cursor != INTSET_END; cursor = cursor->next)
    benson[cursor->i] = fill;

  IntSetDelete(border);
  IntSetDelete(empty);
  IntSetDelete(enemy);

  return ret;
}

bool 
Agent::IsVital(lint point, lint gro)
{

  lint s, s1, di, nstack, cont, enemy = COMP_COL(group[gro].color);
  IntSet *region = INTSET_END;
  int Dir[4] = {+GO_BOARDTABLESIZE, +1, -1, -GO_BOARDTABLESIZE};

  //  printf("IsVital %c%d for group at %c%d? ",X2CHAR(POINT2X(point)),Y2INT(POINT2Y(point)),
  //	 X2CHAR(group[gro].x),Y2INT(group[gro].y));

  nstack = 0;
  stack[0] = point;
  if (!IntSetContains(group[gro].liberties, point)) {
    //    printf("No.\n");
    return false;
  }
  IntSetAdd(&region, point);

  // every empty intersection should be a liberty
  while (nstack > -1) {
    // s1 is needed since nstack can change
    s1 = stack[nstack];
    for (di = 0; di < 4; di++) {
      s = s1 + Dir[di];
      cont = group[contents[s]].color;
      if (cont == GO_EMPTY) {
	// empty
	if (IntSetAdd(&region, s)) {
	  if (!IntSetContains(group[gro].liberties, s)) {
	    //	    printf("No.\n");
	    IntSetDelete(region);
	    return false;
	  }
	  stack[nstack++] = s;
	}
      } else if (cont == enemy) {
	// enemy
        if (IntSetAdd(&region, s))
          stack[nstack++] = s;	
      }
    }
    nstack--;
  }

  //  printf("Yes.\n");
  IntSetDelete(region);
  return true;
}

void
MarkAsNotVital(lint point, int *benson)
{
  lint s, s1, nstack, di;
  int Dir[4] = {+GO_BOARDTABLESIZE, +1, -1, -GO_BOARDTABLESIZE};

  if (point <= POINT_PASS || benson[point] != point) 
    return;
  nstack = 0;
  stack[0] = point;
  benson[point] = POINT_NOT_HEALTHY;

  while (nstack > -1) {
    s1 = stack[nstack];
    for (di = 0; di < 4; di++) {
      s = s1 + Dir[di];
      if (benson[s] == point) {
        benson[s] = POINT_NOT_HEALTHY;
        stack[nstack++] = s;
      }
    }
    nstack--;
  }
}

// After Benson's algorithm. Is a region at x,y safe for color?
// Region in here may contain unsafe friendly stones, too.
bool Agent::IsSafe(lint x, lint y, lint color, int *benson)
{
  lint s, s1, di, nstack, cont, ret = true, fill, couldfit, firsteye = POINT_UNDECIDED;
  IntSet *region = INTSET_END, *cursor;
  int Dir[4] = {+GO_BOARDTABLESIZE, +1, -1, -GO_BOARDTABLESIZE};

  nstack = 0;
  stack[0] = XY2POINT(x, y);
  IntSetAdd(&region, stack[0]);

  //  printf("IsSafe %c%d for color %d? ",X2CHAR(x),Y2INT(y),color);

  // map the region
  while (nstack > -1) {
    // s1 is needed since nstack can change
    s1 = stack[nstack];
    // could this be an eye for the enemy?
    couldfit = 2;
    for (di = 0; di < 4; di++) {
      s = s1 + Dir[di];
      cont = group[contents[s]].color;
      if (cont == color) {
	// own
	if (group[contents[s]].status == GS_BENSON_ALIVE) {
	  couldfit = 0; // border so impossible to be an eye for the enemy
	} else { 
	  couldfit = 1; // smaller chanche of becoming an eye for the enemy
	  if (IntSetAdd(&region, s))
	    stack[nstack++] = s;
	}
      } else if (cont == GO_EMPTY || cont == COMP_COL(color)) {
	// empty or enemy
	if (IntSetAdd(&region, s))
	  stack[nstack++] = s;
      }
    }
    couldfit = (couldfit & (2 + (group[contents[s1]].color==color))) 
      && group[contents[s1]].color != COMP_COL(color);
    nstack--;
    if (couldfit) {
      if (firsteye == POINT_UNDECIDED)
	firsteye = s1;
      else {
	// second eye possibility. if too near, a third will do.
	if (ABS(firsteye-s1)==1 || ABS(firsteye-s1)==GO_BOARDTABLESIZE)
	  firsteye = GO_PASS;
	else
	  ret = false;
      }
    }
  }

  //  if (ret) printf("Yes.\n"); else printf("No.\n");

  // mark region as either safe (by index) or not safe (by point_pass)
  if (ret) 
    fill = XY2POINT(x,y);
  else
    fill = POINT_PASS;
  for (cursor = region; cursor != INTSET_END; cursor = cursor->next)
    benson[cursor->i] = fill;

  IntSetDelete(region);

  return ret;
}


// Benson's algorithm
// the table will contain points on the board if alive and something else if not.
// table benson contains point_undecided in the beginning, point_not_healthy if so,
// healthy points get a unique point on the board for each region.
// region not vital gets a point_not_healthy.
// safe groups get the status GS_BENSON_ALIVE and points of its stones are marked as themselves
// in the benson table.
void Agent::Benson(char bcolor, int **_benson)
{
  IntSet *alive = NULL, *cursor, *libecur, *cur2;
  lint q,x,y,change = true, firstvital, gro, di;
  int Dir[4] = {+GO_BOARDTABLESIZE, +1, -1, -GO_BOARDTABLESIZE};

  if (*_benson == NULL) {
    *_benson = (int *)PNEW_SIZE(sizeof(int)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
  }
  int *benson = *_benson;

  for (q=0; q<GO_BOARDTABLESIZE*GO_BOARDTABLESIZE; q++) {
    benson[q] = POINT_UNDECIDED;
  }
  // find healthy regions
  for (y=0;y<size;y++) {
    for (x=0;x<size;x++) {
      if (benson[XY2POINT(x,y)]==POINT_UNDECIDED) {
        IsHealthy(x, y, bcolor, benson);
      }
    }
  }
  // gather groups
  for (q=ngroups;q>=0;q--) 
    if (group[q].color == bcolor) {
      IntSetAdd(&alive,q);
      if (group[q].status == GS_BENSON_ALIVE)
        group[q].status = GS_UNKNOWN;
    }
  // Remove groups that do not have two vital regions until there is no change.
  // Those healthy regions that remain are vital.
  while (change) {
    change = false;
    cursor = alive;
    while (cursor != INTSET_END) {
      gro = cursor->i;
      firstvital = POINT_UNDECIDED;
      for (libecur = group[gro].liberties; libecur != INTSET_END; libecur = libecur->next) {
        if (benson[libecur->i] > POINT_PASS 
	    && benson[libecur->i] != firstvital 
	    && IsVital(benson[libecur->i], gro)) {
          if (firstvital == POINT_UNDECIDED)
            firstvital = benson[libecur->i];
	      else if (firstvital != benson[libecur->i]) {
	        // second vital found
            firstvital = POINT_PASS;
            break;
          }
        }
      }	
      // cursor must be moved here because of the intsetremove below
      cursor = cursor->next;
      // no two vital regions found --> not unconditionally alive
      if (firstvital != POINT_PASS) {
        IntSetRemove(&alive, gro);
        change = true;
        // mark its neighbouring regions as not vital
        // (liberties cannot be used because region may contain enemy stones)
        for (libecur = group[gro].stones; libecur != INTSET_END; libecur = libecur->next)
          for (di=0;di<4;di++)
            MarkAsNotVital(benson[libecur->i + Dir[di]], benson);
      }
    }
  }
  for (cursor = alive; cursor != INTSET_END; cursor = cursor->next) {
    group[cursor->i].status = GS_BENSON_ALIVE;
    for (cur2 = group[cursor->i].stones; cur2 != INTSET_END; cur2 = cur2->next)
      benson[cur2->i] = cur2->i;
  }
  IntSetDelete(alive);
  // find safe regions
  for (y=0;y<size;y++) {
    for (x=0;x<size;x++) {
      if (benson[XY2POINT(x,y)]==POINT_NOT_HEALTHY) {
        IsSafe(x, y, bcolor, benson);
      }
    }
  }
}


// ------------- class Group ------------------

Group::Group()
{
  stones = NULL;
  liberties = NULL;
  neighbours = NULL;
  color = GO_EMPTY;
  libe = 0;
}

void
Group::Clean()
{
  IntSetDelete(stones);
  IntSetDelete(liberties);
  IntSetDelete(neighbours);
  stones = NULL;
  liberties = NULL;
  neighbours = NULL;
  color = GO_EMPTY;
  libe = 0;
  status = GS_UNKNOWN;
}

Group::~Group(void)
{
  Clean();
}

Group &
Group::operator=(const Group &g)
{
  x = g.x;
  y = g.y;
  color = g.color;
  size = g.size;
  libe = g.libe;
  status = g.status;
  stones = IntSetCopy(g.stones);
  liberties = IntSetCopy(g.liberties);
  neighbours = IntSetCopy(g.neighbours);

  return *this;
}

lint 
Group::AddLiberty(lint i)
{
  IntSet **atthis = &liberties;
  libe += IntSetAdd(&liberties, i);
  return 0;
}

lint 
Group::RemoveLiberty(lint i)
{
  libe -= IntSetRemove(&liberties, i);
  return 0;
}

lint
Group::AddLiberties(class IntSet *il)
{
  IntSet **atthis = &liberties;
  libe += IntSetUnion(&liberties, il);
  return 0;
}






