/******************************************************************************\
   Go81 Plays the game of go on PalmOS devices.
   Copyright (C) 2002-2004 Tapani Raiko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
\******************************************************************************/

#pragma pack(2)

//#include <stdio.h>
#include "go81_id.h"
#include "go81.h"
#include "tapani_common.h"

// Is it the visualization version or the normal one?
//#define _VISU
// Whether the option of Japanese counting is included?
//#define  JAPANESE

DmOpenRef  Go81DB;
char    Go81DBName[] = "Go81DB";
Go81Database datab, undodatab;
static bool undoavailable = false, continueavailable=true;

static Player player[2];
static int pushbuttonblack = 0;
static int pushbuttonwhite = 2;
static int pushbuttonsize = 9;
static int pushbuttonkomi = 6;
static int pushbuttonhandicap = 0;
static int pushbuttoncounting = COUNTING_CHINESE;
static char scorestring[30]="----- wins by                ";
static char scorestring2[30]="----- leads by               ";
static char prisonerstring[30]="                             ";
static char *owners = NULL, *owner1 = NULL, *halfeyes = NULL, *vital = NULL;
static int *armyeyes = NULL, *neutralsize = NULL;
static Agent *fagentb = NULL, *fagentw = NULL;
static int *benson_w = NULL, *benson_b = NULL, *benson_end_w = NULL, *benson_end_b = NULL;
//static int debug_now = 0;


static int StartApplication(void);
static Boolean OpenDatabase(void);
static void EventLoop(void);
static void EventLoop(void);
static void StopApplication(void);
static Boolean handler_Go81(EventPtr event);
#ifndef _VISU
static Boolean handler_options(EventPtr event);
#endif
static void ChangeBitmap(void);
static void DrawBoard(void);
static int HandlePenDown(int screenX, int screenY);
static void InitBoard(void);
static void InitPlayers(void);
static void PutStoneToAllBoards(int x, int y, int turn);
static int ApplyMove(int x, int y, char turn);
static void ComputerCanMove();
static void UpdatePlayerBoards();
static void TryPass(bool resign);
static void DrawScoreMarks(lint *bscore, lint *wscore);
static void ScoreAndShow();
static void WriteCaptured(int bla, int whi);
static void EmptyChars();


DWord  PilotMain (Word cmd, Ptr cmdPBP, Word launchFlags)
{
  int error;

  if (cmd == sysAppLaunchCmdNormalLaunch)
 {

    error = StartApplication();  // Application start code
    if (error) return error;

    EventLoop();  // Event loop

    StopApplication ();  // Application stop code
  }
  return 0;
}

static int StartApplication(void)
{
  int error;

  IntSetInit(INTSETMEMORY);
 
  {
    lint q;

    fagentb = PNEW(Agent);
    fagentw = PNEW(Agent);
    owners = (char *)PNEW_SIZE(sizeof(char)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
    owner1 = (char *)PNEW_SIZE(sizeof(char)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
    halfeyes = (char *)PNEW_SIZE(sizeof(char)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
    armyeyes = (int *)PNEW_SIZE(sizeof(int)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
    vital = (char *)PNEW_SIZE(sizeof(char)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
    neutralsize = (int *)PNEW_SIZE(sizeof(int)*GO_BOARDTABLESIZE*GO_BOARDTABLESIZE);
    if (!fagentb || !fagentw || !owners || !owner1 || !halfeyes || !armyeyes || !vital || !neutralsize)
      Panic();
    for (q=0;q<GO_BOARDTABLESIZE*GO_BOARDTABLESIZE;q++) {
      owners[q] = GO_BORDER;
      vital[q] = 0;
    }
  }

  error = OpenDatabase();
  if (error) return error;
//  InitGo81Tactics();

#ifndef _VISU
  if (datab.nmoves==0 || datab.status==STATUS_SCORED || datab.turn==GO_EMPTY)
    FrmGotoForm(formID_options);
  else
#endif
    FrmGotoForm(formID_Go81);

//  FrmGotoForm(formID_newgame);
}

static void CreateEmptyDatabase(void)
{
  UInt          index = RECORD_INDEX_BOARD;
  VoidHand      RecHandle;
  Ptr           RecPointer;

    if (DmCreateDatabase(0, Go81DBName, Go81AppID, Go81DBType, false)) 
       return;   // unable to create. Bad thing.
    Go81DB = DmOpenDatabaseByTypeCreator(Go81DBType, Go81AppID, dmModeReadWrite);
    RecHandle = DmNewRecord(Go81DB, &index, sizeof(Go81Database));
    RecPointer = (Ptr)MemHandleLock(RecHandle);
//    my_memcpy((char *)RecPointer,(char *)board,BOARD_DB_DBSIZE);
    DmWrite(RecPointer,0,(void *)&datab,sizeof(Go81Database));
    MemHandleUnlock(RecHandle);
    DmReleaseRecord(Go81DB, index, false);  
}


static Boolean OpenDatabase(void)
{
  UInt          index = RECORD_INDEX_BOARD;
  VoidHand      RecHandle;
  Ptr           RecPointer;
  LocalID       dbid;
  UInt          dummyint;
  unsigned char dummybool;
  int x,y,count;
  int latest;

  Go81DB = DmOpenDatabaseByTypeCreator(Go81DBType, Go81AppID, dmModeReadWrite);
  if (!Go81DB) {
    // no database found. create new.
    // before to init the board for the database
    InitBoard();
    CreateEmptyDatabase();
    // after to init the players
    InitPlayers();
  } else {
    // Open an existing database.
    RecHandle = DmGetRecord(Go81DB, index);
    if (MemHandleSize(RecHandle)!=sizeof(Go81Database))
    {
      // Size has changed. Create a new database.
      DmOpenDatabaseInfo(Go81DB, &dbid, &dummyint, &dummyint, &dummyint, &dummybool);
      DmReleaseRecord(Go81DB, index, false);  
      DmDeleteDatabase(0,dbid);
      InitBoard();
      CreateEmptyDatabase();
      InitPlayers();
    } 
    else
    {
      // Read from database
      RecPointer = (Ptr)MemHandleLock(RecHandle);
      my_memcpy((char *)&datab,(char *)RecPointer,sizeof(Go81Database));

      // first look which players to use
      pushbuttonblack=datab.blackplayer;
      pushbuttonwhite=datab.whiteplayer;
      pushbuttonsize=datab.boardsize;
//      pushbuttondelay=datab.delay;
      pushbuttonhandicap=datab.handicap;
      pushbuttonkomi=datab.komi;
      pushbuttoncounting=datab.counting;
      // init players (empties the board)
      InitBoard();
      InitPlayers();
      // read the board again
      my_memcpy((char *)&datab,(char *)RecPointer,sizeof(Go81Database));
      MemHandleUnlock(RecHandle);
      DmReleaseRecord(Go81DB, index, false);  
      // inform agents
      UpdatePlayerBoards();
    }
  }

  return 0;
}



static void InitBoard()
{
  datab.boardsize=pushbuttonsize;
  datab.handicap=pushbuttonhandicap;
  my_memset((char *)&datab.board,GO_EMPTY,MAXBOARDSIZE*MAXBOARDSIZE);
  datab.latest=XY2POINT(datab.boardsize, datab.boardsize);
  datab.blackplayer=pushbuttonblack;
  datab.whiteplayer=pushbuttonwhite;
  datab.komi=pushbuttonkomi;
  if (pushbuttonhandicap>0) datab.komi = 0; //note: komi automatically selected.
  else datab.komi = 6;
  datab.counting = pushbuttoncounting;
  datab.prisoners_black = 0;
  datab.prisoners_white = 0;
  datab.turn=GO_BLACK;
  datab.nmoves=0;
  datab.npasses=0;
  datab.current_ko_point = XY2POINT(GO_NO_KO, GO_NO_KO);
  datab.status = STATUS_STARTING;
  // place handicap stones
  int handi1 = 2, handi2 = 4, handi3 = 6;
  if (datab.boardsize>11) {
    handi1 = 3;
    handi2 = datab.boardsize/2;
    handi3 = datab.boardsize-4;
  }
  if (pushbuttonhandicap>=2)
  {
    datab.board[handi3+pushbuttonsize*handi1] = GO_BLACK;
    datab.board[handi1+pushbuttonsize*handi3] = GO_BLACK;
    datab.turn = GO_WHITE;
  }
  if (pushbuttonhandicap>=3)
    datab.board[handi1+pushbuttonsize*handi1] = GO_BLACK;
  if (pushbuttonhandicap>=4)
    datab.board[handi3+pushbuttonsize*handi3] = GO_BLACK;
  if (pushbuttonhandicap==5 || pushbuttonhandicap==7 || pushbuttonhandicap==9)
    datab.board[handi2+pushbuttonsize*handi2] = GO_BLACK;
  if (pushbuttonhandicap>=6)
  {
    datab.board[handi1+pushbuttonsize*handi2] = GO_BLACK;
    datab.board[handi3+pushbuttonsize*handi2] = GO_BLACK;
  }
  if (pushbuttonhandicap>=8)
  {
    datab.board[handi2+pushbuttonsize*handi1] = GO_BLACK;
    datab.board[handi2+pushbuttonsize*handi3] = GO_BLACK;
  }
}


static void EventLoop(void)
{
  unsigned short err;
  int formID, delayticks;
  FormPtr form;
  EventType event;
  
  do {
    
//    delayticks=100*(int)datab.delay;
//    if (datab.delay==3) 
    delayticks = 50; // pause done elsewhere
     //wait infinitely long
//    if (scoreflag) delayticks = 0;
    EvtGetEvent(&event, delayticks); 
    
    if (SysHandleEvent(&event)) continue;
    if (MenuHandleEvent((void *)0, &event, &err)) continue;
    
    if (event.eType == frmLoadEvent) {
      formID = event.data.frmLoad.formID;
      form = FrmInitForm(formID);
      FrmSetActiveForm(form);
      switch (formID) {
      case formID_Go81:
        FrmSetEventHandler(form, (FormEventHandlerPtr) handler_Go81);
        break;
#ifndef _VISU
      case formID_options:
        FrmSetEventHandler(form, (FormEventHandlerPtr) handler_options);
        break;
#endif
      }
    }
    FrmDispatchEvent(&event);
  } while(event.eType != appStopEvent);
}

static void StopApplication(void)
{
  VoidHand      RecHandle;
  Ptr           RecPointer;

  RecHandle = DmGetRecord(Go81DB, RECORD_INDEX_BOARD);
  RecPointer = (Ptr)MemHandleLock(RecHandle);
  DmWrite(RecPointer,0,(void *)&datab,sizeof(Go81Database));
  MemHandleUnlock(RecHandle);
  DmReleaseRecord(Go81DB, RECORD_INDEX_BOARD, false);
  DmCloseDatabase(Go81DB);
}


static void ScoreAndShow()
{
  lint bs,ws, dummy;
  float komi = datab.komi;
  if (komi<-0.1) komi-=0.5;
  else komi+=0.5;
  EmptyChars();
  WinDrawChars("Scoring...",10,OFFSET_X_SCORE,OFFSET_Y_SCORE);
  ScoreGame(datab.counting,&bs,&ws);
  DrawScoreMarks(&dummy, &dummy);
  WriteScore(bs, ws, komi);
}


static Boolean handler_Go81(EventPtr event)
{
  FormPtr   form;
  int       handled = 0,border;

  switch (event->eType) 
  {
  case frmOpenEvent:
    undoavailable = false;
    form = FrmGetActiveForm();
    FrmDrawForm(form);
    DrawBoard();
/*    if (datab.status == STATUS_SCORED)
      ScoreAndShow(); */
    if (datab.status == STATUS_STARTING || datab.status == STATUS_CONTINUING)
      ComputerCanMove();
    handled = 1;
    break;
  case menuEvent:
    switch (event->data.menu.itemID)
    {
    case menuitemID_new:
//      FrmGotoForm(formID_newgame);
      InitBoard();
      InitPlayers();
      UpdatePlayerBoards();
      FrmGotoForm(formID_Go81);
      break;
#ifndef _VISU
    case menuitemID_options:
      FrmGotoForm(formID_options);
      break;
#endif
    case menuitemID_score:
      if (datab.status != STATUS_SCORED)
        datab.status = STATUS_PLEASE_SCORE;
      else 
        datab.status = STATUS_RESCORE;
      break;
    case menuitemID_undo:
      if (undoavailable) {
        undoavailable = false;
        my_memcpy((char *)&datab,(char *)&undodatab,sizeof(Go81Database));
        InitPlayers();
        UpdatePlayerBoards();
        FrmGotoForm(formID_Go81);
      } else 
        FrmAlert(alertID_no_undo);
      break;
    case menuitemID_resign:
      TryPass(true);
      break;
    case menuitemID_rules:
      FrmHelp(stringID_rules);
      break;
    case menuitemID_about:
      FrmAlert(alertID_about);
      break;
    }
    handled = 1;
    break;
  case ctlSelectEvent:  // A control button was pressed and released.
    if (event->data.ctlEnter.controlID == buttonID_pass)
      TryPass(false);
    break;
  case penDownEvent:
    handled = HandlePenDown( event->screenX, event->screenY);
    break;
  case nilEvent:
    if (datab.status==STATUS_PLEASE_SCORE || datab.status == STATUS_RESCORE) {
      ScoreAndShow();
      datab.status--;
    }
//    if (datab.delay<3 || datab.status == STATUS_STARTING) // if not pause
//      ComputerCanMove();
    handled = 1;
    break;
  }
  return handled;
}

#ifndef _VISU
static Boolean handler_options(EventPtr event)
{
  FormPtr   form;
  int       handled = 0, pbid=0;

  switch (event->eType) 
  {
  case frmOpenEvent:
    form = FrmGetActiveForm();
    // get pushbuttonvalues from the database
    pushbuttonblack=datab.blackplayer;
    pushbuttonwhite=datab.whiteplayer;
    pushbuttonsize=datab.boardsize;
    pushbuttonhandicap=datab.handicap;
    pushbuttonkomi=datab.komi;
    pushbuttoncounting=datab.counting;
    // set pushbuttons to have values to begin with
    if (pushbuttonsize==9) pbid = pushbuttonID_size9;
    else if (pushbuttonsize==13) pbid = pushbuttonID_size13;
//    else if (pushbuttonsize==19) pbid = pushbuttonID_size19;
    FrmSetControlValue(form, FrmGetObjectIndex(form, pbid), 1);
    pbid = pushbuttonID_handi0 + pushbuttonhandicap;
/*    if (pushbuttonhandicap==0) pbid = pushbuttonID_handi0;
    else if (pushbuttonhandicap==1) pbid = pushbuttonID_handi1;
    else if (pushbuttonhandicap==2) pbid = pushbuttonID_handi2;
    else if (pushbuttonhandicap==3) pbid = pushbuttonID_handi3;
    else if (pushbuttonhandicap==4) pbid = pushbuttonID_handi4;
    else if (pushbuttonhandicap==5) pbid = pushbuttonID_handi5;
    else if (pushbuttonhandicap==6) pbid = pushbuttonID_handi6;
    else if (pushbuttonhandicap==7) pbid = pushbuttonID_handi7;
    else if (pushbuttonhandicap==8) pbid = pushbuttonID_handi8;
    else if (pushbuttonhandicap==9) pbid = pushbuttonID_handi9;*/
    FrmSetControlValue(form, FrmGetObjectIndex(form, pbid), 1);
/*    if (pushbuttonkomi==0) pbid = pushbuttonID_komi_0;
    else if (pushbuttonkomi==-5) pbid = pushbuttonID_komi_m5;
    else if (pushbuttonkomi==5) pbid = pushbuttonID_komi_5;
    else if (pushbuttonkomi==6) pbid = pushbuttonID_komi_6;
    FrmSetControlValue(form, FrmGetObjectIndex(form, pbid), 1);*/
#ifdef JAPANESE
    if (pushbuttoncounting==COUNTING_CHINESE) pbid = pushbuttonID_chinese;
    else if (pushbuttoncounting==COUNTING_JAPANESE) pbid = pushbuttonID_japanese;
    FrmSetControlValue(form, FrmGetObjectIndex(form, pbid), 1);
#endif
    pushbuttonblack=datab.blackplayer;
    pushbuttonwhite=datab.whiteplayer;
    pushbuttonsize=datab.boardsize;
//    pushbuttondelay=datab.delay;
    pushbuttonhandicap=datab.handicap;
    pushbuttonkomi=datab.komi;
    // set pushbuttons to have values to begin with
    if (pushbuttonblack==0) pbid = pushbuttonID_black_h;
    else if (pushbuttonblack==1) pbid = pushbuttonID_black_p1;
    else if (pushbuttonblack==2) pbid = pushbuttonID_black_p2;
//    else if (pushbuttonblack==3) pbid = pushbuttonID_black_p3;
//    else if (pushbuttonblack==4) pbid = pushbuttonID_black_p4;
    FrmSetControlValue(form, FrmGetObjectIndex(form, pbid), 1);
    if (pushbuttonwhite==0) pbid = pushbuttonID_white_h;
    else if (pushbuttonwhite==1) pbid = pushbuttonID_white_p1;
    else if (pushbuttonwhite==2) pbid = pushbuttonID_white_p2;
//    else if (pushbuttonwhite==3) pbid = pushbuttonID_white_p3;
//    else if (pushbuttonwhite==4) pbid = pushbuttonID_white_p4;
    FrmSetControlValue(form, FrmGetObjectIndex(form, pbid), 1);
    FrmDrawForm(form);
    handled = 1;
    break;

  case ctlSelectEvent:  // A control button was pressed and released.
    handled = 1;
    if (event->data.ctlEnter.controlID== buttonID_opt_new)
    {
      InitBoard();
      InitPlayers();
      UpdatePlayerBoards();
      FrmGotoForm(formID_Go81);
    } else if (event->data.ctlEnter.controlID== buttonID_opt_cancel)
    {
      FrmGotoForm(formID_Go81);
    } else if (event->data.ctlEnter.controlID== pushbuttonID_size9)
    {
//      if (pushbuttonsize!=9)
//        CtlSetEnabled((ControlType *)FrmGetObjectPtr(form,buttonID_new_continue),false);
      pushbuttonsize=9;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_size13)
    {
//      if (pushbuttonsize!=13)
//        CtlSetEnabled((ControlType *)FrmGetObjectPtr(form,buttonID_new_continue),false);
      pushbuttonsize=13;
/*    } else if (event->data.ctlEnter.controlID== pushbuttonID_size19) 
    {
//      if (pushbuttonsize!=19)
//        CtlSetEnabled((ControlType *)FrmGetObjectPtr(form,buttonID_new_continue),false);
      pushbuttonsize=19;
      handled = 1;*/
    } else if (event->data.ctlEnter.controlID >= pushbuttonID_handi0 &&
               event->data.ctlEnter.controlID <= pushbuttonID_handi9)
    {
      pushbuttonhandicap=event->data.ctlEnter.controlID - pushbuttonID_handi0;
/*
//    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi1)
//    {
//      pushbuttonhandicap=1;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi2)
    {
      pushbuttonhandicap=2;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi3)
    {
      pushbuttonhandicap=3;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi4)
    {
      pushbuttonhandicap=4;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi5)
    {
      pushbuttonhandicap=5;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi6)
    {
      pushbuttonhandicap=6;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi7)
    {
      pushbuttonhandicap=7;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi8)
    {
      pushbuttonhandicap=8;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_handi9)
    {
      pushbuttonhandicap=9;*/
/*    } else if (event->data.ctlEnter.controlID== pushbuttonID_komi_m5)
    {
      pushbuttonkomi=-5;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_komi_0)
    {
      pushbuttonkomi=0;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_komi_5)
    {
      pushbuttonkomi=5;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_komi_6)
    {
      pushbuttonkomi=6;*/
#ifdef JAPANESE
    } else if (event->data.ctlEnter.controlID== pushbuttonID_chinese)
    {
      pushbuttoncounting=COUNTING_CHINESE;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_japanese)
    {
      pushbuttoncounting=COUNTING_JAPANESE;
#endif
    } else if (event->data.ctlEnter.controlID== buttonID_opt_continue)
    {
      InitPlayers();      
      UpdatePlayerBoards();
      FrmGotoForm(formID_Go81);
      datab.status = STATUS_CONTINUING;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_black_h)
    {
      pushbuttonblack=0;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_black_p1)
    {
      pushbuttonblack=1;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_black_p2)
    {
      pushbuttonblack=2;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_white_h)
    {
      pushbuttonwhite=0;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_white_p1)
    {
      pushbuttonwhite=1;
    } else if (event->data.ctlEnter.controlID== pushbuttonID_white_p2)
    {
      pushbuttonwhite=2;
    }
    else handled = 0;
    break;
    
  case menuEvent:
    switch (event->data.menu.itemID)
    {
    case menuitemID_opt_close:
      FrmGotoForm(formID_Go81);
      break;
    }
    handled = 1;
    break;

  case penDownEvent:
    break;
  case nilEvent:
    handled = 1;
    break;
  }
  return handled;
}
#endif


static void DrawUpdate(lint count, lint x, lint y)
{
  if (x>=datab.boardsize || x<0 || y<0 || y>=datab.boardsize) 
    return;
  if (datab.board[count]==GO_BLACK) {
    if (XY2POINT(x,y)==datab.latest) {
      DrawBitmap(bitmapID_black_latest, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
    } else {
      DrawBitmap(bitmapID_black, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
    }
  } else if (datab.board[count]==GO_WHITE) {
    if (XY2POINT(x,y)==datab.latest) {
      DrawBitmap(bitmapID_white_latest, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
    } else {
      DrawBitmap(bitmapID_white, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
    }
  } else {
    // board:
    if (y==0) {
      if (x==0) {
        DrawBitmap(bitmapID_top_left, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      } else if (x==datab.boardsize-1) {
        DrawBitmap(bitmapID_top_right, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      } else {
        DrawBitmap(bitmapID_top, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      }
    } else if (y==datab.boardsize-1) {
      if (x==0) {
        DrawBitmap(bitmapID_bottom_left, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      } else if (x==datab.boardsize-1) {
        DrawBitmap(bitmapID_bottom_right, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      } else {
        DrawBitmap(bitmapID_bottom, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      }
    } else {
      if (x==0) {
        DrawBitmap(bitmapID_left, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      } else if (x==datab.boardsize-1) {
        DrawBitmap(bitmapID_right, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
      } else {
        DrawBitmap(bitmapID_middle, XOFFSET+BITMAPSIZE*x,YOFFSET+BITMAPSIZE*y);
        // star points:
        if (datab.boardsize==9 && ((x==2 || x==6) && (y==2 || y==6)))
          DrawBitmap(bitmapID_starpoint, XOFFSET_MARK+BITMAPSIZE*x+1,YOFFSET_MARK+BITMAPSIZE*y+1);
        if (datab.boardsize==13 && (((x==3 || x==9) && (y==3 || y==9)) || (x==6 && y==6)))
          DrawBitmap(bitmapID_starpoint, XOFFSET_MARK+BITMAPSIZE*x+1,YOFFSET_MARK+BITMAPSIZE*y+1);
      }
    }
  }
}

static void DrawUpdateMark(lint x, lint y, lint owner)
{
  if (x>=datab.boardsize || x<0 || y<0 || y>=datab.boardsize) 
    return;
  if (owner==GO_BLACK) {
    DrawBitmap(bitmapID_black_mark, XOFFSET_MARK+BITMAPSIZE*x,YOFFSET_MARK+BITMAPSIZE*y);
  } else if (owner==GO_WHITE) {
    DrawBitmap(bitmapID_white_mark, XOFFSET_MARK+BITMAPSIZE*x,YOFFSET_MARK+BITMAPSIZE*y);
  } else {
    DrawBitmap(bitmapID_gray_mark, XOFFSET_MARK+BITMAPSIZE*x,YOFFSET_MARK+BITMAPSIZE*y);
  }
}


static void UpdateBoardsFromAgent(Agent *agent)
{
  lint x,y,count,con,s;
//  lint xoffset, yoffset;

  count=0;
  for (y = 0; y < datab.boardsize; y++)
  {
    for (x = 0; x < datab.boardsize; x++)
    {
      con=agent->GetCont(x,y);
      if (con!=datab.board[count]) {
        datab.board[count]=con;
        DrawUpdate(count, x, y);
      }
      count++;
    }
  }
  //remove the "latest" mark, which is now old
  s = datab.latest;
  datab.latest = XY2POINT(datab.boardsize, datab.boardsize);
  DrawUpdate(POINT2X(s) + datab.boardsize * POINT2Y(s), POINT2X(s), POINT2Y(s));
  // update ko to the database
  datab.current_ko_point = agent->current_ko_point;
  // get captured
  agent->GetCaptured(&datab.prisoners_black, &datab.prisoners_white);
}

static void DrawBoard(void) 
{
  int x,y,count;

  count=0;
  for (y = 0; y < datab.boardsize; y++)
  {
    for (x = 0; x < datab.boardsize; x++)
    {
      DrawUpdate(count,x,y);
      count++;
    }
  }
  my_memset(prisonerstring,' ',6);
  WinDrawChars(prisonerstring,6,OFFSET_X_CAPTURED,OFFSET_Y_CAPTURED+10);
  WinDrawChars(prisonerstring,6,OFFSET_X_CAPTURED,OFFSET_Y_CAPTURED+30);
  WriteCaptured(datab.prisoners_black, datab.prisoners_white);
  EmptyChars();
}

static void EmptyChars()
{
  WinDrawChars("                                             ",45,OFFSET_X_SCORE,OFFSET_Y_SCORE);
}

void DrawBitmap(int bitmapID, int x, int y)
{
  VoidHand   bitmaphandle;
  BitmapPtr  bitmap;

  bitmaphandle = DmGet1Resource('Tbmp', bitmapID + 
        (bitmapID <= bitmapID_white_latest)*20*(datab.boardsize==13));
  bitmap = (BitmapPtr)MemHandleLock(bitmaphandle);
  WinDrawBitmap(bitmap, x, y);
  MemHandleUnlock(bitmaphandle);
}

void TryPass(bool resign)
{
  char turn=datab.turn;
  Player *pla;
  int ret;

  if (turn==GO_BLACK) pla = &player[0];
  else pla = &player[1];
  if (pla->human)
  {
    if (resign)
      ApplyMove(GO_PASS,GO_NO_KO,turn);
    else {
      ret = ApplyMove(GO_PASS,GO_PASS,turn);
      ComputerCanMove();
    }
  }
}


static int HandlePenDown(int screenX, int screenY)
{
  int x,y,ind,ret=0;
  char turn;
  Player *pla;
  
  turn=datab.turn;

  // is game already finished? Start a new game.
  if (turn == GO_EMPTY || datab.status == STATUS_SCORED) {
#ifndef _VISU
    FrmGotoForm(formID_options);
#endif
#ifdef _VISU
    InitBoard();
    InitPlayers();
    UpdatePlayerBoards();
    DrawBoard();
    datab.status = STATUS_STARTING;
    ComputerCanMove();
#endif
    return 1;
  }
  // clear estimated score
  if (datab.status == STATUS_SCORE_ESTIMATED)
  {
    DrawBoard();
  }
  if (turn==GO_BLACK) pla = &player[0];
  else pla = &player[1];
  if (pla->human)
  {
    x= (screenX-XOFFSET)/BITMAPSIZE;
    y= (screenY-YOFFSET)/BITMAPSIZE;
    if (x>=0 && x < datab.boardsize && y >= 0 && y < datab.boardsize) {
      undoavailable = true;
      my_memcpy((char *)&undodatab,(char *)&datab,sizeof(Go81Database));
      ret = ApplyMove(x,y,turn);
    }
    ComputerCanMove();
  } else {
    ComputerCanMove();
  }
  if (datab.status == STATUS_SCORE_ESTIMATED)
    datab.status = STATUS_PLAYING;
  return ret;
}

static int ApplyMove(int x, int y, char turn)
{
  lint ret, blackcells, whitecells, dummy;
  int latest;
   
  if (turn == GO_EMPTY || datab.status == STATUS_SCORED) return -1;
 
  ret = player[0].agent->PlaceStone(x, y, turn);
  player[1].agent->PlaceStone(x, y, turn);
// illegal move?
//  if (board[ind]!=GO_EMPTY) {
  if (!legal(ret)) {
    if (datab.status != STATUS_SCORE_ESTIMATED && ret!=GO_MOVE_UNEMPTY)
      FrmAlert(alertID_illegal);
    return 1;
  }
// apply move
  UpdateBoardsFromAgent(player[0].agent);
  datab.turn=COMP_COL(datab.turn);
  datab.nmoves++;
  datab.latest = XY2POINT(x,y);
  DrawUpdate(y*datab.boardsize+x,x,y);
  if (x==GO_PASS)
  {
    if (XY2POINT(x,y)==POINT_RESIGN) {
      if (turn==GO_BLACK)
        WinDrawChars("Black resigns.",14,OFFSET_X_SCORE,OFFSET_Y_SCORE);
      else
        WinDrawChars("White resigns.",14,OFFSET_X_SCORE,OFFSET_Y_SCORE);
      datab.status = STATUS_SCORED;
      datab.turn = GO_EMPTY;
      return 0;
    }
    if (POINT2X(datab.current_ko_point)>=0 &&
        player[0].agent->MoveType(POINT2X(datab.current_ko_point),
                                  POINT2Y(datab.current_ko_point), COMP_COL(datab.turn))
                                  ==GO_MOVE_KO)
    {
      // do not count as a pass since a move was illegal because of ko.
    } else
      datab.npasses++;
    EmptyChars();
    if (turn==GO_BLACK)
      WinDrawChars("Black passes.",13,OFFSET_X_SCORE,OFFSET_Y_SCORE);
    else
      WinDrawChars("White passes.",13,OFFSET_X_SCORE,OFFSET_Y_SCORE);
    if (datab.npasses==2)
    {
      lint bs,ws;
      float komi = datab.komi;
      if (komi<-0.1) komi-=0.5;
      else komi+=0.5;
      if (datab.counting== COUNTING_CHINESE && 
         (datab.status == STATUS_PALM2PASSED ||
          (turn == GO_BLACK && (player[0].agent->onoff & ONOFF_SLOW)) ||
          (turn == GO_WHITE && (player[1].agent->onoff & ONOFF_SLOW))))
        DrawScoreMarks(&bs,&ws);
      else
      {
        EmptyChars();
        WinDrawChars("Scoring...",10,OFFSET_X_SCORE,OFFSET_Y_SCORE);
        ScoreGame(datab.counting,&bs,&ws);
        DrawScoreMarks(&dummy, &dummy);
      }
      WriteScore(bs, ws, komi);
/*      if (bs>ws+komi) 
        FrmAlert(alertID_blackwins);
      else 
        FrmAlert(alertID_whitewins);*/
      datab.status = STATUS_SCORED;
      datab.turn = GO_EMPTY;
    } else {
      if ((turn == GO_BLACK && (player[0].agent->onoff & ONOFF_SLOW)) ||
          (turn == GO_WHITE && (player[1].agent->onoff & ONOFF_SLOW)))
        datab.status = STATUS_PALM2PASSED;
      else datab.status = STATUS_PASSED;
    }
  }
  else 
  { 
    datab.npasses = 0;
    datab.status = STATUS_PLAYING;
    EmptyChars();
    WriteCaptured(datab.prisoners_black, datab.prisoners_white);
  }
  
  return 0;
}


static void ComputerCanMove()
{
  char turn;
  Player *pla;
  lint point,ran,x,y,s, count;
  
  turn=datab.turn;
  if (turn==GO_BLACK) pla = &player[0];
  else pla = &player[1];
  
  if (turn != GO_EMPTY && datab.status != STATUS_SCORED && pla->human==false)
  {
    // the very first (computer) move on 9 times 9 board is random
    if (datab.nmoves == 0 && datab.boardsize==9 && datab.handicap < 2) {
      //      if (datab.boardsize==9) {
	ran = rand() % 7;
	if (ran < 4) x = 4;
	else if (ran < 6) x = 5;
	else x = 6;
	if (ran < 2) y = 4;
	else if (ran == 2 || ran == 4) y = 5;
	else y = 6;
	//      }
      // mirroring
      ran = rand();
      if (ran & 1) x = datab.boardsize - 1 - x;
      if (ran & 2) y = datab.boardsize - 1 - y;
      if (ran & 4) {
	ran = x;
	x = y;
	y = ran;
      }	
      point = XY2POINT(x,y);
    } else {
//      WinDrawChars("Benson...         ",18,OFFSET_X_SCORE,OFFSET_Y_SCORE);
      EmptyChars();
      WinDrawChars("Thinking...",11,OFFSET_X_SCORE,OFFSET_Y_SCORE);
     
      // benson
      if (datab.nmoves > NMOVES_BENSON) {
        player[0].agent->Benson(GO_BLACK, &benson_b);
        player[0].agent->Benson(GO_WHITE, &benson_w);
        count = 0;
        for (y = 0; y < datab.boardsize; y++) {
	      for (x = 0; x < datab.boardsize; x++) {
	        s = XY2POINT(x, y);
	        if (player[0].agent->group[player[0].agent->contents[s]].status == GS_BENSON_ALIVE)
	          player[1].agent->group[player[1].agent->contents[s]].status = GS_BENSON_ALIVE;
	        if (benson_b[s] > POINT_PASS || benson_w[s] > POINT_PASS) {
	          player[0].agent->movetype[s] = GO_MOVE_BENSON;
	          player[0].agent->worth1[s] = -9999999.9;
	          player[1].agent->movetype[s] = GO_MOVE_BENSON;
	          player[1].agent->worth1[s] = -9999999.9;
/*            if (benson_b[s]>POINT_PASS) 
              DrawUpdateMark(count, x, y, GO_BLACK);
            else
              DrawUpdateMark(count, x, y, GO_WHITE);*/
            }
	        count++;
	      }
	    }
      }
      // consult agent
      point = pla->agent->SelectPoint(turn, true, datab.latest);
    }
    ApplyMove(POINT2X(point), POINT2Y(point), turn);
  }
}



// no checking!
static void PutStoneToAllBoards(int x, int y, int turn)
{
  datab.board[x + y * datab.boardsize]=turn;
  datab.latest=XY2POINT(x,y);
  if (turn!=GO_EMPTY) {
    player[0].agent->PlaceStone(x, y, turn);
    player[1].agent->PlaceStone(x, y, turn);
  }
  datab.current_ko_point = player[0].agent->current_ko_point;
}

// returns chinese score
static void DrawScoreMarks(lint *bscore, lint *wscore)
{
  lint x,y,count;
  lint score[4];
  score[GO_BLACK]=0;
  score[GO_WHITE]=0;
  
  for (y = 0; y < datab.boardsize; y++)
  {
    for (x = 0; x < datab.boardsize; x++)
    {
      DrawUpdateMark(x, y, owners[XY2POINT(x,y)]);
      score[owners[XY2POINT(x,y)]]++;
    }
  }
  *bscore = (lint)score[GO_BLACK];
  *wscore = (lint)score[GO_WHITE];
}



// ------------------- PLAYERS -----------------------------


static void UpdatePlayerBoards()
{
  int count,x,y;
  int latest, ko_p;
  latest = datab.latest;
  ko_p = datab.current_ko_point;
  
  // place stones to all boards
  count=0;
  for (y = 0; y < datab.boardsize; y++)
  {
    for (x = 0; x < datab.boardsize; x++)
    {
      PutStoneToAllBoards(x,y,datab.board[count]);
      count++;
    }
  }
  // fixes the latest move.
  datab.latest=latest;
  // fix ko
  datab.current_ko_point = ko_p;
  player[0].agent->current_ko_point = ko_p;
  player[1].agent->current_ko_point = ko_p;
  // set captured
  player[0].agent->SetCaptured(datab.prisoners_black, datab.prisoners_white);
  player[1].agent->SetCaptured(datab.prisoners_black, datab.prisoners_white);
}

// uses values pushbuttonblack and pushbuttonwhite
void InitPlayers(void)
{
  lint onoff = 0, q;
  datab.blackplayer=pushbuttonblack;
  datab.whiteplayer=pushbuttonwhite;
  //datab.delay=pushbuttondelay;
  if (benson_b)
    for (q=0; q<GO_BOARDTABLESIZE*GO_BOARDTABLESIZE; q++) {
      benson_b[q] = POINT_UNDECIDED;
    }
  if (benson_w)
    for (q=0; q<GO_BOARDTABLESIZE*GO_BOARDTABLESIZE; q++) {
      benson_w[q] = POINT_UNDECIDED;
    }
  
// human
  if (pushbuttonblack==0)
    player[0].InitPlayer(GO_BLACK, true, onoff);
// computer
  if (pushbuttonblack==1)
    player[0].InitPlayer(GO_BLACK, false, onoff);
  if (pushbuttonblack==2)
    player[0].InitPlayer(GO_BLACK, false, onoff | ONOFF_SLOW);
//  if (pushbuttonblack==3)
//    player[0].InitPlayer(GO_BLACK, false, onoff | ONOFF_DEPTH1);
// human
  if (pushbuttonwhite==0)
    player[1].InitPlayer(GO_WHITE, true, onoff);
// computer
  if (pushbuttonwhite==1)
    player[1].InitPlayer(GO_WHITE, false, onoff);
  if (pushbuttonwhite==2)
    player[1].InitPlayer(GO_WHITE, false, onoff | ONOFF_SLOW);
//  if (pushbuttonwhite==3)
//    player[1].InitPlayer(GO_WHITE, false, onoff | ONOFF_DEPTH1);
}

Player::Player()
{
  human = true;
  agent = NULL;
}

Player::Player(int _color, char _human, int _onoff)
{
  InitPlayer(_color, _human, _onoff);
}

// gets size for agent from datab.boardsize!
void Player::InitPlayer(int _color, char _human, int _onoff)
{
  color = _color;
  human = _human;
  if (agent==NULL) {
    agent = PNEW(Agent);
    my_memset((char *)agent,0,sizeof(Agent));
  }
  agent->InitAgent(datab.boardsize, color, _onoff);
}

Player::~Player()
{
  PDEL(agent);
}


void WriteCaptured(int bla, int whi)
{
  char *wr = prisonerstring;

  // black has captured white stones
  WinDrawChars("B",1,OFFSET_X_CAPTURED,OFFSET_Y_CAPTURED);
  if (whi>99)
    *(wr++)='0'+(char)(whi/(int)100);
  if (whi>9)
    *(wr++)='0'+(char)((whi%(int)100)/(int)10);
  *(wr++)='0'+(char)(whi%(int)10);
  *(wr++)=' ';
  WinDrawChars(prisonerstring,(int)(wr-prisonerstring),OFFSET_X_CAPTURED,OFFSET_Y_CAPTURED+10);
  // white has captured black stones
  WinDrawChars("W",1,OFFSET_X_CAPTURED,OFFSET_Y_CAPTURED+22);
  wr = prisonerstring;
  if (bla>99)
    *(wr++)='0'+(char)(bla/(int)100);
  if (bla>9)
    *(wr++)='0'+(char)((bla%(int)100)/(int)10);
  *(wr++)='0'+(char)(bla%(int)10);
  *(wr++)=' ';
  WinDrawChars(prisonerstring,(int)(wr-prisonerstring),OFFSET_X_CAPTURED,OFFSET_Y_CAPTURED+32);
}

void WriteScore(int blackp, int whitep, float komi)
{
  char *wr = scorestring;
  if (datab.status==STATUS_PLEASE_SCORE)
    wr = scorestring2;
  float diff;
  
  if (blackp > (whitep+komi)) {
    *(wr++)='B';
    *(wr++)='l';
    *(wr++)='a';
    *(wr++)='c';
    *(wr++)='k';
    diff = blackp-(whitep+komi);
  } else {
    *(wr++)='W';
    *(wr++)='h';
    *(wr++)='i';
    *(wr++)='t';
    *(wr++)='e';
    diff = (whitep+komi)-blackp;
  }
  diff -= 0.4;
  wr = &scorestring[14];
  if (datab.status==STATUS_PLEASE_SCORE)
    wr = &scorestring2[15];

  if (diff>99.9)
    *(wr++)='0'+(char)((((int)diff)/(int)100)%(int)10);
  if (diff>9.9)
    *(wr++)='0'+(char)((((int)diff)/(int)10)%(int)10);
  *(wr++)='0'+(char)(((int)diff)%(int)10);
  *(wr++)='.';
  *(wr++)='5';
  if (datab.status==STATUS_PLEASE_SCORE)
    WinDrawChars(scorestring2,(int)(wr-scorestring2),OFFSET_X_SCORE,OFFSET_Y_SCORE);
  else
    WinDrawChars(scorestring,(int)(wr-scorestring),OFFSET_X_SCORE,OFFSET_Y_SCORE);
}

// these should be .cpp files but this way avoids some linker problems

#include "go81_agent_cpp.h"

#include "intset_cpp.h"


