run.evolution <- function(simulation.name, n.generations, n.strain.types, n.core.genes.to.simulate, gene.length, mutation.rate, recombination.rate, recombination.acceptance.par, niche.overlap, migration, save.interval, save.path, save.extension, print.output=TRUE, population=NULL, start.generation=1) {
	
	# A population from a previous simulation can be given as 
	# the "population" parameter. The index of the first generation
	# to be simulated is specified by "start.generation". So, if
	# a population is given, start.generation should be the generation
	# for the given population + 1.
	
	# Initialization of the population, unless given as a parameter:
	if (is.null(population)) {
		population <- initialize.data.structures(n.strain.types=n.strain.types, n.core.genes.to.simulate=n.core.genes.to.simulate, gene.length=gene.length, niche.overlap=niche.overlap)
		# population has fields: alleles, allele.snps, allele.distances,
		# strain.info, event.counts, n.strain.types
	}
	
	time.start <- proc.time()['elapsed']
	
	# Simulate the given number of generations
	for (generation.index in seq(start.generation,n.generations)) {
		
		#print(paste(generation.index, ' ' elapsed))
		
		# Sample descendants
		population <- sample.offspring(pop=population, migration=migration, n.strain.types=n.strain.types)
		
		# Recombinations
		population <- add.core.recombinations(pop=population, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par)	
		
		# Mutations
		population <- add.mutations(pop=population, mutation.rate=mutation.rate)	
		
		# Clean alleles (Remove nonexistent alleles, check that alleles are
		# defined using SNPs present in the minority of a population)
		population <- clean.allele.descriptions(pop=population, gen=generation.index)
		
		# Save at regular intervals
		if (generation.index%%save.interval == 0) {
			elapsed <- proc.time()['elapsed']-time.start
			save(population, elapsed, file=paste(save.path, '/gen', generation.index, save.extension, '.RData', sep=''))
#			
			if (print.output) {
				print(paste('Gen ', generation.index, ' complete. Time=', elapsed, '.', sep=''))
			}
			check.population.validity(pop=population, generation.index=generation.index, save.extension=save.extension)
		}
	}
	
	return(population)
}



################################
## Initialize data structures ##
################################
initialize.data.structures <- function(n.strain.types, n.core.genes.to.simulate, gene.length, niche.overlap) {
	
	# Check that niche.overlap has a sensible value
	if (any(niche.overlap>sum(n.strain.types))) {stop('Niche overlap too large.')}
	if (n.core.genes.to.simulate<2) {stop('Num genes to simulate <2')}
	
	pop <- list()
	pop$gene.length <- gene.length
	pop$n.strain.types <- n.strain.types
	pop$max.n.alleles <- 40 # Maximum number of alleles at any locus
	n.strains.in.population <- sum(n.strain.types)
	
	# Initialize alleles.
	pop$alleles <- matrix(1, nrow=n.strains.in.population, ncol=n.core.genes.to.simulate)
	# Minor SNPs in the alleles. The SNPs must be ordered by their site
	# all the time.
	pop$allele.snps <- list()
	for (i in 1:n.core.genes.to.simulate) {
		# Initially, there is only one allele per gene.
		pop$allele.snps[[i]] <- list()
		pop$allele.snps[[i]][[1]] <- numeric() # The allele has only major SNPs.
		# pop$allele.snps[[i]][[j]] lists minor count SNPs in the j_th allele 
		# in the i_th locus.
	}
	
	# Allocate an unnecessarily large matrix for allele distances, such that 
	# its size does not change. Only upper-left corner of the matrix is used
	# to record the distances of the alleles.
	pop$allele.distances <- list()
	for (i in 1:n.core.genes.to.simulate) {
		pop$allele.distances[[i]] <- matrix(0, nrow=pop$max.n.alleles, ncol=pop$max.n.alleles)
		# Number of distinct alleles at any locus should not exceed the 
		# the specified maximum allele count. If it does at some point, the 
		# maximum should be increased.
	}
	
	counts <- initial.counts(niche.overlap=niche.overlap, n.strain.types=n.strain.types)
	
	type <- c(rep('A', n.strain.types['A']), rep('B', n.strain.types['B']))
	environment <- c(rep('A', counts['A','A']), rep('AB',sum(counts[,'AB'])), rep('B', counts['B','B']))
	
	pop$strain.info <- list(type=type, environment=environment)
	
	# Variable to record the realized number of events in the history of 
	# the population
	pop$event.counts <- list()
	pop$event.counts$core.mut <- 0
	pop$event.counts$core.rec <- 0
	pop$event.counts$rec.snp.counts <- rep(0, 500)
	
	return(pop)
}




sample.offspring <- function(pop, migration, n.strain.types, updated.overlap=NULL) {
	
	# Current counts of different kinds of strains
	counts <- matrix(NA, nrow=2, ncol=3)
	rownames(counts) <- c('A','B')
	colnames(counts) <- c('A','AB','B')
	for (type in c('A','B')) {
		for (env in c('A','AB','B')) {
			counts[type, env] <- length(which(pop$strain.info$type==type & pop$strain.info$environment==env))
		}
	}
	
	niche.overlap <- counts['A','AB'] + counts['B','AB'] # Current overlap size
	
	if (!is.null(updated.overlap)) {
		new.overlap <- updated.overlap
		new.counts <- initial.counts(niche.overlap=updated.overlap, n.strain.types=n.strain.types)
		# These are used to specify the counts in 
		# different environments after sampling.
	} else {
		new.overlap <- niche.overlap
		new.counts <- counts
	}
	
	
	
	# Sample strains for environments A and B
	new.strains <- list()
	for (env.now in c('A','B')) {
		prob.of.AB <- migration * counts[env.now,'AB'] / (migration * counts[env.now,'AB'] + counts[env.now,env.now])
		num.from.AB <- rbinom(n=1, size=new.counts[env.now,env.now], prob=prob.of.AB)
		if (num.from.AB > 0) {
			strains.in.AB <- which(pop$strain.info$type==env.now & pop$strain.info$environment=='AB')
			if (length(strains.in.AB)==1) {
				new.strains.from.AB <- rep(strains.in.AB, num.from.AB)
			} else {
				new.strains.from.AB <- sample(x=strains.in.AB, size=num.from.AB, replace=T)
			}
		} else {
			new.strains.from.AB <- NULL
		}
		num.from.env <- new.counts[env.now,env.now] - num.from.AB
		if (num.from.env > 0) {
			strains.in.env <- which(pop$strain.info$type==env.now & pop$strain.info$environment==env.now)
			if (length(strains.in.env)==1) {
				new.strains.from.env <- rep(strains.in.env, num.from.env)
			} else {
				new.strains.from.env <- sample(x=strains.in.env, size=num.from.env, replace=T)
			}
		} else {
			new.strains.from.env <- NULL
		}
		new.strains[[env.now]] <- c(new.strains.from.AB, new.strains.from.env)
	}
	
	
	# Sample strains for environment AB
	num.strains.in.A.or.B <- counts['A','A'] + counts['B','B']
	prob.of.A.or.B <- migration * num.strains.in.A.or.B / (migration * num.strains.in.A.or.B + niche.overlap)
	num.from.A.or.B <- rbinom(n=1, size=new.overlap, prob=prob.of.A.or.B)
	
	if (num.from.A.or.B > 0) {
		strains.in.A.or.B <- which(pop$strain.info$environment=='A' | pop$strain.info$environment=='B')
		if (length(strains.in.A.or.B)==1) {
			strains.from.A.or.B <- rep(strains.in.A.or.B, num.from.A.or.B)
		} else {
			strains.from.A.or.B <- sample(x=strains.in.A.or.B, size=num.from.A.or.B, replace=T)
		}
	} else {
		strains.from.A.or.B <- NULL
	}
	num.from.AB <- new.overlap - num.from.A.or.B
	if (num.from.AB > 0) {
		strains.in.AB <- which(pop$strain.info$environment=='AB')
		if (length(strains.in.AB)==1) {
			strains.from.AB <- rep(strains.in.AB, num.from.AB)
		} else {
			strains.from.AB <- sample(x=strains.in.AB, size=num.from.AB, replace=T)
		}
	} else {
		strains.from.AB <- NULL
	}
	new.strains[['AB']] <- c(strains.from.A.or.B, strains.from.AB)
	
	
#	# Code for testing
#	for (env.now in c('A','B','AB')) {
#		strains.now <- new.strains[[env.now]]
#		new.counts <- matrix(NA, nrow=2, ncol=3)
#		rownames(new.counts) <- c('A','B')
#		colnames(new.counts) <- c('A','AB','B')
#		for (type in c('A','B')) {
#			for (env in c('A','AB','B')) {
#				new.counts[type, env] <- length(which(pop$strain.info$type[strains.now]==type & pop$strain.info$environment[strains.now]==env))
#			}
#		}
#				
#		print(paste('New cluster ', env.now,':', sep=''))
#		print(paste('Proportion of A from A:', new.counts['A','A']/sum(new.counts), sep=''))
#		print(paste('Proportion of B from B:', new.counts['B','B']/sum(new.counts), sep=''))
#		print(paste('Proportion of A from AB:', new.counts['A','AB']/sum(new.counts), sep=''))
#		print(paste('Proportion of B from AB:', new.counts['B','AB']/sum(new.counts), sep=''))
#	}
	
	#	test.counts <- matrix(NA, nrow=2, ncol=3)
	#	rownames(test.counts) <- c('A','B')
	#	colnames(test.counts) <- c('A','AB','B')
	#	for (type in c('A','B')) {
	#		for (env in c('A','AB','B')) {
	#			test.counts[type, env] <- length(which(pop$strain.info$type==type & pop$strain.info$environment==env))
	#		}
	#	}
	#	if (test.counts['A','B']>0 | test.counts['B','A']>0) {
	#		browser()
	#	}
	
	if (is.null(updated.overlap)) {
		to.remove <- list()
		to.be.used.in.replace <- list()
		for (env.now in c('A','B','AB')) {
			
			offspring.indices <- new.strains[[env.now]]
			current.strains <- which(pop$strain.info$environment == env.now)
			
			# (The following code avoids allocation of new tables..)
			# Remove current strains that don't appear in the offspring
			to.remove[[env.now]] <- setdiff(current.strains, offspring.indices)
			# Replace with offspring indices that are duplicated. In addition to the
			# duplicates, add once each new index.
			new.indices <- setdiff(offspring.indices, current.strains)
			to.be.used.in.replace[[env.now]] <- offspring.indices[duplicated(offspring.indices)]
			to.be.used.in.replace[[env.now]] <- c(to.be.used.in.replace[[env.now]], new.indices)
			if (length(to.be.used.in.replace[[env.now]])>1) {
				# Reshuffle such that the new indices are not in the end.
				to.be.used.in.replace[[env.now]] <- to.be.used.in.replace[[env.now]][sample(length(to.be.used.in.replace[[env.now]]))]
			}
			
		}
		
		all.to.remove <- unlist(to.remove)
		all.to.be.used.in.replace <- unlist(to.be.used.in.replace)
		
		#pop$core.genes[all.to.remove,] <- pop$core.genes[all.to.be.used.in.replace,]
		pop$alleles[all.to.remove,] <- pop$alleles[all.to.be.used.in.replace,]
		pop$strain.info$type[all.to.remove] <- pop$strain.info$type[all.to.be.used.in.replace]
	
	} else {
	
		# Create completely new data structures "alleles" and "strain.info"
		all.offspring <- c(new.strains[['A']], new.strains[['AB']], new.strains[['B']])
		pop$alleles <- pop$alleles[all.offspring,]
		pop$strain.info$type <- pop$strain.info$type[all.offspring]
		pop$strain.info$environment <- c(rep('A',new.counts['A','A']), rep('AB',new.overlap), rep('B', new.counts['B','B']))
		
	}
	
	return(pop)

}





add.mutations <- function(pop, mutation.rate) {
	# Randomly select the given number of strains
	# to be mutated at each core gene.
	n.strains.in.population <- sum(pop$n.strain.types)
	n.genes <- ncol(pop$alleles)
	n.strains.to.mutate <- rbinom(n=n.genes, size=n.strains.in.population, prob=mutation.rate/n.strains.in.population)
	
	pop$event.counts$core.mut <- pop$event.counts$core.mut + sum(n.strains.to.mutate)
	
	for (gene.index in 1:n.genes) {
		if (n.strains.to.mutate[gene.index] > 0) {
			
			strains.to.mutate <- sample(x=n.strains.in.population, size=n.strains.to.mutate[gene.index])
			
			for (strain.index in strains.to.mutate) {
				
				site.to.mutate <- sample(x=pop$gene.length, size=1)
				# Mutate strain "strain.index" at site "site.to.mutate" in 
				# gene "gene.index"
#				
				# First compute the updated SNP set
				old.allele.label <- pop$alleles[strain.index, gene.index]
				old.allele.snps <- pop$allele.snps[[gene.index]][[old.allele.label]]
				if (length(old.allele.snps)==0) {
					new.allele.snps <- site.to.mutate
					snp.added <- TRUE
				} else if (any(old.allele.snps==site.to.mutate)) {
					# Remove the SNP from list
					if (length(old.allele.snps)==1) {new.allele.snps <- numeric()} 
					else {new.allele.snps <- sort(setdiff(old.allele.snps, site.to.mutate))}
					snp.added <- FALSE
				} else {
					new.allele.snps <- sort(union(old.allele.snps, site.to.mutate))
					snp.added <- TRUE
				}
#				
				# Check if the updated SNP set corresponds
				# to some already existing allele
				is.same.allele <- lapply(pop$allele.snps[[gene.index]], function(x){
					return.value <- FALSE
					if (length(x)==length(new.allele.snps)) {
						if (all(x==new.allele.snps)) {
							# For this to work, SNPs in alleles must be ordered
							return.value <- TRUE
						}
					}
					return(return.value)
				})
				is.same.allele <- unlist(is.same.allele) # Max one should be true
#				
				same.allele.index <- which(is.same.allele)
				if (length(same.allele.index)==0) {
					# The new allele did not exist previously
#					
					# Select new allele label
					new.allele.label <- length(pop$allele.snps[[gene.index]]) + 1
#
					if (new.allele.label > pop$max.n.alleles) {
						# Allocate more space to allele distances
						for (i in 1:n.genes) {
							aux <- matrix(0, nrow=pop$max.n.alleles*2, ncol=pop$max.n.alleles*2)
							aux[seq(1,pop$max.n.alleles),seq(1,pop$max.n.alleles)] <- pop$allele.distances[[i]]
							pop$allele.distances[[i]] <- aux
						}
						pop$max.n.alleles <- 2*pop$max.n.alleles
					}
#
					# Update allele code of the strain
					pop$alleles[strain.index,gene.index] <- new.allele.label
#					
					# Update allele descriptions
					pop$allele.snps[[gene.index]][[new.allele.label]] <- new.allele.snps
#					
					# Retrieve distances of the old allele to all other alleles
					old.allele.distances <- pop$allele.distances[[gene.index]][old.allele.label,seq(1,new.allele.label-1)]
#					
					# Compute distances of the new allele to other alleles
					had.the.snp <- lapply(pop$allele.snps[[gene.index]], function(x){
						any(x==site.to.mutate)
					})
					had.the.snp <- which(unlist(had.the.snp)[seq(1,new.allele.label-1)])
					did.not.have.the.snp <- setdiff(seq(1,new.allele.label-1),had.the.snp)
					new.allele.distances <- old.allele.distances
#					
					# If a SNP was added, the distances to all old alleles
					# which did not have the SNP increased by one and 
					# the distances to all old alleles which had the SNP
					# decreased by one.
					if (length(had.the.snp)>0) {new.allele.distances[had.the.snp] <- new.allele.distances[had.the.snp] + ifelse(snp.added, -1, 1) }
#					
					if (length(did.not.have.the.snp)>0) {new.allele.distances[did.not.have.the.snp] <- new.allele.distances[did.not.have.the.snp] + ifelse(snp.added, 1, -1) }
#					
					# Update allele distance matrix
					pop$allele.distances[[gene.index]][seq(1,new.allele.label-1),new.allele.label] <- new.allele.distances
					pop$allele.distances[[gene.index]][new.allele.label,seq(1,new.allele.label-1)] <- new.allele.distances
#					
				} else if (length(same.allele.index)==1) {
					# The new allele is one of the already existing alleles.
					# Just update the allele label.
					pop$alleles[strain.index, gene.index] <- same.allele.index
#				
				} else {
					save(list=ls(), file=paste('error_state', save.extension, '.RData', sep=''))
					stop('Multiple identical alleles')
				}
			}
		}
	}
	return(pop)
}




add.core.recombinations <- function(pop, recombination.rate, recombination.acceptance.par, add.bookkeeping=FALSE) {
	# Randomly select pairs of strains to recombine and try if the 
	# recombination is accepted.
	#
	# If add.bookkeeping=TRUE, then record information of each recombination
	# trial.
	
	if (add.bookkeeping) {
		donor.type <- rep(NA, 10000)
		recipient.type <- rep(NA, 10000)
		donor.env <- rep(NA, 10000)
		recipient.env <- rep(NA, 10000)
		allele.dist <- rep(NA, 10000)
		accepted <- rep(FALSE, 10000)
		n.trials <- 0
	}
	
	n.strains.in.population <- sum(pop$n.strain.types)
	n.genes <- ncol(pop$alleles)
	
	# Loop over all core genes. Add recombinations at each.
	for (gene.index in 1:n.genes) {
		
		n.rec.trials <- rpois(n=1, lambda=recombination.rate)
		
		if (n.rec.trials > 0) {
			
			for (trial.index in 1:n.rec.trials) {
				
				# Sample recipient and donor
				pair.of.strains <- sample(n.strains.in.population, 2)
				recipient <- pair.of.strains[1]
				donor <- pair.of.strains[2]
				
				if (add.bookkeeping) {
					n.trials <- n.trials + 1
					donor.type[n.trials] <- pop$strain.info$type[donor]
					recipient.type[n.trials] <- pop$strain.info$type[recipient]
					donor.env[n.trials] <- pop$strain.info$environment[donor]
					recipient.env[n.trials] <- pop$strain.info$environment[recipient]
				}
				
				# Recombination may only be possible if the donor
				# and the recipient are in the same environment
				if (pop$strain.info$environment[recipient]==pop$strain.info$environment[donor]) {
					
					# Check the local Hamming distance of recipient and donor
					# sequences at the gene.
					recipient.allele <- pop$alleles[recipient, gene.index]
					donor.allele <- pop$alleles[donor, gene.index]
					hamming.dist <- pop$allele.distances[[gene.index]][recipient.allele, donor.allele]
					divergence <- hamming.dist / pop$gene.length
					
					if (add.bookkeeping) {
						allele.dist[n.trials] <- divergence
					}
					
					# Check the acceptance of the recombination
					acceptance.probability <- 10^(-recombination.acceptance.par*divergence)
					if (runif(n=1) < acceptance.probability) {
						# Recombination trial accepted
						pop$alleles[recipient,gene.index] <- pop$alleles[donor,gene.index]
						
						# Record recombination statistics
						pop$event.counts$core.rec <- pop$event.counts$core.rec + 1
						pop$event.counts$rec.snp.counts[hamming.dist+1] <- pop$event.counts$rec.snp.counts[hamming.dist+1] + 1
						
						if (add.bookkeeping) {
							accepted[n.trials] <- TRUE
						}	
					}
				}
			}
		}
	}
	
	if (add.bookkeeping) {
		donor.type <- donor.type[1:n.trials]
		recipient.type <- recipient.type[1:n.trials]
		donor.env <- donor.env[1:n.trials]
		recipient.env <- recipient.env[1:n.trials]
		allele.dist <- allele.dist[1:n.trials]
		accepted <- accepted[1:n.trials]
		trials <- data.frame(donor.type=donor.type, recipient.type=recipient.type, donor.env=donor.env, recipient.env=recipient.env, allele.dist=allele.dist, accepted=accepted)
		pop$trials <- trials
	}
	
	return(pop)
}



clean.allele.descriptions <- function(pop, gen) {
	
	n.strains.in.population <- sum(pop$n.strain.types)
	n.genes <- ncol(pop$alleles)
	
	for (gene.index in 1:n.genes) {
		
		# Count the numbers of alleles
		n.alleles.with.labels <- length(pop$allele.snps[[gene.index]])
		if (n.alleles.with.labels>pop$max.n.alleles) {stop('Max alleles exceeded')}
		counts <- table(pop$alleles[,gene.index])
		to.keep <- as.numeric(names(counts))
		if (any(to.keep>n.alleles.with.labels)){stop('Incorrect alleles')}
		to.remove <- setdiff(seq(1,n.alleles.with.labels), to.keep)
		
		# Count the numbers of SNPs
		snp.counts <- rep(0, pop$gene.length)
		for (allele.index in 1:n.alleles.with.labels) {
			snp.counts[pop$allele.snps[[gene.index]][[allele.index]]] <- snp.counts[pop$allele.snps[[gene.index]][[allele.index]]] + counts[allele.index]
		}
		
		# Remove alleles that are not present in any strain
		if (length(to.remove)>0) {
			
			# Remove allele descriptions
			pop$allele.snps[[gene.index]] <- pop$allele.snps[[gene.index]][-to.remove]
			
			# Mapping of old labels to new ones
			label.mapping <- rep(0, n.alleles.with.labels)
			label.mapping[to.keep] <- seq(1, length(to.keep))
			
			# Re-label alleles
			pop$alleles[,gene.index] <- label.mapping[pop$alleles[,gene.index]]
			
			# Re-order allele distances
			pop$allele.distances[[gene.index]][seq(1,length(to.keep)),seq(1,length(to.keep))] <- pop$allele.distances[[gene.index]][to.keep,to.keep]
			
			to.discard <- seq(length(to.keep)+1, n.alleles.with.labels)
			pop$allele.distances[[gene.index]][to.discard, 1:n.alleles.with.labels] <- 0
			pop$allele.distances[[gene.index]][1:n.alleles.with.labels, to.discard] <- 0
			
			# Update the number of alleles for subsequent use
			n.alleles.with.labels <- length(pop$allele.snps[[gene.index]])
		}
		
		
		# Change SNP encoding for SNPs present in more than half of strains
		snps.to.update <- which(snp.counts > (n.strains.in.population/2))
		if (length(snps.to.update)>0) {
			# SNP is removed from all alleles having the SNP 
			# and added to all alleles without the SNP.
			
			for (snp.index in snps.to.update) {
				# Check which alleles have the SNP
				has.the.snp <- unlist(lapply(pop$allele.snps[[gene.index]], function(x) {
					any(x==snp.index)
				}))
				labels.with.snp <- which(has.the.snp)
				labels.without.snp <- setdiff(1:n.alleles.with.labels, labels.with.snp)
				
				# Add SNP to alleles without the SNP
				if (length(labels.without.snp)>0) {
					pop$allele.snps[[gene.index]][labels.without.snp] <- lapply(pop$allele.snps[[gene.index]][labels.without.snp], function(x) {
						sort(union(x, snp.index))
					})
				}
				
				# Remove SNP from alleles with the SNP
				if (length(labels.with.snp)>0) {
					pop$allele.snps[[gene.index]][labels.with.snp] <- lapply(pop$allele.snps[[gene.index]][labels.with.snp], function(x) {
						sort(setdiff(x, snp.index))
					})
				}
				
			}
		}
	}
	return(pop)
}





check.population.validity <- function(pop, generation.index, save.extension) {
	
	n.genes <- ncol(pop$alleles)
	is.ok <- TRUE
	
	# Check that SNPs in alleles are ordered
	for (gene.index in 1:n.genes) {
		n.alleles <- length(pop$allele.snps[[gene.index]])
		for (allele.index in 1:n.alleles) {
			sorted.snps <- sort(unique(pop$allele.snps[[gene.index]][[allele.index]]))
			if (!all(pop$allele.snps[[gene.index]][[allele.index]]==sorted.snps)) {
				is.ok <- FALSE
			}
		}
	}
	
	# Check that all alleles in a gene are different
	for (gene.index in 1:n.genes) {
		n.alleles <- length(pop$allele.snps[[gene.index]])
		if (n.alleles>1) {
			for (i in seq(1,n.alleles-1)) {
				for (j in seq(i+1, n.alleles)) {
					snps.i <- pop$allele.snps[[gene.index]][[i]]
					snps.j <- pop$allele.snps[[gene.index]][[j]]
					if (setequal(snps.i,snps.j)) {
						is.ok <- FALSE
					}
				}
			}
		}
	}
	
	# Check allele distances
	if (length(pop$allele.distances) != n.genes) {
		is.ok <- FALSE
	}
	for (gene.index in 1:n.genes) {
		n.alleles <- length(pop$allele.snps[[gene.index]])
		for (i in seq(1,n.alleles-1)) {
			for (j in seq(i+1, n.alleles)) {
				snps.i <- pop$allele.snps[[gene.index]][[i]]
				snps.j <- pop$allele.snps[[gene.index]][[j]]
				num.differing.snps <- length(union(snps.i,snps.j)) - length(intersect(snps.i, snps.j))
				dist.now <- num.differing.snps
				if (abs(pop$allele.distances[[gene.index]][i,j]-dist.now)>1e-10) {
					is.ok <- FALSE
				}
			}
		}
		
		# Other elements in allele.distances
		if (any(dim(pop$allele.distances[[gene.index]])!=rep(pop$max.n.alleles,2))) {
			is.ok <- FALSE
		}
		zero.inds <- seq(n.alleles+1,pop$max.n.alleles)
		if (length(zero.inds)>0) {
			zero.rows <- as.vector(pop$allele.distances[[gene.index]][zero.inds,])
			zero.cols <- as.vector(pop$allele.distances[[gene.index]][,zero.inds])
			all.zero.elements <- c(zero.rows, zero.cols)
			if (!all(all.zero.elements==0)) {
				is.ok <- FALSE
			}
		}
	}

	if (!is.ok) {
		save(generation.index, file=paste('error_in_gen', generation.index, save.extension, '.RData', sep=''))
		stop('Illegal population state')
	}
}


check.allele.snps <- function(pop) {
	
	n.genes <- ncol(pop$alleles)
	is.ok <- TRUE
	for (gene.index in 1:n.genes) {
		n.alleles <- length(pop$allele.snps[[gene.index]])
		if (n.alleles>1) {
			for (i in seq(1,n.alleles-1)) {
				for (j in seq(i+1, n.alleles)) {
					if (setequal(pop$allele.snps[[gene.index]][[i]], pop$allele.snps[[gene.index]][[j]])) {is.ok <- FALSE}
				}
			}
		}
	}
	return(is.ok)
}



initial.counts <- function(niche.overlap, n.strain.types) {
	counts <- matrix(0, nrow=2, ncol=3)
	rownames(counts) <- c('A','B')
	colnames(counts) <- c('A','AB','B')
	overlap.proportion <- niche.overlap / sum(n.strain.types)
	for (env.now in c('A','B')) {
		counts[env.now,'AB'] <- n.strain.types[env.now] * overlap.proportion
		counts[env.now,env.now] <- n.strain.types[env.now] * (1-overlap.proportion)
	}
	
	# Check that all counts are integers and that the 
	# correct number of strains of each type is produced.
	counts <- round(counts)
	for (env.now in c('A','B')) {
		rounding.error <- sum(counts[env.now,]) - n.strain.types[env.now]
		counts[env.now,env.now] <- counts[env.now,env.now] - rounding.error
		# Remove the extra strains from A population.
	}
	return(counts)
}