rm(list=ls())
source('analytical_functions.R')
require(gdata)

method.to.use <- 'combined'  # 'old', 'newton' or 'combined'

# PARAMETERS:

# How many strains of type A/B.
n.mult <- 1
n.strain.types <- c('A'=9500, 'B'=500)
n.strains.total <- sum(n.strain.types)

# Number of strains in the shared 'ab' environment.
niche.overlap <- 4000

# Determines the amount of migration between environments
migration <- 0.5

# Mean number of mutations per gene per generation
mutation.rate <- 2.4

# Number of genes
n.genes <- 40

# The length of a gene
gene.length <- 500

# Mean number of recombinations per gene per generation
recombination.rate <- 18

# Parameter to determine whether a recombination is accepted.
# The acceptance probability is 10^(-Ax), where x the Hamming 
# distance between the old and new alleles.
recombination.acceptance.par <- 18

# Given distances
groups <- c('A.in.A', 'A.in.AB', 'B.in.AB', 'B.in.B')
given.distances <- matrix(0,nrow=4, ncol=4)
colnames(given.distances) <- groups
rownames(given.distances) <- groups
within.A <- 0.01
within.B <- 0.01
between.A.B <- 0.017
given.distances[,] <- c(rep(c(within.A,within.A,between.A.B, between.A.B), 2), rep(c(between.A.B, between.A.B, within.B, within.B), 2))

# Estimate r.per.m corresponding to the given settings
r.per.m <- compute.r.per.m(given.distances, niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types)

print(r.per.m)



#analysis.type <- 'find.equilibrium.distances'
analysis.type <- 'find.equilibrium.parameters'



if (analysis.type=='find.equilibrium.distances') {
	# Find the stationary distance distribution, which
	# follows from the given parameter values.
	
	init.dist <- given.distances
	
	if (method.to.use=='old') {
		# OLD ROBUST OPTIMIZATION
		iter <- 0
		max.iter <- 40000
		relative.diff <- matrix(1, nrow=4, ncol=4)
		dist.now <- init.dist
		while((iter < max.iter) & !all(abs(relative.diff) < 1e-6)) {
			
			iter <- iter + 1
			dist.old <- dist.now
			
			res <- get.distances.next.generation(init.dist=dist.old, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
			
			dist.now <- res$new.dist
			dist.changes <- res$dist.changes
			#print(dist.now)
			
			dist.diff <- dist.now - dist.old
			
			relative.diff <- abs(dist.diff / dist.old)
		}
	}
	
	
	if (method.to.use=='newton') {
		# FAST NEW NEWTON'S OPTIMIZATION
		dist.vector <- upperTriangle(init.dist, diag=TRUE)
		
		res <- optim(par=dist.vector, fn=compute.target.value, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types, method='BFGS', control=list(ndeps=rep(1e-9,10), abstol=1e-8, trace=5, maxit=300))
		
		learned.dist <- init.dist
		upperTriangle(learned.dist, diag=TRUE) <- res$par
		aux <- upperTriangle(learned.dist, diag=FALSE)
		learned.dist <- t(learned.dist)
		upperTriangle(learned.dist, diag=FALSE) <- aux
	}
	
	
	# COMBINED OPTIMIZATION
	if (method.to.use=='combined') {
		res <- optimize.analytical.model(niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
	}
}




if (analysis.type == 'find.equilibrium.parameters') {
	# Find parameter value (niche.overlap) that produces the given 
	# within and between distances as an equilibrium state.
	
	overlap.grid <- c(1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000)
	
	# Run 50 iterations with each parameter combination
	max.iter <- 100
	dist.now <- given.distances
	dist.diff <- array(NA, dim=c(4,4,length(overlap.grid)))
	dim.names <- list()
	dim.names[[1]] <- groups
	dim.names[[2]] <- groups
	dim.names[[3]] <- overlap.grid
	dimnames(dist.diff) <- dim.names
	
	for (niche.overlap in overlap.grid) {
		for (iter in 1:max.iter) {
			
			res <- get.distances.next.generation(init.dist=dist.now, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
			
			dist.now <- res$new.dist
		}
		res <- get.distances.next.generation(init.dist=dist.now, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
		
		dist.diff[,,as.character(niche.overlap)] <- res$new.dist - dist.now
		
		
	}

	
	# Compute the optimal overlap
	between.dist.change <- apply(dist.diff,3,function(x){x[1,4]})
	if (all(between.dist.change > 0)) {
		# Populations are always diverging
		overlap.estimate <- NA
	
	} else if (all(between.dist.change < 0)) {
		# Populations are always converging
		overlap.estimate <- NA
	
	} else {
		# With small overlaps the populations are diverging, but
		# with large overlaps they are converging
		largest.diverging.overlap <- max(which(between.dist.change > 0))
		divergence.rate <- between.dist.change[largest.diverging.overlap]
		convergence.rate <- between.dist.change[largest.diverging.overlap + 1]
		
		# Estimate the stationary point using linear interpolation between
		# last positive and first negative overlaps.
		overlaps <- overlap.grid / n.strains.total
		diverging.overlap <- overlaps[largest.diverging.overlap]
		converging.overlap <- overlaps[largest.diverging.overlap + 1]
		
		overlap.estimate <- diverging.overlap + (converging.overlap - diverging.overlap) * divergence.rate / (divergence.rate + abs(convergence.rate))
	}
	
	print(paste('Estimated overlap proportion=',signif(overlap.estimate,3), sep=''))
	
}

