optimize.analytical.model <- function(niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	#INITIAL DISTANCES
	groups <- c('A.in.A', 'A.in.AB', 'B.in.AB', 'B.in.B')
	init.dist <- matrix(0, nrow=4, ncol=4)
	colnames(init.dist) <- groups
	rownames(init.dist) <- groups
	
	# FIRST RUN THE FAST NEWTON METHOD
	dist.vector <- upperTriangle(init.dist, diag=TRUE)
	
	res <- optim(par=dist.vector, fn=compute.target.value, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types, method='BFGS', control=list(ndeps=rep(1e-10,10), abstol=1e-9, maxit=300))
	
	converged <- res$convergence==0
	
	if (converged) {
		
		# Update dist.now to the estimate from Newton's method
		dist.now <- init.dist
		upperTriangle(dist.now, diag=TRUE) <- res$par
		aux <- upperTriangle(dist.now, diag=FALSE)
		dist.now <- t(dist.now)
		upperTriangle(dist.now, diag=FALSE) <- aux
		
		# CONFIRM CONVERGENCE WITH THE OLD AND ROBUST METHOD.
		# Start from the fast estimate from Newton's method.
		max.iter <- 10000
		iter <- 0
		relative.diff <- matrix(1, nrow=4, ncol=4)
		while((iter < max.iter) & !all(abs(relative.diff) < 1e-6)){
			iter <- iter + 1
			dist.old <- dist.now
			
			res <- get.distances.next.generation(init.dist=dist.old, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
			
			dist.now <- res$new.dist
			dist.changes <- res$dist.changes
			
			dist.diff <- dist.now - dist.old
			
			relative.diff <- abs(dist.diff / dist.old)
		}
		converged <- !(iter==max.iter)
	} else {
		dist.now <- NA
		iter <- 0
	}
	
	return(list(dist.learned=dist.now, converged=converged, iter=iter))
	
}



compute.constrained.target.value <- function(dist.vector.free, dist.vector.constrained, niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	# dist.vector.constrained contains the constrained 
	# distances (within and between environments):
	# init.dist[c(1,13,16)]
	#
	# dist.vector.free contains unconstrained distance elements:
	# init.dist[c(5,6,9,10,11,14,15)]
	
	full.vector <- c(dist.vector.free, dist.vector.constrained)
	element.inds <- c(5,6,9,10,11,14,15, 1,13,16)
	full.ordering <- order(element.inds)
	full.vector <- full.vector[full.ordering]
	
	target.value <- compute.target.value(dist.vector=full.vector, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
	
	return(target.value)
}



compute.target.value <- function(dist.vector, niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types) {
	# Computes the value that must be minimized to solve all distances.
	# 
	# dist.vector is obtained from init.dist by upperTriangle(x, diag=T)
	
	require(gdata)
	groups <- c('A.in.A','A.in.AB','B.in.AB','B.in.B')
	dist.old <- matrix(0, nrow=4, ncol=4)
	colnames(dist.old) <- groups
	rownames(dist.old) <- groups
	upperTriangle(dist.old, diag=TRUE) <- dist.vector
	aux <- upperTriangle(dist.old, diag=F)
	dist.old <- t(dist.old)
	upperTriangle(dist.old) <- aux
	
	res <- get.distances.next.generation(init.dist=dist.old, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
	
	dist.new <- res$new.dist
	dist.diff <- dist.new - dist.old
	target.val <- sqrt(sum(upperTriangle(dist.diff, diag=T)^2))
	
	return(target.val)
}


return.independent.elements <- function(dist.matrix) {
	independent.elements <- lowerTriangle(dist.matrix, diag=T)[1:6]
	return(independent.elements)
}

fix.distance.matrix <- function(independent.elements) {
	groups <- c('A.in.A','A.in.AB','B.in.AB','B.in.B')
	dist.mat <- matrix(0, nrow=4, ncol=4)
	colnames(dist.mat) <- groups
	rownames(dist.mat) <- groups
	
	dist.mat[1,1:4] <- independent.elements[1:4]
	dist.mat[2,2:3] <- independent.elements[5:6]
	dist.mat[2,4] <- independent.elements[3]
	dist.mat[3,3:4] <- independent.elements[c(5,2)]
	dist.mat[4,4] <- independent.elements[1]
	
	aux <- upperTriangle(dist.mat)
	dist.mat <- t(dist.mat)
	upperTriangle(dist.mat) <- aux
	return(dist.mat)
}


get.distances.next.generation <- function(init.dist, niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	counts <- compute.environment.counts(niche.overlap=niche.overlap, n.strain.types=n.strain.types)
	
	dist.changes <- list()
	group.names <- c('A.in.A','A.in.AB','B.in.AB','B.in.B')
	for (op in c('mutation','recombination','sampling')) {
		dist.changes[[op]] <- matrix(0, nrow=4, ncol=4)
		colnames(dist.changes[[op]]) <- group.names
		rownames(dist.changes[[op]]) <- group.names
	}
	
	
	# SAMPLING (within A.in.A and B.in.B)
	for (env.now in c('A','B')) {
		
		g.name <- paste(env.now, '.in.', env.now, sep='')
		dist.changes[['sampling']][g.name,g.name] <- compute.effect.of.sampling.on.A.or.B.environment.within.distances(init.dist=init.dist, env.now=env.now, counts=counts, migration=migration)
		
	}
	
	
	# SAMPLING (within A.in.AB and B.in.AB)
	for (type.now in c('A','B')) {
		
		g.name <- paste(type.now, '.in.AB', sep='')
		dist.changes[['sampling']][g.name,g.name] <- compute.effect.of.sampling.on.AB.within.distances(init.dist=init.dist, type.now=type.now, counts=counts, migration=migration)
		
	}
	
	
	# SAMPLING (for between distances)
	type <- c('A','A','B','B')
	environment <- c('A','AB','B','AB')
	combinations <- data.frame(type=type, environment=environment, stringsAsFactors=F)
	n.combinations <- nrow(combinations)
	for (i in seq(1,n.combinations-1)) {
		
		type1 <- combinations[i,'type']
		env1 <- combinations[i,'environment']
		g1.name <- paste(type1, '.in.', env1, sep='')
		
		for (j in seq(i+1, n.combinations)) {
			
			type2 <- combinations[j,'type']
			env2 <- combinations[j,'environment']
			g2.name <- paste(type2, '.in.', env2, sep='')
			
			dist.changes[['sampling']][g1.name,g2.name] <- compute.effect.of.sampling.for.between.distances(init.dist=init.dist, types=c(type1,type2), environments=c(env1,env2), counts=counts, migration=migration)
			
			dist.changes[['sampling']][g2.name,g1.name] <- dist.changes[['sampling']][g1.name,g2.name]
			
		}
	}
	
	
	# MUTATION
	n.strains.in.population <- sum(n.strain.types)
	dist.changes[['mutation']][,] <- 2 * mutation.rate / n.strains.in.population / gene.length
	
	
	
	# RECOMBINATION (easy cases)
	dist.changes[['recombination']]['A.in.A','A.in.A'] <- 0
	dist.changes[['recombination']]['B.in.B','B.in.B'] <- 0
	dist.changes[['recombination']]['A.in.A','B.in.B'] <- 0
	
	
	# RECOMBINATION (A.in.AB and B.in.AB within distances)
	for (type.now in c('A','B')) {
		
		g.name <- paste(type.now, '.in.AB', sep='')
		dist.changes[['recombination']][g.name, g.name] <- compute.effect.of.recombination.on.AB.within.distances(init.dist=init.dist, type.now=type.now, counts=counts, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
		
	}
	
	
	# RECOMBINATION (A.in.AB and B.in.AB between distances)
	dist.changes[['recombination']]['A.in.AB', 'B.in.AB'] <- compute.effect.of.recombination.on.AB.between.distances(init.dist=init.dist, counts=counts, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)

	dist.changes[['recombination']]['B.in.AB','A.in.AB'] <- 
		dist.changes[['recombination']]['A.in.AB','B.in.AB']


	# RECOMBINATION (A.in.A and A.in.AB between distances or
	# B.in.B and B.in.AB between distances)
	for (type.now in c('A','B')) {
		
		g1.name <- paste(type.now, '.in.', type.now, sep='')
		g2.name <- paste(type.now, '.in.AB', sep='')
		
		dist.changes[['recombination']][g1.name,g2.name] <- compute.effect.of.recombination.on.same.type.between.environment.distances(type.now=type.now, init.dist=init.dist, counts=counts, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
		
		dist.changes[['recombination']][g2.name,g1.name] <- dist.changes[['recombination']][g1.name,g2.name]
		
	}
	
	
	# RECOMBINATION (A.in.A and B.in.AB between distances or
	# B.in.B and A.in.AB between distances)
	for (type.now in c('A','B')) {
		
		g1.name <- paste(type.now, '.in.', type.now, sep='')
		other.type <- setdiff(c('A','B'), type.now)
		
		g2.name <- paste(other.type, '.in.AB', sep='')
		
		dist.changes[['recombination']][g1.name,g2.name] <- compute.effect.of.recombination.on.different.type.between.environment.distances(type.now=type.now, init.dist=init.dist, counts=counts, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
		
		dist.changes[['recombination']][g2.name,g1.name] <- dist.changes[['recombination']][g1.name,g2.name]
	
	}
	
	total.changes <-  dist.changes[['mutation']] + dist.changes[['recombination']] + dist.changes[['sampling']]
	
	new.dist <- init.dist + total.changes
	
	return(list(new.dist=new.dist, dist.changes=dist.changes))
}



compute.effect.of.sampling.on.A.or.B.environment.within.distances <- function(init.dist, env.now, counts, migration) {
	
	# How much within cluster distances (in A and B) get smaller:
	g.name <- paste(env.now, '.in.', env.now, sep='')
	overlap.name <- paste(env.now, '.in.AB', sep='')
	
	dist.env <- init.dist[g.name,g.name]
	dist.overlap <- init.dist[overlap.name,overlap.name]
	dist.between <- init.dist[g.name,overlap.name]
	
	prob.of.AB <- migration * counts[env.now,'AB'] / (migration * counts[env.now,'AB'] + counts[env.now,env.now])
	num.from.AB <- counts[env.now,env.now] * prob.of.AB
	num.from.env <- counts[env.now,env.now] - num.from.AB
	
	new.total.dist <- choose(num.from.env,2) * dist.env
	
	# Improvement:
	new.total.dist <- new.total.dist - dist.env * counts[env.now,env.now] / 2 * (num.from.env / counts[env.now,env.now])^2
	
	new.total.dist <- new.total.dist + num.from.env * num.from.AB * dist.between
	
	new.total.dist <- new.total.dist + choose(num.from.AB,2) * dist.overlap
	
	# Improvement:
	new.total.dist <- new.total.dist - dist.overlap * counts[env.now,'AB'] / 2 * (num.from.AB / counts[env.now,'AB'])^2
	
	expected.change <- new.total.dist / choose(counts[env.now,env.now],2) - init.dist[g.name,g.name]
	
	return(expected.change)
	
}


compute.effect.of.sampling.on.AB.within.distances <- function(init.dist, type.now, counts, migration) {
	
	overlap.name <- paste(type.now, '.in.AB', sep='')
	home.env.name <- paste(type.now, '.in.', type.now, sep='')
	
	dist.home <- init.dist[home.env.name,home.env.name]
	dist.AB <- init.dist[overlap.name,overlap.name]
	dist.between <- init.dist[overlap.name, home.env.name]
	
	prob.of.home <- migration * counts[type.now,type.now] / (migration * counts[type.now,type.now] + counts[type.now,'AB'])
	
	num.from.home <- counts[type.now, 'AB'] * prob.of.home
	num.from.AB <- counts[type.now,'AB'] - num.from.home
	
	new.total.dist <- choose(num.from.AB,2) * dist.AB
	
	# Improvement:
	new.total.dist <- new.total.dist - dist.home * counts[type.now,type.now] / 2 *
		(num.from.home / counts[type.now,type.now])^2
	
	new.total.dist <- new.total.dist + num.from.home * num.from.AB * dist.between
	
	# Improvement:
	new.total.dist <- new.total.dist - dist.AB * counts[type.now, 'AB'] / 2 *
		(num.from.AB / counts[type.now, 'AB'])^2
	
	new.total.dist <- new.total.dist + choose(num.from.home,2) * dist.home
	
	expected.change <- new.total.dist / choose(counts[type.now,'AB'],2) - init.dist[overlap.name,overlap.name]
	
	return(expected.change)
}


compute.effect.of.sampling.for.between.distances <- function(init.dist, types, environments, counts, migration) {
	
	ancestors <- list()
	for (i in 1:2) {
		ancestors[[i]] <- return.sampled.ancestor.group.sizes(counts=counts, migration=migration, type.now=types[i], env.now=environments[i])
	}
	
	total.dist <- 0
	for (anc1 in names(ancestors[[1]])) {
		for (anc2 in names(ancestors[[2]])) {
			if (anc1==anc2) {
				# Both groups contain strains from the same ancestral
				# group. Decreased distances due to sampling of the same
				# individuals in the different groups must be accounted for.
				split.res <- strsplit(anc1,'[.]')[[1]]
				env.now <- split.res[3]
				type.now <- split.res[1]
				correction.term <- (1 - 1 / counts[type.now, env.now])
			} else {
				correction.term <- 1
			}
			total.dist <- total.dist + ancestors[[1]][anc1] * ancestors[[2]][anc2] * init.dist[anc1,anc2] * correction.term
		}
	}

	total.n.pairs <- prod(unlist(lapply(ancestors,sum)))
	new.avg.dist <- total.dist / total.n.pairs
	
	g1.name <- paste(types[1], '.in.', environments[1], sep='')
	g2.name <- paste(types[2], '.in.', environments[2], sep='')
	expected.change <- new.avg.dist - init.dist[g1.name, g2.name]
	
	return(expected.change)
}


return.sampled.ancestor.group.sizes <- function(counts, migration, type.now, env.now) {
		
	own.name <- paste(type.now, '.in.', env.now, sep='')
	if (env.now != 'AB') {
		other.env <- 'AB'
	} else {
		other.env <- type.now
	}
	other.name <- paste(type.now, '.in.', other.env, sep='')
	
	ancestor.names <- c(own.name, other.name)
	ancestor.sizes <- rep(0,2)
	names(ancestor.sizes) <- ancestor.names
	
	if (env.now != 'AB') {
		prob.of.AB <- migration * counts[env.now,'AB'] / (migration * counts[env.now,'AB'] + counts[type.now,type.now])
		num.from.AB <- counts[env.now,env.now] * prob.of.AB
		num.from.env <- counts[env.now,env.now] - num.from.AB
		
		ancestor.sizes[own.name] <- num.from.env
		ancestor.sizes[other.name] <- num.from.AB	
	
	} else {
		
		prob.of.home <- migration * counts[type.now,type.now] / (migration * counts[type.now,type.now] + counts[type.now,'AB'])
		
		num.from.home <- counts[type.now, 'AB'] * prob.of.home
		num.from.AB <- counts[type.now,'AB'] - num.from.home
		
		ancestor.sizes[own.name] <- num.from.AB
		ancestor.sizes[other.name] <- num.from.home
	}
	
	return(ancestor.sizes)
}




compute.effect.of.recombination.on.AB.within.distances <- function(init.dist, type.now, counts, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	n.strains.in.population <- sum(n.strain.types)
	
	type.other <- setdiff(c('A','B'), type.now)
	g.name <- paste(type.now, '.in.AB', sep='')
	g.other.name <- paste(type.other, '.in.AB', sep='')
	n.other.in.AB <- counts[type.other, 'AB']
	
	expected.change <- n.other.in.AB*recombination.rate /
		n.strains.in.population / (n.strains.in.population-1) * 2 *
		10^(-recombination.acceptance.par *	init.dist[g.name,g.other.name]) * 
		(init.dist[g.name,g.other.name] -  init.dist[g.name,g.name])
	
	return(expected.change)
	
}


compute.effect.of.recombination.on.AB.between.distances <- function(init.dist, counts, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	n.strains.in.population <- sum(n.strain.types)
	
	Z <- recombination.rate / n.strains.in.population /
		(n.strains.in.population-1) * 
		10^(-recombination.acceptance.par *	init.dist['A.in.AB','B.in.AB'])
	
	expected.decrease <- 
		counts['B','AB'] * (init.dist['A.in.AB','B.in.AB'] - 
		init.dist['B.in.AB','B.in.AB']) + 
		counts['A','AB'] * (init.dist['A.in.AB','B.in.AB'] -
		init.dist['A.in.AB','A.in.AB'])
	
	expected.change <- -1*expected.decrease * Z
	
	return(expected.change)
	
}


compute.effect.of.recombination.on.same.type.between.environment.distances <- function(type.now, init.dist, counts, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	n.strains.in.population <- sum(n.strain.types)
	
	other.type <- setdiff(c('A','B'), type.now)
	
	Z <- counts[other.type, 'AB'] * recombination.rate / 
		n.strains.in.population / (n.strains.in.population - 1) *
		10^(-recombination.acceptance.par * init.dist['A.in.AB','B.in.AB'])
	
	g0.name <- paste(type.now, '.in.', type.now, sep='')
	g1.name <- paste(type.now, '.in.AB', sep='')
	g2.name <- paste(other.type, '.in.AB', sep='')
	expected.change <- Z * (init.dist[g0.name, g2.name] - init.dist[g0.name, g1.name])
	
	return(expected.change)
}


compute.effect.of.recombination.on.different.type.between.environment.distances <- function(type.now, init.dist, counts, recombination.rate, recombination.acceptance.par, n.strain.types) {
	
	n.strains.in.population <- sum(n.strain.types)
	
	other.type <- setdiff(c('A','B'), type.now)
	
	Z <- counts[type.now, 'AB'] * recombination.rate /
		n.strains.in.population / (n.strains.in.population - 1) *
		10^(-recombination.acceptance.par * init.dist['A.in.AB', 'B.in.AB'])
	
	g0.name <- paste(type.now, '.in.', type.now, sep='')
	g1.name <- paste(type.now, '.in.AB', sep='')
	g2.name <- paste(other.type, '.in.AB', sep='')
	expected.change <- Z * (init.dist[g0.name,g1.name] - init.dist[g0.name,g2.name])
	
	return(expected.change)
}





learn.the.shape.of.the.mode <- function(niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types, max.dist, dist.interval) {
	
	# FORMULATE THE INITIAL STATE WITH FULL SEPARATION
		
	cluster.diameter <- 2*mutation.rate/gene.length
	
	#INITIAL DISTANCES
	groups <- c('A.in.A', 'A.in.AB', 'B.in.AB', 'B.in.B')
	init.dist <- matrix(0, nrow=4, ncol=4)
	colnames(init.dist) <- groups
	rownames(init.dist) <- groups
	init.dist[c(1,2,5,6,11,12,15,16)] <- cluster.diameter
	
	fixed.between.dist.values <- seq(dist.interval, max.dist, by=dist.interval)
	between.dist.change <- rep(NA, length(fixed.between.dist.values))
	names(between.dist.change) <- fixed.between.dist.values
	for (fixed.between.dist in fixed.between.dist.values) {
		# FORCE INITIAL BETWEEN DISTANCE TO A GIVEN VALUE
		dist.start <- init.dist
		zeros <- which(dist.start == 0)
		dist.start[zeros] <- fixed.between.dist
		
		# RUN 10 GENERATIONS
		dist.now <- dist.start
		for (iter in 1:50) {
			
			res <- get.distances.next.generation(init.dist=dist.now, niche.overlap=niche.overlap, migration=migration, mutation.rate=mutation.rate, n.genes=n.genes, gene.length=gene.length, recombination.rate=recombination.rate, recombination.acceptance.par=recombination.acceptance.par, n.strain.types=n.strain.types)
			
			dist.now <- res$new.dist
		
		}
		
		between.dist.change[as.character(fixed.between.dist)] <- dist.now['A.in.A','B.in.B'] - fixed.between.dist
	}
	
	return(between.dist.change)
}




compute.environment.counts <- function(niche.overlap, n.strain.types) {
	
	# COUNTS (expected numbers of strains in different environments)
	counts <- matrix(0, nrow=2, ncol=3)
	rownames(counts) <- c('A','B')
	colnames(counts) <- c('A','AB','B')
	
	overlap.proportion <- niche.overlap / sum(n.strain.types)
	
	for (env.now in c('A','B')) {
		counts[env.now,'AB'] <- n.strain.types[env.now] * overlap.proportion
		counts[env.now,env.now] <- n.strain.types[env.now] * (1-overlap.proportion)
	}
	# Check that all counts are integers and that the 
	# correct number of strains of each type is produced.
	counts <- round(counts)
	for (env.now in c('A','B')) {
		rounding.error <- sum(counts[env.now,]) - n.strain.types[env.now]
		counts[env.now,env.now] <- counts[env.now,env.now] - rounding.error
		# Remove the extra strains from A population.
	}
	
	return(counts)
}



compute.r.per.m <- function(init.dist, niche.overlap, migration, mutation.rate, n.genes, gene.length, recombination.rate, recombination.acceptance.par, n.strain.types) {
	#print('aa')
	
	n.strains.in.population <- sum(n.strain.types)
	counts <- compute.environment.counts(niche.overlap=niche.overlap, n.strain.types=n.strain.types)
	
	n.mut.from.rec <- init.dist
	n.mut.from.rec[] <- 0
	
	# Recombinations within different sub-populations
	for (type.now in c('A','B')) {
		
		for (env.now in c(type.now, 'AB')) {
			
			name.now <- paste(type.now, '.in.', env.now, sep='')
			
			# Number of actual recombinations in one gene
			n.rec <- counts[type.now, env.now] * (counts[type.now, env.now] - 1) *
				recombination.rate / n.strains.in.population /
				(n.strains.in.population-1) * 
				10^(-recombination.acceptance.par *	init.dist[name.now, name.now])
			
			# Each recombination introduces this many mutations
			mut.per.rec <- init.dist[name.now, name.now] * gene.length
			
			n.mut.from.rec[name.now, name.now] <- n.rec * mut.per.rec
		}
	}
	
	# Recombinations from A.in.AB into B.in.AB
	n.mut.from.rec['A.in.AB','B.in.AB'] <- counts['A','AB'] * counts['B','AB'] / 
		n.strains.in.population / (n.strains.in.population-1) *
		recombination.rate * 
		10^(-recombination.acceptance.par * init.dist['A.in.AB', 'B.in.AB'])
	
	# Due to symmetry:
	n.mut.from.rec['B.in.AB','A.in.AB'] <- n.mut.from.rec['A.in.AB','B.in.AB']
	
	total.mut.from.rec <- sum(n.mut.from.rec)
	r.per.m <- total.mut.from.rec / mutation.rate
	
	return(r.per.m)
}
