simulate.genotypes <- function(n.patients, n.snps) {
    #
    # A function for simulating independent SNP genotypes.
    #
    # Inputs:
    #   n.patients: number of patients
    #   n.snps: number of genotypes
    #
    # Outputs:
    #   genotypes (unnormalized)

	mafs <- runif(n=n.snps, min=0.05, 0.5)
	prob.minor.homozygote <- mafs^2
	prob.heterozygote <- 2* mafs * (1-mafs)
	prob.major.homozygote <- (1-mafs)^2
	
	table <- matrix(runif(n=n.patients*n.snps, min=0, max=1),nrow=n.patients, ncol=n.snps)
	
	genotypes <- matrix(2, nrow=n.patients, ncol=n.snps)
	heterozygote <- t(t(table)>prob.minor.homozygote & t(table)<prob.minor.homozygote+prob.heterozygote)
	genotypes[heterozygote]=1
	major.homozygote <- t(t(table)>prob.minor.homozygote+prob.heterozygote)
	genotypes[major.homozygote]=0
	
	#return(list(genotypes=genotypes, mafs=mafs))
	return(genotypes)
}


check.mcmc.result <- function(context, mcmc.output, name, plot.path = NULL, plot.title = NULL) {
	# "name" is the name of the variable
	# "context" contains the true variable values
	# mcmc.output contains the "traces" variable
	

	
	if (name=='coefMat') {
	
		# Compare the coefficient matrices (obtainable from Psi and Gamma)
		
		res <- compute.coef.matrix.mcmc.estimate(mcmc.output)
		coef.matrix.mean <- res$coef.matrix.mean
		coef.matrix.std <- res$coef.matrix.std
		
		correct <- context$Psi %*% context$Gamma
		comp <- compare.matrices(correct=correct, est.mean=coef.matrix.mean, est.std=coef.matrix.std)
		
		
		plot.matrix.comparison(correct=correct, est.mean=coef.matrix.mean, plot.path = plot.path, plot.title)
	
		#if (any(names(mcmc.output$traces)=='brr.rank')) {
		#	x11()
		#	plot(1:n.iter, mcmc.output$traces$brr.rank)
		#}
				
	} else if (name=='covMat') {
	
		n.iter <- length(mcmc.output$traces[['Lambda']])
		n.pheno <- nrow(context$Lambda)
		
		cov.matrix.samples <- array(0, dim=c(n.pheno,n.pheno,n.iter))

		for (iter in 1:n.iter) {
			cov.matrix.samples[,,iter] <- tcrossprod(mcmc.output$traces$Lambda[[iter]]) + diag(mcmc.output$traces$variances[[iter]])
		}
		est.mean <- apply(cov.matrix.samples, 1:2, mean)
		est.std <- apply(cov.matrix.samples, 1:2, sd)
		correct <- tcrossprod(context$Lambda) + diag(context$variances) 
	
	
		comp <- compare.matrices(correct=correct, est.mean=est.mean, est.std=est.std)
		
		plot.matrix.comparison(correct=correct, est.mean=est.mean)
		
	} else {
		n.iter <- length(mcmc.output$traces[[name]])
		if (is.matrix(context[[name]])) {
			values <- unlist(mcmc.output$traces[[name]])
			samples <- array(values, dim=c(dim(context[[name]]), n.iter))
			
			est.mean <- apply(samples, 1:2, mean)
			est.std <- apply(samples, 1:2, sd)
			correct <- context[[name]]
			
			comp <- compare.matrices(correct=correct, est.mean=est.mean, est.std=est.std)
		} else {
			dim1 <- length(context[[name]])
				
			samples <- matrix(unlist(mcmc.output$traces[[name]]), nrow=dim1, ncol=n.iter)
	
			est.mean <- as.matrix(apply(samples,1,mean))
			est.std <- as.matrix(apply(samples,1,sd))
			correct <- as.matrix(context[[name]])
			
			comp <- compare.matrices(correct=correct, est.mean=est.mean, est.std=est.std)	
		}
	}
	return(comp)
}



compute.coef.matrix.mcmc.estimate <- function(mcmc.output) {
	n.iter <- length(mcmc.output$traces[['Gamma']])
	n.pheno <- ncol(mcmc.output$traces$Gamma[[1]])
	n.snps <- nrow(mcmc.output$traces$Psi[[1]])	

	coef.matrix.samples <- array(0, dim=c(n.snps, n.pheno, n.iter))
	
	for (i in 1:n.iter) {
		coef.matrix.samples[,,i] <- mcmc.output$traces$Psi[[i]] %*% mcmc.output$traces$Gamma[[i]]
	}
	coef.matrix.mean <- apply(coef.matrix.samples, 1:2, mean)
	coef.matrix.std <- apply(coef.matrix.samples, 1:2, sd)
	
	return(list(coef.matrix.mean=coef.matrix.mean, coef.matrix.std=coef.matrix.std))
}


compare.matrices <- function(correct, est.mean, est.std) {
	comp <- list()
	comp$mse <- mean((correct-est.mean)^2)
	lower.bound <- est.mean-2*est.std
	upper.bound <- est.mean+2*est.std
	num.in.interval <- length(which(correct>lower.bound & correct<upper.bound))
	comp$prop.in.interval <- num.in.interval / prod(dim(correct))
	comp$correct <- correct
	comp$est.mean <- est.mean
	comp$est.std <- est.std 
	return(comp)
}



plot.matrix.comparison <- function(correct, est.mean, plot.path = NULL, plot.title) {

	n.col <- ncol(correct)
	n.row <- nrow(correct)

	cols=rich.colors(30)
	if (is.null(plot.path)) {
		x11()
	} else {
		pdf(file = paste(plot.path, 'true_params', '.pdf', sep = ''))
	}
	image(x=seq(1,n.row), y=seq(1,n.col), z=correct, col=cols, axes=FALSE, xlab='', ylab='',main='correct')
	image.legend( n.row/5*4, n.col/10, zlim = range(correct), bg='white', yju=0, col=cols, at.z=range(correct), legnd=range(round(correct, digits=1)))
	if (!is.null(plot.path)) {
		dev.off()
	}	
	if (is.null(plot.path)) {
		x11()
	} else {
		pdf(file = paste(plot.path, 'estimated_params', '.pdf', sep = ''))
	}
	image(x=seq(1,n.row), y=seq(1,n.col), z=est.mean, col=cols, axes=FALSE, xlab='', ylab='',main=paste('estimated', plot.title))
	image.legend( n.row/5*4, n.col/10, zlim = range(est.mean), bg='white', yju=0, col=cols, at.z=range(est.mean), legnd=range(round(est.mean, digits=1)))
	
	if (!is.null(plot.path)) {
	dev.off()
}
}

image.legend <- function(x,y, zlim, at.z = NULL, col = heat.colors(12), legnd=NULL,
             lwd = max(3,32/length(col)), bg = NA, bty = "", ...)
  ## * kein y.i -- Benutzer soll rein ueber lwd steuern; sollte reichen.
  ## * legnd koennte interessant sein, falls Text geschrieben werden soll
  ##   (weiss mal wieder nicht, wie man aus legnd legend als option
  ##     macht)
  ## * lwd wird per default in Abh. von col gewaehlt.
{
    ## Purpose:
    ## Authors: Martin Maechler,   9 Jul 2001
    ##          Martin Schlather, 24 Jul 2001

  if (!is.null(legnd) && is.null(at.z))
      stop("at.z must be given if legnd is") ## falls legnd darf at.z
    ##                                nicht automatisch gewaehlt werden

    if(!is.numeric(zlim) || zlim[1] > zlim[2])
        stop("`zlim' must be numeric; zlim[1] <= zlim[2]")
    if(is.null(at.z)) {
        ## hier ein Versuch in Abhaengigkeit von n
        ## die Anzahl der labels zu bestimmen:
        n <- min(5, max(1,length(col)/10))
        at.z <- pretty(zlim,n=n,min.n=max(n %/% 3,1))

        ## es sieht nicht schoen aus, wenn pretty die letzte oder
        ## erste zahl weit ausserhalb des zlim legt.
        ## heuristisch nur 25%  (oder so) ueberschreitung bzgl
        ## intervalllaenge zulassen:
        tol <- diff(at.z)[1] / 4
        at.z <- at.z[(at.z>=zlim[1]-tol) & (at.z<=zlim[2]+tol)]
      }
    if(!is.numeric(at.z) || is.unsorted(at.z))
        stop("`at.z' must be numeric non-decreasing")
    n.at <- length(at.z)
    nc   <- length(col)
    if(n.at >= nc)
        stop("length(at.z) must be (much) smaller than length(col)")
    dz <- diff(zlim)
    ## The colors must run equidistantly from zlim[1] to zlim[2];
    ## col[i] is for z-interval zlim[1] + [i-1, i) * dz/nc  ; i = 1:nc
    ## i.e., an at.z[] value z0 is color i0 = floor(nc * (z0 - zlim[1])/dz)
    at.i <- floor(nc * (at.z - zlim[1])/dz )
    ## Possibly extend colors by `background' to the left and right
    bgC <- if(is.null(bg)) NA else bg
    if((xtra.l <- 1 - at.i[1]) > 0) {
        at.i <- at.i + xtra.l
        col <- c(rep(bgC, xtra.l), col)
    }
    if((xtra.r <- at.i[n.at] - nc) > 0)
        col <- c(col, rep(bgC, xtra.r))
    lgd <- character(length(col))

    ## folgende if-Anweisung ist neu:
    if (is.null(legnd)) lgd[at.i] <-format(at.z, dig = 3)
    else {
      if (length(legnd)!=length(at.z))
        stop("at.z and legnd must have the same length")
      lgd[at.i] <- legnd
    }
    if((V <- R.version)$major <= 1 && V$minor <= 3.0 && V$status == "")
{
        ## stop-gap fix around the bug that "NA" is not a valid color:
        if(is.na(bgC)) {
            lgd <- lgd[!is.na(col)]
            col <- col[!is.na(col)]
        }
    }
    legend(x,y, legend = rev(lgd), col = rev(col), y.i = lwd/16, bty = bty, lwd = lwd, bg = bg, ...)
}


rich.colors <- function(n,
                        palette="temperature",
                        alpha=1,
                        rgb=FALSE,
                        plot=FALSE)
{
  if(n <= 0)
    return(character(0))

  palette <- match.arg(palette, c("temperature","blues"))
  x <- seq(0, 1, length=n)

  if(palette == "temperature")
  {
    r <- 1 / (1+exp(20-35*x))
    g <- pmin(pmax(0,-0.8+6*x-5*x^2), 1)
    b <- dnorm(x,0.25,0.15) / max(dnorm(x,0.25,0.15))
  }
  else
  {
    r <-        0.6*x + 0.4*x^2
    g <-        1.5*x - 0.5*x^2
    b <- 0.36 + 2.4*x - 2.0*x^2
    b[x>0.4] <- 1
  }

  rgb.m <- matrix(c(r,g,b), ncol=3,
                  dimnames=list(NULL,c("red","green","blue")))
  col <- mapply(rgb, r, g, b, alpha)

  if(rgb) 
    attr(col, "rgb") <- cbind(rgb.m, alpha)
  
  if(plot)
  {
    opar <- par("fig", "plt")
    par(fig=c(0,1,0,0.7), plt=c(0.15,0.9,0.2,0.95))
    plot(NA, xlim=c(-0.01,1.01), ylim=c(-0.01,1.01), xlab="Spectrum", ylab="",
         xaxs="i", yaxs="i", axes=FALSE)
    title(ylab="Value", mgp=c(3.5,0,0))
    matlines(x, rgb.m, col=colnames(rgb.m), lty=1, lwd=3)
    matpoints(x, rgb.m, col=colnames(rgb.m), pch=16)
    axis(1, at=0:1)
    axis(2, at=0:1, las=1)
    par(fig=c(0,1,0.75,0.9), plt=c(0.08,0.97,0,1), new=TRUE)
    midpoints <- barplot(rep(1,n), col=col, border=FALSE, space=FALSE,
                         axes=FALSE)
    axis(1, at=midpoints, labels=1:n, lty=0, cex.axis=0.6)
    par(opar)
  }

  return(col)
}

