## The Gibbs sampling algorithm
gibbs.full.low.rank.brr <- function(model, data, n.iter, thin, fixed.brr.rank=NULL, brr.vars.to.record=c('a3a4','brr.rank'), fa.vars.to.record=c('a1a2','rank'), simple.brr.update=TRUE, tpve.lower.stopping.value=0, brr.vars.to.fix= c('Gamma.local.shrinkage'), fix.gamma.iteration=1000) {
	#
	# Function for simulating from the posterior distribution of 
	# a the infinite Bayesian low-rank reduced-rank regression model, 
	#
	# Inputs:
	#	n.iter: The number of iterations to run
	#
	#	model: This is model from which the MCMC is started.
	#          This contains fields:
	#          fa$context: Context of the covariance part.
	#          fa$prior: Priors for the covariance part.
	#          brr$context: Context for the reduced-rank coef matrix.
	#          brr$prior: Priors for the reduced-rank coef matrix.
	#          A: Confounder to phenotype coefficient matrix.
	#
	#	data: this list has elements genotypes, phenotypes,
	#          confounders, crossprod.genotypes
	#
	#	thin: thinning parameter for the MCMC
	#
	#   fixed.brr.rank: If this is a numeric value, this value 
	#          will be used as the fixed brr.rank
	#
	#	simple.brr.update: if true, update each variable only once.
	#			Otherwise, do several updates of those variables 
	#			in BRR part of the model which are fast to update.
	#
	#	tpve.lower.stopping.value: If the proportion of total variation
	#			explained goes below this value, then stop the algorithm.
	#
	#	brr.vars.to.fix: these brr variables are not updated
	#
	#	fix.gamma.iteration: After this many iterations, gamma will
	#			not be updated anymore (only its scale may change)
	#
	# Outputs, a list containing:
	#	model: updated model
	#
	#	traces: MCMC traces of the variables.

	
	
	pheno.mean.components <- NULL

	if (!is.null(fixed.brr.rank)) {
		# If brr.rank has been fixed, check
		# that the correct value is in the
		# brr context.
		if (model$brr$context$brr.rank != fixed.brr.rank) {
			stop('Initial brr rank not equal to the fixed value')
		}
	}

	# Initialize variables for storing MCMC outputs:
	traces <- add.to.trace(model=model)
	cpu.times=NULL

	t.res <- 0

	curr.tpve <- 1

	# This list is used for saving early gamma
	# samples at higher frequency than the trace
	# is usually updated.
	gamma.early.samples <- list()


	for (iter in 1:n.iter) {
		
		if (curr.tpve > tpve.lower.stopping.value) {

			## Update confounder to phenotype coefficients. (Only if there are
			## any confounders in the model...)
			if (!(is.na(data$confounders)[1])) {
				tX <- proc.time()[3]
				residuals <- compute.residuals(type='confounder', model=model, data=data)
				tY <- proc.time()[3]
				t.res <- t.res + tY-tX
				conf.updated <- update.confounder.part(data=data, residuals=residuals, model=model)
				model <- conf.updated$model
			} else {
				conf.updated <- NULL
			}

			## Update the low-rank covariance part of the model.
			tX <- proc.time()[3]
			residuals <- compute.residuals(type='fa', model=model, data=data)
			tY <- proc.time()[3]
			t.res <- t.res + tY-tX
			fa.res <- sparse.fa.gibbs(n.iter=1, context=model$fa$context, prior=model$fa$prior, Y=residuals)

			model$fa$context <- fa.res$context
			
			

			## Update the reduced-rank regression part of the model.
			tX <- proc.time()[3]
			residuals <- compute.residuals(type='brr', model=model, data=data)
			tY <- proc.time()[3]
			t.res <- t.res + tY-tX
			
			
			brr.updated <- update.brr.part(data=data, residuals=residuals, model=model, fixed.brr.rank=fixed.brr.rank, simple.brr.update=simple.brr.update, brr.vars.to.fix=brr.vars.to.fix, fix.gamma.iteration=fix.gamma.iteration, current.iter=iter, gamma.early.samples=gamma.early.samples)
			
			model <- brr.updated$model
			gamma.early.samples <- brr.updated$gamma.early.samples
			
			

			## Add current state to trace
			if (iter %% thin==0) {
				traces <- add.to.trace(model=model, data=data, trace=traces, iter=iter, brr.vars.to.record=brr.vars.to.record, fa.vars.to.record=fa.vars.to.record)

				curr.length <- length(traces$tpve)
				curr.tpve <- traces$tpve[[curr.length]]
			}


			## Update times consumed by each different update
			cpu.times <- record.cpu.times(cpu.times=cpu.times, conf.res=conf.updated, fa.res=fa.res, brr.res=brr.updated)

		}

	}

	return(list(model=model, traces=traces, cpu.times=cpu.times, t.res=t.res))
}



update.confounder.part <- function(data, residuals, model) {

	t1 <- proc.time()
	n.confounders <- ncol(data$confounders)
	n.pheno <- ncol(data$phenotypes)

	variances <- model$fa$context$variances

	if (any(is.na(model$A))) {
		model$A <- matrix(NA, nrow=n.confounders, ncol=n.pheno)	
	}
	

	pars <- fit.bayes.lm.ref.prior(X=data$confounders, y=residuals, noise.var=variances, crossprod.X=data$crossprod.X)

	for (i in 1:n.pheno) {
		# Regression for each phenotype can be
		# fitted separately, as independent error
		# terms are assumed (after removing the
		# effects of the latent factors).

		#model$A[,i] <- mvrnorm(n=1, mu=pars$posterior.mean[[i]], Sigma=pars$posterior.cov[[i]])
		model$A[,i] <- mvr.norm.own(mu=pars$posterior.mean[[i]], Sigma=pars$posterior.cov[[i]])
	}
	t2 <- proc.time()
	cpu.times <- t2[3]-t1[3]
	names(cpu.times) <- 'A'
	return(list(model=model, cpu.times=cpu.times))
}



update.brr.part <- function(data, residuals, model, fixed.brr.rank, simple.brr.update, brr.vars.to.fix, fix.gamma.iteration, current.iter, gamma.early.samples) {
	
	# BRR updates require knowledge of the variance parameters
	# that are contained in FA context. These are given as prior
	# information for the BRR updates.
	prior <- model$brr$prior
	prior$variances <- model$fa$context$variances
	
	# Divide the parameters into three groups depending on 
	# how long the corresponding updates take.

	if (is.null(fixed.brr.rank)) {
		# Update all variables
		vars.fast <- c('star.deltas','a3a4','brr.rank')
	} else {
		# Update all variables, except brr.rank, which is fixed.
		vars.fast <- c('star.deltas','a3a4')
	}
	

	vars.slow <- c('Psi', 'Gamma')  



	if (current.iter < fix.gamma.iteration) {
		
		# Update gamma in the early phase


		vars.medium <- c('Gamma','Psi.local.shrinkage','Gamma.local.shrinkage')

		

		# Save samples
		if ((current.iter %% 10 == 0) & (current.iter>=fix.gamma.iteration/2)) {
			current.length <- length(gamma.early.samples)
			gamma.early.samples[[current.length+1]] <- model$brr$context$Gamma
		}

	} else if (current.iter==fix.gamma.iteration) {
		# Fix gamma to the final value

		gamma.estimate <- gamma.early.samples[[1]]
		if (length(gamma.early.samples)>1) {
			for (i in 2:length(gamma.early.samples)) {
				gamma.estimate <- gamma.estimate+gamma.early.samples[[i]]
			}
		}
		gamma.estimate <- gamma.estimate / length(gamma.early.samples)
		model$brr$context$Gamma <- gamma.estimate

		vars.medium <- c('Gamma.scale','Psi.local.shrinkage','Gamma.local.shrinkage')

	} else {
		# After fix.gamma.iteration iterations, only the scale
		# of Gamma is updated
		vars.medium <- c('Gamma.scale','Psi.local.shrinkage','Gamma.local.shrinkage')
	}
	vars.all <- c(vars.slow,vars.medium,vars.fast)

	# Don't update variables specified in brr.vars.to.fix
	vars.all <- setdiff(vars.all, brr.vars.to.fix)

	# Update once all variables
	brr.res <- infinite.brr.gibbs(n.iter=1, vars.to.update=vars.all, context=model$brr$context, prior=prior, genotypes=data$genotypes, phenotypes=residuals, crossprod.genotypes=data$crossprod.genotypes)
	
	cpu.times <- brr.res$cpu.times
	model$brr$context <- brr.res$updated.context

	model$brr$prior$Psi.Gamma.adaptation.pars$accepted <- model$brr$prior$Psi.Gamma.adaptation.pars$accepted + brr.res$traces$accepted

	model$brr$prior$Psi.Gamma.adaptation.pars$rejected <- model$brr$prior$Psi.Gamma.adaptation.pars$rejected + brr.res$traces$rejected

	if (!simple.brr.update) {
		for (i in 1:3) {
			# Repeat the following operations three times

			# Update the fast variables 100 times
			brr.res <- infinite.brr.gibbs(n.iter=100, vars.to.update=vars.fast, context=model$brr$context, prior=prior, genotypes=data$genotypes, phenotypes=residuals, crossprod.genotypes=data$crossprod.genotypes)
		
			cpu.times <- cpu.times + brr.res$cpu.times
			model$brr$context <- brr.res$updated.context

			# Update the medium variables once
			brr.res <- infinite.brr.gibbs(n.iter=1, vars.to.update=vars.medium, context=model$brr$context, prior=prior, genotypes=data$genotypes, phenotypes=residuals, crossprod.genotypes=data$crossprod.genotypes)

			cpu.times <- cpu.times + brr.res$cpu.times
			model$brr$context <- brr.res$updated.context
		}
	}

	return(list(model=model, cpu.times=cpu.times, gamma.early.samples=gamma.early.samples))
}



add.to.trace <- function(model, data, trace=NULL, iter=NA, brr.vars.to.record=c('Psi','Gamma','Psi.local.shrinkage','Gamma.local.shrinkage','star.deltas','a3a4','brr.rank'), fa.vars.to.record=c('variances','local.shrinkage','rank','Eta','a1a2','deltas','Lambda','A')) {

	if (is.null(trace)) {
		# Initialize trace
		trace <- list()

		for (update.type in c('brr','fa')) {
			#var.names <- names(model[[update.type]]$context)
			for (name in c(brr.vars.to.record, fa.vars.to.record)) {
				trace[[name]] <- list()
			}
		}

		trace$A <- list()
		trace$iter <- NULL
		trace$tpve <- list() # Always included
		
		if (is.element('maf.group.tpve', brr.vars.to.record)) {
			trace$maf.group.tpve <- list()
		}
		
	} else {
		
		curr.length <- length(trace$A)

		for (name in brr.vars.to.record) {
			if (name != 'maf.group.tpve') {
				trace[[name]][[curr.length+1]] <- model[['brr']]$context[[name]]
			}
		}

		for (name in fa.vars.to.record) {
			trace[[name]][[curr.length+1]] <- model[['fa']]$context[[name]]
		}
		
		trace$A[[curr.length+1]] <- model$A
		trace$iter[curr.length+1] <- iter

		total.variation.explained <- compute.amount.total.variance.explained(genotypes=data$genotypes, Psi=model$brr$context$Psi, Gamma=model$brr$context$Gamma)

		total.variation.in.data <- sum(apply(data$phenotypes, 2, var))

		trace$tpve[[curr.length+1]] <- total.variation.explained / total.variation.in.data
		trace$factorwise.tpve[[curr.length+1]] <- compute.factorwise.variance(data=data, Psi=model$brr$context$Psi, Gamma=model$brr$context$Gamma) / total.variation.in.data
		
		if (is.element('maf.group.tpve', brr.vars.to.record)) {
			# Compute the explained variance by SNPs with 
			# MAF below or above a given threshold		
			variants <- list()
			maf.threshold <- model$brr$prior$rare.maf.threshold	
			variants[['rare']] <- which(data$maf < maf.threshold)
			variants[['common']] <- which(data$maf >= maf.threshold)
	
			explained.variance <- rep(0,3)
			names(explained.variance) <- c('rare','common','joint')
	
			for (type in c('rare','common')) {
				if (length(variants[[type]])>0) {
					explained.variance[type] <- compute.amount.total.variance.explained(genotypes=data$genotypes[,variants[[type]], drop=FALSE], Psi=model$brr$context$Psi[variants[[type]],, drop=FALSE], Gamma=model$brr$context$Gamma)
				}
			}
			#explained.variance['joint'] <- compute.joint.explained.variance(data=data, Psi=model$brr$context$Psi, Gamma=model$brr$context$Gamma, groups=variants)
			explained.variance['joint'] <- total.variation.explained - sum(explained.variance[c('rare','common')])
		
			trace$maf.group.tpve[[curr.length+1]] <- explained.variance / total.variation.in.data
		}
		
	}

	return(trace)
}



record.cpu.times <- function(cpu.times=NULL, conf.res=NULL, fa.res=NULL, brr.res=NULL) {
# The function updates a vector specifying CPU-times taken by 
# updates of each variable.
#
# Inputs:
#	cpu.times: a vector where the total time spent at each update
#              is written
#
#	conf.res, fa.res, brr.res: lists, each having cpu.times as one 
#              of the components. The cpu.times is a list where the
#              elements tell the cpu time taken by updates of FA,
#              BRR, or confounder parts of the models, respectively.
	
	if (is.null(cpu.times)) {
		# Initialize trace

		var.names <- NULL
		var.names <- names(brr.res$cpu.times)
		var.names <- c(var.names, names(fa.res$cpu.times))
		var.names <- c(var.names, names(conf.res$cpu.times))
		cpu.times <- rep(0, length(var.names))
		names(cpu.times) <- var.names

	}


	for (res in c('conf.res','fa.res','brr.res')) {
		
		if (!is.null(res)) {
			cpu.times.to.add <- eval(parse(text=paste(res, '$cpu.times', sep='')))
			var.names <- names(cpu.times.to.add)
			
			for (name in var.names) {
				cpu.times[name] <- cpu.times[name] + cpu.times.to.add[name]
			}
		}
	}

	return(cpu.times)
}




## Initialization of the model
initialize.from.prior <- function(n.pheno=10, n.snps=15, n.patients=200, fa.rank=10, brr.rank=5, alpha0=-1, alpha1=-0.005, fa.relevance.cutoff=0.01, brr.relevance.cutoff=0.01, local.shrinkage.nu=3, n.confounders=2, a3.shape=10, a3.rate=0.28, a3.lower.bound=2, a4.shape=4.1, a4.rate=0.31, a4.lower.bound=3, output.clustering = NULL) {
	
	


	model <- list()

	# Simulate the low-rank covariance:
	fa <- initialize.fa.from.prior(a1.shape=18, a1.rate=2, a1.lower.bound=2, a2.shape=18, a2.rate=2, a2.lower.bound=3, a.sigma=2.2, b.sigma=0.3, local.shrinkage.nu=local.shrinkage.nu, factor.relevance.cutoff=fa.relevance.cutoff, alpha0=alpha0, alpha1=alpha1, rank=fa.rank, n.patients=n.patients, n.pheno=n.pheno)
	
	model$fa$context <-fa$context
	model$fa$prior <- fa$prior

	
	# Simulate the reduced-rank regression coefficients:
	inf.brr <- initialize.infinite.brr(local.shrinkage.nu=local.shrinkage.nu, a3.shape=a3.shape, a3.rate=a3.rate, a3.lower.bound=a3.lower.bound, a4.shape=a4.shape, a4.rate=a4.rate, a4.lower.bound=a4.lower.bound, brr.factor.relevance.cutoff=brr.relevance.cutoff, alpha0=alpha0, alpha1=alpha1, a.sigma=NA, b.sigma=NA, brr.rank=brr.rank, n.snps=n.snps, n.pheno=n.pheno, input.clustering=NULL, output.clustering=output.clustering)

	model$brr$context <- inf.brr$context
	model$brr$prior <- inf.brr$prior
	# NOTE: a.sigma is NA, therefore, variances are not simulated
	# (they are already in model$fa$context)


	# Simulate the regression coefficients for the confounders:
	model$A <- matrix(rnorm(n=n.confounders*n.pheno, mean=0, sd=1), nrow=n.confounders, ncol=n.pheno)  
	# Just some random initialization.
	# Formally, A has improper prior.

	return(model)
}


simulate.from.full.low.rank.brr <- function(true.model) {

	# Variables related to the low-rank covariance
	variances <- true.model$fa$context$variances
	Eta <- true.model$fa$context$Eta
	Lambda <- true.model$fa$context$Lambda
	
	# Variables related to the reduced-rank regression
	Gamma <- true.model$brr$context$Gamma
	Psi <- true.model$brr$context$Psi
	
	# Regression coefficient matrix for the confounders
	A <- true.model$A

	n.snps <- nrow(Psi)
	n.patients <- nrow(Eta)
	n.confounders <- nrow(A)
	n.pheno <- length(variances)

	# Simulate genotypes and confounders
	genotypes <- simulate.genotypes(n.patients=n.patients, n.snps=n.snps)
	confounders <- matrix(rnorm(n=n.patients*n.confounders, mean=0, sd=1), nrow=n.patients, ncol=n.confounders)

	crossprod.genotypes <- crossprod(genotypes)

	# Simulate the phenotypes
	noise.covariance <- diag(variances)
	noise <- mvrnorm(n=n.patients, mu=rep(0,n.pheno), Sigma=noise.covariance)

	phenotypes <- genotypes %*% Psi %*% Gamma + confounders %*% A + Eta %*% t(Lambda) + noise

	# Return the simulated genotypes, confounders and phenotypes, and pre-computed
	# cross product for genotypes.
	return(list(genotypes=genotypes, phenotypes=phenotypes, confounders=confounders, crossprod.genotypes=crossprod.genotypes))
}


compute.residuals <- function(type, model, data) {
	#
	# Helper function for the Gibbs sampling. The function
	# computes the residuals 
	#
	# Inputs:
	#	type: One of 'brr', 'fa', or 'confounder', depending on
	#         which part of the model we are going to update
	#         using the obtained residuals.
	#
	#	model: The current model
	#	
	#	data: List with fields: 'genotypes', 'phenotypes', and
	#		  'confounders'
	#
	#
	# Outputs, a list containing:
	#	residuals: matrix of residuals

	phenotypes <- data$phenotypes
	if (type=='brr') {

		Eta <- model$fa$context$Eta
		Lambda <- model$fa$context$Lambda
 		A <- model$A
 		confounders <- data$confounders
		if (!is.na(confounders)[1]) {
			residuals <- phenotypes - confounders %*% A - tcrossprod(Eta,Lambda)
		} else {
			residuals <- phenotypes - tcrossprod(Eta,Lambda)
		}

	} else if (type=='fa') {

 		A <- model$A
 		confounders <- data$confounders
		genotypes <- data$genotypes
 		Psi <- model$brr$context$Psi
 		Gamma <- model$brr$context$Gamma
		if (!is.na(confounders)[1]) {
			residuals <- phenotypes - genotypes %*% Psi %*% Gamma - confounders %*% A
		} else {
			residuals <- phenotypes - genotypes %*% Psi %*% Gamma
		}
		

	} else if (type=='confounder') {

		Eta <- model$fa$context$Eta
		Lambda <- model$fa$context$Lambda
		genotypes <- data$genotypes
 		Psi <- model$brr$context$Psi
 		Gamma <- model$brr$context$Gamma

		residuals <- phenotypes - genotypes %*% Psi %*% Gamma - tcrossprod(Eta,Lambda)

	} else {
		stop('Unknown residual type')
	}

	return(residuals)
}



compute.amount.total.variance.explained <- function(genotypes, Psi, Gamma) {
	#
	# Function for computing the amount of total variation
	# explained by the reduced rank part of the model, given
	# fixed parameters Psi and Gamma.
	#
	# Inputs:
	#	genotypes: matrix of centered genotypes
	#
	#	Psi, Gamma: low rank representation of the 
	#		coefficient matrix
	#
	#
	# Outputs:
	#	amount.total.var.explained: a scalar specifying the amount
	#		of variance explained by the BRR part of the model.	
	#

	aux <- genotypes %*% Psi %*% Gamma
	#browser()
	amount.total.var.explained <- sum(apply(aux,2,var))

	#coef.mat <- Psi %*% Gamma
	#genotype.cov <- crossprod(genotypes) / 4701 #cov(genotypes)

	#aux <- t(coef.mat) %*% genotype.cov %*% coef.mat

}

compute.joint.explained.variance <- function(data, Psi, Gamma, groups) {
	# Computes the variation explained due to the
	# correlation of variants in the two different
	# groups. (If variants were independent, then,
	# assuming additivity, the estimated variations
	# could just be summed).
	#
	# Inputs:
	#	data
	#
	#	Psi, Gamma: low rank representation of the 
	#		coefficient matrix
	#
	# 	groups: division of SNPs into two groups whose
	#		joint effect is estimated here.
	#
	# Outputs:
	#	joint.var.explained: a scalar specifying the amount
	#		of joint explained variance.	
	#
	coef.mat <- Psi %*% Gamma
	n.patients <- nrow(data$genotypes)
	genotype.cov <- data$crossprod.genotypes / (n.patients-1)
	
	cross.cov <- genotype.cov[groups[[1]],groups[[2]]]
	
	aux <- t(coef.mat[groups[[1]], , drop=FALSE]) %*% cross.cov %*% coef.mat[groups[[2]],,drop=FALSE]
	
	joint.var.explained <- sum(diag(aux))*2
}


compute.factorwise.variance <- function(data, Psi, Gamma) {
	
	n.patients <- nrow(data$genotypes)
	genotype.cov <- data$crossprod.genotypes / (n.patients-1)
	aux.Gamma <- tcrossprod(Gamma)  # same as Gamma %*% t(Gamma)
	aux.Psi <- t(Psi) %*% genotype.cov %*% Psi

	total <- aux.Gamma * aux.Psi

	# Total explained variance is obtained by summing elements
	# in total. Note that off-diagonal elements are added twice, 
	# as should.
	return (total)

}


compute.tpve.trace <- function(data, mcmc.output) {
	#
	# Function computes the posterior samples of the 
	# proportion of total variation explained.
	#
	# Inputs:
	# 	data: must contains fields genotypes and 
	#         phenotypes
	#
	#   mcmc.output: must contain field traces, which
	#         has further fields Psi and Gamma.
	#
	#
	# Outputs:
	#	tpve.trace: posterior samples of the proportion
	#          of total variation explained.
	#
	
	n.samples <- length(mcmc.output$traces$Psi)
	tpve.trace <- rep(NA, n.samples)
	
	total.variance <- sum(apply(data$phenotypes, 2, var))
	
	for (i in 1:n.samples) {
		 amount.explained <- compute.amount.total.variance.explained(genotypes=data$genotypes, Psi=mcmc.output$traces$Psi[[i]], Gamma=mcmc.output$traces$Gamma[[i]])
		 
		 tpve.trace[i] <- amount.explained / total.variance
	}
	
	return(tpve.trace)
}



plot.tpve.distribution <- function(data, mcmc.output, true.model=NULL) {
	# 
	# Function plots a histogram of the sampled
	# proportions of total variation explained.
	# If the true model is given, then the function
	# draws a line to the plot showing the value of
	# the true proportion of total variation explained.
	#
	
	if (!is.null(true.model)) {
		total.variance <- sum(apply(data$phenotypes, 2, var))
	
		true.tpve <- compute.amount.total.variance.explained(genotypes=data$genotypes, Psi=true.model$brr$context$Psi, Gamma=true.model$brr$context$Gamma) / total.variance
		
		print(true.tpve)
	
	} else {
		true.tpve = NA
	}
	
	tpve.trace <- compute.tpve.trace(data, mcmc.output)
	
	if (is.na(true.tpve)) {
		# True value is not known
		
		x.axis.min <- min(tpve.trace) * 0.9
		x.axis.max <- max(tpve.trace) * 1.1
		
	} else {
	
		x.axis.min <- min(tpve.trace) * 0.9
 		x.axis.max <- max(c(tpve.trace, true.tpve)) * 1.1
	}
	x.axis.limits <- c(x.axis.min, x.axis.max)
	
	#x11()
	postscript(file='tpve.eps',width=7,height=5, pointsize=10, bg='white')
	hist(tpve.trace, xlim=x.axis.limits, main='Posterior TPVE')
	
	if (!is.na(true.tpve)) {
		abline(v=true.tpve)
	}
	
	dev.off()
	return(tpve.trace)
}


simulate.tpve.from.prior <- function(a3.shape=52, a3.rate=2, a3.lower.bound=2, a4.shape=4.1, a4.rate=0.31, a4.lower.bound=2.7, n.simulations=100, brr.rank=5, data) {
	#
	# Given a certain data, the function samples from the prior
	# the values of explained total proportion for the data.
	#
	# Inputs:
	#	data: has at least fields phenotypes and genotypes
	#
	#	a3,a4: hyperparameter values that determine the
	#		   prior of the genotype to phenotype coefficient
	#		   matrix.
	#		
	#	n.simulations: the number of samples to simulate-
	#
	# Outputs:
	#	tpve.samples
	#	
	tpve.samples <- rep(NA, n.simulations)
	
	n.snps <- ncol(data$genotypes)
	n.pheno <- ncol(data$phenotypes)
	n.patients <- nrow(data$phenotypes)
	
	total.variance <- sum(apply(data$phenotypes, 2, var))
	
	for (sim.index in 1:n.simulations) {
		
		# It is sufficient to simulate only the
		# reduced rank regression part of the model
		
		# Note: only those parameters are given as
		# arguments, which are not set to default values.
		inf.brr <- initialize.infinite.brr(a3.shape=a3.shape, a3.rate=a3.rate, a3.lower.bound=a3.lower.bound, a4.shape=a4.shape, a4.rate=a4.rate, a4.lower.bound=a4.lower.bound, brr.rank=brr.rank, n.snps=n.snps, n.pheno=n.pheno)
		
		tpve.samples[sim.index] <- compute.amount.total.variance.explained(genotypes=data$genotypes, Psi=inf.brr$context$Psi, Gamma=inf.brr$context$Gamma) / total.variance
	
	}
	
	return(tpve.samples)
	
}


compute.a3.interval <- function(tpve=c(0.001,0.05), a4.shape=4.1, a4.rate=0.31, a4.lower.bound=2.7, a3.lower.bound=2, local.shrinkage.nu=3, snp.total.variation=1, phenotype.avg.variance=1) {
	#
	# This function is used for determining suitable
	# values for the gamma distribution of a3.
	#
	# Inputs:
	#	tpve: Interval for the proportion of total variation
	#		explained.
	#
	#	a4.shape/rate/lower.bound: These are the
	#		parameters that determine the prior
	#		gamma distribution of the a4 parameter.
	#
	#	a3.lower.bound: Lower bound for a3.
	#
	#	local.shrinkage.nu: the parameter that determines
	#		the variance of the local shrinkage parameters.
	#
	#	snp.total.variation: Total variation in the SNP data
	# 		(with fixed hyperparameters, the tpve is linearly
	#		proportional to this)
	#
	#	phenotype.avg.variance: the average variance of the
	#		phenotypes.
	#
	# Outputs:
	#	A list with two elements:
	#		a3.min: value of a3 that gives expected tpve 
	#		equal to tpve[2]
	# 		
	#		a3.max: value of a3 that gives expected tpve
	#		euqla to tpve[1]
	
	a4.samples <- rgamma(n=1000, shape=a4.shape, rate=a4.rate)
	illegal.indices <- which(a4.samples<a4.lower.bound)
	a4.samples <- a4.samples[-illegal.indices]
	
	delta2.inv.values <- gamma(a4.samples-2) / gamma(a4.samples)
	
	mean.delta2.inv.squared <- mean(delta2.inv.values)
	
	rhs <- (local.shrinkage.nu - 2) / local.shrinkage.nu * (1-mean.delta2.inv.squared) / snp.total.variation * phenotype.avg.variance * tpve
		 
	f <- function(x,a) {lgamma(x-2)-lgamma(x)-log(a)}
	
	a3.max <- uniroot(f, c(2.1,1000000), a=rhs[1])$root
	a3.min <- uniroot(f, c(2.1,1000000), a=rhs[2])$root

	return(list(a3.max=a3.max, a3.min=a3.min))
}

determine.gamma.pars <- function(quantiles, values) {
	#
	# The function determines the shape and rate parameter
	# of a gamma distribution with specified quantiles.
	#
	# Inputs: 
	# 	quantiles: two quantiles, e.g. c(0.01, 0.99)
	#
	#	values: values of the quantiles, e.g. c(10,40)

	# Go through a range of mean (mu) parameters.
	# For each mean parameter value, select (shape,rate)
	# which fix the first quantile to correct. See how far
	# the second quantile is from the desired value.
	# Finally, select the values that best matched the
	# second quantile.
	
	diff <- (values[2]-values[1])/1000
	sequence.of.mean.vals.to.test <- seq(values[1]+diff,values[2]-diff, by=diff)
	#sequence.of.mean.vals.to.test <- sequence.of.mean.vals.to.test[-c(1,length(sequence.of.mean.vals.to.test))]
	
	shapes <- rep(NA, length(sequence.of.mean.vals.to.test))
	rates <- rep(NA, length(sequence.of.mean.vals.to.test))
	errors.in.second.quantile <- rep(NA, length(sequence.of.mean.vals.to.test))
	
	counter <- 1
	for (mu in sequence.of.mean.vals.to.test) {
		#print(mu)
		f <- function(x, p, mu, rhs) {qgamma(p=p, shape=x, rate=x/mu) - rhs}			
		ans <- uniroot(f, c(0.1,100000000), p=quantiles[1], mu=mu, rhs=values[1])
		shape <- ans$root
		rate <- shape/mu
		shapes[counter] <- shape
		rates[counter] <- rate
		errors.in.second.quantile[counter] <- abs(qgamma(p=quantiles[2], shape=shape, rate=rate) - values[2])
		counter <- counter+1		
	}
	
	opt <- which.min(errors.in.second.quantile)
	
	shape <- shapes[opt]
	rate <- rates[opt]
	
	return(list(shape=shape, rate=rate))
}

remove.burnin <- function(mcmc.output, burnin) {
	for (name in names(mcmc.output$traces)) {
		mcmc.output$traces[[name]] <- mcmc.output$traces[[name]][-seq(1:burnin)]
	}
	
	return(mcmc.output)
}

