function plotCombinedGenes(rootDirectory, datasetName, treeFileName, profilesWidth, colorOrdering)
    
    if nargin == 3
        profilesWidth = 5; % Appropriate for the example with 54 genes
        colorOrdering = [];
    elseif nargin == 4
        profilesWidth = str2num(profilesWidth);
        colorOrdering = [];
    elseif nargin == 5
        profilesWidth = str2num(profilesWidth);
        colorOrdering = str2num(colorOrdering);
    end
    
    delimiter = getDelimiter();
    rootToSummaries = [rootDirectory datasetName 'Summaries' delimiter];
    
    treeFileWithPath = [rootToSummaries delimiter treeFileName];
    
    namesToPlot = initializeThePlot(treeFileWithPath);
    nStrains = length(namesToPlot);
    
    load([rootToSummaries delimiter 'combinedPopStructure.mat']);
    % 'combinedPopStructure', 'combinedSnpPositions', 'totalCoreLength', 
    % 'maxClusterIndex', 'collectedGeneLengths'
    load([rootToSummaries delimiter 'globalNames.mat']);
    
    load([rootToSummaries delimiter 'allSegments.mat']); % allSegments
    nLineages = double(maxClusterIndex);
    
    xLimOrig = get(gca, 'XLim');
    annotationSpaceOnLeft = 0.1;
    phyloEnd = xLimOrig(2);
    
    segmentStart = phyloEnd + (xLimOrig(2)-xLimOrig(1)) .* annotationSpaceOnLeft;
    
    emptySpaceLength = (xLimOrig(2) - xLimOrig(1)) .* (profilesWidth + 0.15);
    segmentEnd = segmentStart + (xLimOrig(2) - xLimOrig(1)) .* profilesWidth;
    
    set(gca, 'XLim',[xLimOrig(1) segmentStart + emptySpaceLength]);
    
    % Set colormap
    colors = distinguishable_colors(nLineages);
    
    cmap = colors(end:-1:1,:);
    
    if ~isempty(colorOrdering)
        cmap = cmap(colorOrdering,:);
    end
    
    cmap(nLineages+1,:) = [0 0 0]; % Black color for the empty cluster.
    cmap(nLineages+2,:) = [1 1 1]; % White color for the recent recombinations which are removed.
    cmap = brighten(cmap, 0.5);
    
    
    % Draw segments:
    for yCoord = 1:nStrains
        % Draw the strain at this y-coord (this corresponds to the
        % index in the tree, note that the combined results were
        % ordered using this.
            
            strainNameNow = namesToPlot{yCoord};
            strainIndex = strmatch(strainNameNow, globalNames);
            
            
            % Identify segments
            segments = allSegments{strainIndex};
            
            % Draw segments
            plotSegments(yCoord, segments, segmentStart, segmentEnd, totalCoreLength, cmap);
        
    end
    
    tickInfo = determineTicks(segmentStart, segmentEnd, totalCoreLength);
    % "tickInfo" has two fields "locations" and "values"
    set(gca,'XTick',tickInfo.locations, 'XTickLabel', tickInfo.values);
    
    % Show gene boundaries
    cumLengths = [0 cumsum(collectedGeneLengths)];
    geneBoundaries = segmentStart + (segmentEnd-segmentStart) .* cumLengths ./ totalCoreLength;
    %delete(ll)
    ll = line([geneBoundaries;geneBoundaries],repmat([0.5; nStrains+0.5], [1,length(geneBoundaries)]), 'Color', 'black');
    
    writeNamesInThePlot(namesToPlot, rootToSummaries, delimiter, treeFileName)
    
end



function plotSegments(yCoord, segments, segmentStart, segmentEnd, totalSequenceLength, cmap)
    
    nSegments = size(segments,1);
    totalLengthInCoordinates = segmentEnd - segmentStart;
    lengthRatio = totalLengthInCoordinates / totalSequenceLength;
    
    for segmentIndex = 1:nSegments
        
        lengthOfSegment = segments(segmentIndex,2) - segments(segmentIndex,1) + 1;
        lengthInCoordinates = lengthOfSegment .* lengthRatio;
        
        totalLengthOfSegmentsSoFar = segments(segmentIndex,1) - 1;
        totalLengthOfSegmentsSoFar = totalLengthOfSegmentsSoFar .* lengthRatio;
        
        x1 = segmentStart + totalLengthOfSegmentsSoFar;
        x2 = x1 + lengthInCoordinates;
        
        origin = segments(segmentIndex,3);
        if isempty(cmap)
            color = [0.8 0.8 0.8];
            edgeColor = [0.8 0.8 0.8];
        else
            color = cmap(origin,:);
            edgeColor = 'none';
        end
        patch([x1 x2 x2 x1], [yCoord+0.5 yCoord+0.5 yCoord-0.5 yCoord-0.5],color,'EdgeColor',edgeColor);
        
    end
    
end


function writeNamesInThePlot(namesToPlot, rootToSummaries, delimiter, treeFileName)
    
    fileForNames = [rootToSummaries 'LEAF_NAMES_' treeFileName(1:end-4) '.txt'];
    
    fid = fopen(fileForNames, 'w');
    
    nStrains = length(namesToPlot);
    for rowIndex = nStrains:-1:1
        fprintf(fid, '%s\n', namesToPlot{rowIndex});
    end
    
    fclose(fid);
    
end

