function plotTreeAndRecombinations(resultFileName, recombinationType, treeFileName)
    % "resultFileName" is the fastGEAR result file 
    %
    % "treeFileName" is the name of the file in which a newick
    % tree is located. If treeFileName is not given empty, then the strains 
    % will be drawn in the order they appear in the input alignment 
    % (default). If the value is 1 or "1", then the strains will be ordered 
    % according to the clustering inferred by fastGEAR
    %
    % "recombinationType is either one of 'recent', 'ancestral', or 'both'
    
    if nargin==2
        treeFileName = [];
    end
    
    % Load results
    load(resultFileName); % uData
    snpDataFile = resultFileName(1:end-4);
    snpDataFile = [snpDataFile '_snpData.mat'];
    load(snpDataFile);
    missingnessTable = (snpData <= 0);
    clear snpData
    
    if isempty(treeFileName) || isequal(treeFileName,1) || isequal(treeFileName,'1')
        
        figure
        axes;
        nStrains = length(uData.groupedPartition);
        borderRegion = nStrains * 0.03; % Empty space on top/bottom
        segmentStart = 0.05;
        annotationWidth = 0.05; % Amount of empty space on the right side of the plot
        segmentEnd = 1 - annotationWidth;
        set(gca,'YLim', [0-borderRegion nStrains+borderRegion], 'XLim', [0 1]);
        
        if isempty(treeFileName)
            orderToShow = length(uData.groupedPartition):-1:1;
        else
            % Order the strains according to the clustering inferred by
            % fastGEAR
            orderToShow = sortStrainsByPartition(uData.groupedPartition);
            
        end
        
    else
        
        % Plot the tree in the given file
        tr = phytreeread(treeFileName);
        tr = reroot(tr);
        
        h = plot(tr, 'TerminalLabels', 'false', 'LeafLabels', 'false');
        set(h.BranchDots, 'Marker', 'none');
        set(h.LeafDots, 'Marker', 'none');
        set(h.axes, 'YTick', []);
        set(gcf,'Position',[709 80 770 840]);
        
        nStrains = length(h.leafNodeLabels);
        
        % Modify the layout of the axes
        phyloProportion = 0.25; % Proportion of whole figure in x-axis direction
        proportionSpaceRightFromTree = 0.01; % Proportion of whole figure on right side of tree.
        proportionAnnotationSpace = 0.05; % Proportion of empty space on the right side of the plot
        segmentProportion = 1 - phyloProportion - ...
            proportionSpaceRightFromTree - proportionAnnotationSpace;
        
        xLimOrig = get(gca,'XLim');
        phyloWidth = xLimOrig(2) - xLimOrig(1);
        emptySpace = phyloWidth .* proportionSpaceRightFromTree ./ phyloProportion; % On the rightside of the tree
        segmentsWidth = phyloWidth .* segmentProportion ./ phyloProportion;
        annotationWidth = phyloWidth .* proportionAnnotationSpace ./ phyloProportion;
        
        phyloEnd = xLimOrig(2);
        segmentStart = phyloEnd + emptySpace;
        segmentEnd = segmentStart + segmentsWidth;
        maxX = segmentEnd + annotationWidth;
        set(h.axes,'XLim',[xLimOrig(1) maxX]);
        
        orderToShow = zeros(nStrains,1); % This will be collected later in order to draw the lineage annotation.
    end
    
    cmap = determineColormap(uData, recombinationType);
    
    % Draw segments
    popStructureToPlot = formulatePopStructureToPlot(uData, recombinationType);
    
    % Draw segments:
    for yCoord = 1:nStrains
        % Draw the strain at this y-coord
        
        if exist('h')
            %positionInTree = nStrains - yCoord + 1;
            % Revers order, because labeling starts from top but drawing from bottom 
            strainName = h.leafNodeLabels(yCoord).String;
            strainIndex = find(cellfun(@(x) isequal(x, strainName), uData.strainLabels));
            if isempty(strainIndex)
                % This strain was not included in the alignment
                orderToShow(yCoord) = nan;
            else
                orderToShow(yCoord) = strainIndex;
            end
            % Empty if the strain that appears in the tree does not appear in the alignment.
        else
            strainIndex = orderToShow(yCoord);
        end
        
        if ~isempty(strainIndex)
            
            % Identify segments
            segments = identifySegments(popStructureToPlot, strainIndex, uData.snpPositions, uData.totalSequenceLength);
            
            %keyboard
            % Draw segments
            plotSegments(yCoord, segments, segmentStart, segmentEnd, uData.totalSequenceLength, cmap);
            
            % Draw segments of missing data
            if ~isempty(missingnessTable)
                segments = identifySegments(missingnessTable, strainIndex, uData.snpPositions, uData.totalSequenceLength);
                segments = segments(segments(:,3)==1,:);  % Those segments which were missing.
                plotSegments(yCoord, segments, segmentStart, segmentEnd, uData.totalSequenceLength, []);
            end
            
        end
    end
    
    box off
    axis off
    
    % DRAW CLUSTERS:
    strainsPresent = find(~isnan(orderToShow));
    orderedPartition = zeros(nStrains,1);
    orderedPartition(strainsPresent) = uData.groupedPartition(orderToShow(strainsPresent));
    %orderedPartition = uData.groupedPartition(orderToShow);
    
    %Show clusters only if more than one cluster
    for i=1:length(unique(uData.groupedPartition))
        strainsInCluster = find(orderedPartition==i);
        x1 = segmentEnd + annotationWidth/2;
        x2 = segmentEnd + annotationWidth*5/6;
        
        for strainIndex = strainsInCluster'
            patch([x1 x2 x2 x1],...
                [(strainIndex-0.5)*ones(1,2) (strainIndex+0.5)*ones(1,2)], ...
                cmap(i,:), 'EdgeColor','none');
        end
    end
    
    if exist('h')
        writeOrderInPlot(orderToShow(end:-1:1), uData.outputDir, uData.partition, uData.groupedPartition, uData.strainLabels, treeFileName);
    else
        writeOrderInPlot(orderToShow, uData.outputDir, uData.partition, uData.groupedPartition, uData.strainLabels, treeFileName);
    end
end


function writeOrderInPlot(orderInDendrogram, outputDir, partition, groupedPartition, strainLabels, treeFileName)
    
    sysName = computer;
    if (isequal(sysName(1:2),'PC')) % Windows
        delimiter = '\';
    else
        delimiter = '/';
    end
    
    if isempty(treeFileName)
        clarification = 'orig';
    elseif isequal(treeFileName,'1') || isequal(treeFileName,1)
        clarification = 'by_cluster';
    else
        clarification = 'from_tree';
    end
        
    fid = fopen([outputDir delimiter 'order_in_plot_' clarification '.txt'], 'w');
    
    row = 'PlotIndex Strain  Cluster  Lineage  Name';
    fprintf(fid, '%s\n',row);
    nStrains = length(orderInDendrogram);
    for rowIndex = nStrains:-1:1
        origStrainIndex = orderInDendrogram(rowIndex);
        if ~isnan(origStrainIndex)
            cluster = partition(origStrainIndex);
            lineage = groupedPartition(origStrainIndex);
            strainName = strainLabels{origStrainIndex};
            fprintf(fid,['%-10.0f%-8.0f%-9.0f%-9.0f' strainName '\n'], rowIndex, origStrainIndex, cluster, lineage);
        else
            fprintf(fid,['%-10.0f\n'], rowIndex);
        end
    end
    
    fclose(fid);
    
end




function plotSegments(yCoord, segments, segmentStart, segmentEnd, totalSequenceLength, cmap)
    
    nSegments = size(segments,1);
    totalLengthInCoordinates = segmentEnd - segmentStart;
    lengthRatio = totalLengthInCoordinates / totalSequenceLength;
    
    for segmentIndex = 1:nSegments
        
        lengthOfSegment = segments(segmentIndex,2) - segments(segmentIndex,1) + 1;
        lengthInCoordinates = lengthOfSegment .* lengthRatio;
        
        totalLengthOfSegmentsSoFar = segments(segmentIndex,1) - 1;
        totalLengthOfSegmentsSoFar = totalLengthOfSegmentsSoFar .* lengthRatio;
        
        x1 = segmentStart + totalLengthOfSegmentsSoFar;
        x2 = x1 + lengthInCoordinates;
        
        origin = segments(segmentIndex,3);
        if isempty(cmap)
            %color = [0.8 0.8 0.8];
            %edgeColor = [0.8 0.8 0.8];
            color = [1 1 1];
            edgeColor = [1 1 1];
        else
            color = cmap(origin,:);
            edgeColor = 'none';
        end
        patch([x1 x2 x2 x1], [yCoord+0.5 yCoord+0.5 yCoord-0.5 yCoord-0.5],color,'EdgeColor',edgeColor);
        
    end
    
end



function cmap = determineColormap(uData, recombinationType)
    
    nLineages = max(uData.groupedPartition);
    
    colors = distinguishable_colors(nLineages);
    
    cmap = colors(end:-1:1,:);
    if isequal(recombinationType,'both')
        cmap(nLineages+1,:) = [0 0 0]; % Black color for the empty cluster.
        cmap(nLineages+2,:) = [1 1 1];
    else
        cmap(nLineages+2,:) = [1 1 1]; % White color for the recent recombinations which are removed.
        cmap(nLineages+1,:) = [0 0 0];
    end
    cmap = brighten(cmap, 0.5);
    
end