function res = countRecentRecombinations(uData)
    % "res" has six fields: 
    % 
    % "nRecent" (total number of recent recombinations)
    % 
    % "numAsDonor"
    % "numAsRecipient"
    % "numAsRecipientFromOutside"
    %
    % "recombinationFlow" (matrix where element (i,j) tells the number of
    % recombination events from the lineage of strain i to strain j.
    %
    % "strainLabels"
    
    if length(uData.snpPositions) == 1
        res.nRecent = 0;
        res.numAsDonor = 0;
        res.numAsRecipient = 0;
        res.numAsRecipientFromOutside = 0;
        return
    end
    
    nStrains = length(uData.partition);
    allSegments = zeros(10000,5);
    nSegments = 0;
    for strainIndex = 1:nStrains
        segments = identifySegments(uData.popStructureCleaned, strainIndex, uData.snpPositions, uData.totalSequenceLength);
        nSegmentsNow = size(segments,1);
        if nSegmentsNow > 0
            allSegments(nSegments+1:nSegments+nSegmentsNow,1:3) = segments;
            allSegments(nSegments+1:nSegments+nSegmentsNow,4) = uData.groupedPartition(strainIndex);
            allSegments(nSegments+1:nSegments+nSegmentsNow,5) = strainIndex;
            nSegments = nSegments + nSegmentsNow;
        end
    end
    allSegments = allSegments(1:nSegments,:);
    homeSegmentInds = find(allSegments(:,3) == allSegments(:,4));
    allSegments(homeSegmentInds,:) = [];
    
    nLineages = max(uData.groupedPartition);
    % Find unique recombination events in each lineage
    nRecombinationEvents = zeros(nLineages,1);
    numAsDonor = zeros(nStrains,1);
    numAsRecipient = zeros(nStrains,1);
    numAsRecipientFromOutside = zeros(nStrains,1);
    
    for lineageIndex = 1:nLineages
        segmentsInLineage = find(allSegments(:,4) == lineageIndex);
        if ~isempty(segmentsInLineage)
            %nRecombinationEvents(lineageIndex) = countUniqueRecombinationsInLineage(allSegments(segmentsInLineage,:), uData.groupedPartition);
            eventRes = countUniqueRecombinationsInLineage(allSegments(segmentsInLineage,:), uData.groupedPartition);
            nRecombinationEvents(lineageIndex) = eventRes.nEvents;
            numAsDonor = numAsDonor + eventRes.numAsDonor;
            numAsRecipient = numAsRecipient + eventRes.numAsRecipient;
            numAsRecipientFromOutside = numAsRecipientFromOutside + eventRes.numAsRecipientFromOutside;
        end
    end
    res.nRecent = sum(nRecombinationEvents);
    res.numAsDonor = numAsDonor;
    res.numAsRecipient = numAsRecipient;
    res.numAsRecipientFromOutside = numAsRecipientFromOutside;
    
    
    % Compute the donor-recipient matrix
    recombinationFlow = zeros(nStrains+1,nStrains); % Last row is for recombinations from an unknown origin
    for lineageIndex = 1:nLineages+1 % (the last one for foreign recombinations)
        recombinationsFromThisLineage = find(allSegments(:,3) == lineageIndex);
        if ~isempty(recombinationsFromThisLineage)
            
            if lineageIndex <= nLineages
                donorStrains = find(uData.groupedPartition == lineageIndex);
            elseif lineageIndex == nLineages + 1
                donorStrains = nStrains + 1;
            end
                
            recipientStrains = allSegments(recombinationsFromThisLineage,5);
            
            uniqueRecipients = unique(recipientStrains);
            recipientCounts = histc(recipientStrains, uniqueRecipients);
            
            recombinationFlow(donorStrains, uniqueRecipients) = ...
                recombinationFlow(donorStrains, uniqueRecipients) + ...
                repmat(recipientCounts', [length(donorStrains), 1]);
            
        end
    end
    res.recombinationFlow = recombinationFlow;
    res.strainLabels = uData.strainLabels;
    
end



function res = countUniqueRecombinationsInLineage(segments, groupedPartition)
    %keyboard
    % Some example inputs for debugging:
    %segments = [1 5 1;4 6 1; 3 5 2; 6 7 1];
    %segments = [1 5 1; 2 7 1; 15 20 1; 10 15 1; 6 11 1];
    
    % Sorting rows according to the segment starting position avoids the
    % problem that a segment overlaps two previously labelled segments (one
    % on the left and one on the right side of the segments).
    segments = sortrows(segments,1);
    
    nSegments = size(segments,1);
    eventLabels = zeros(nSegments,1);
    % This tells the label of the event which the segment was part of.
    
    while any(eventLabels == 0)
        
        toProcess = find(eventLabels==0, 1, 'first');
        
        % Find segments that are part of the same recombination event for
        % the "toProcess" segment
        first = segments(toProcess, 1);
        last = segments(toProcess, 2);
        origin = segments(toProcess, 3);
        
        isOverlapping = (segments(:,1)<first & segments(:,2)>=first) | ...
            (segments(:,2)>last & segments(:,1)<=last) | ...
            (segments(:,1)>= first & segments(:,2)<=last);
        
        overlappingSegments = find(isOverlapping & ...
            (segments(:,3) == origin));
        
        % If any of the overlapping segments already has a label, then the
        % "toProcess" segment as well as all overlapping segments will get
        % that same label. Otherwise a new label will be given to all the
        % overlapping segments.
        existingLabels = unique(eventLabels(overlappingSegments));
        existingLabels = setdiff(existingLabels,0);
        if ~isempty(existingLabels)
            
            % The length of existingLabels should always be 1.
            if length(existingLabels) ~= 1
                keyboard
                error('Should not happen.');
            end
            
            eventLabels(overlappingSegments) = existingLabels;
            
        else
            
            % Give a new label to all overlapping segments
            newLabel = max(eventLabels) + 1;
            eventLabels(overlappingSegments) = newLabel;
            
        end
        
    end
    
    nEvents = max(eventLabels);
    
    % Compute summaries about how many times each strain was a donor or a
    % recipient
    numAsDonor = zeros(length(groupedPartition),1);
    numAsRecipient = zeros(length(groupedPartition),1);
    numAsRecipientFromOutside = zeros(length(groupedPartition),1);
    for eventIndex = 1:nEvents
        segmentRowsOfEvent = find(eventLabels == eventIndex);
        recipientStrains = segments(segmentRowsOfEvent, 5);
        donorLineage = segments(segmentRowsOfEvent(1), 3);
        donorStrains = find(groupedPartition == donorLineage);
        numAsDonor(donorStrains) = numAsDonor(donorStrains) + 1;
        numAsRecipient(recipientStrains) = numAsRecipient(recipientStrains) + 1;
        if donorLineage == max(groupedPartition)+1
            % Recombination came from outside
            numAsRecipientFromOutside(recipientStrains) = numAsRecipientFromOutside(recipientStrains) + 1;
        end
    end
    res.nEvents = nEvents;
    res.numAsDonor = numAsDonor;
    res.numAsRecipient = numAsRecipient;
    res.numAsRecipientFromOutside = numAsRecipientFromOutside;
    
end
