function emissionProbs = calcEmissionProbsForMergingTwoClusters(counts1, ...
    sumCounts1, counts2, sumCounts2, adjustedPriorCounts, priorSumCounts)
    % The function calculates emission probabilities for the observed SNPs 
    % in two clusters. The first hidden state corresponds to situation that
    % the two clusters are equal. The second hidden state corresponds to the
    % situation that the clusters are different.
    %
    % "counts1" and "counts2" contains the counts of observed SNPs in the two clusters.
    %
    % "sumCounts1"  and "sumCounts2" contain the total count of observed 
    % SNPs in different positions in different clusters.
    %
    % "adjustedPriorCounts" and "priorSumCounts" are analogous to the previous two, 
    % except that these contain the prior counts (parameters of the Dirichlet
    % distributions)
    %
    % "emissionProbs" table that is returned is 2*nLoci, such that the
    % first row is the probability of observing the data, if the clusters
    % are the same, and the second os the probability of the data, if the
    % clusters are different.
    %
    % USAGE EXAMPLE:
%     counts(:,:,1) = [100 200 200; 400 400 200; 100 0 200];
%     counts(:,:,2) = [0 0 1; 0 1 1; 2 0 0];
%     counts(:,:,3) = [0 0 0; 0 0 0; 0 0 0];
%     sumCounts = [600 600 600;2 1 2;0 0 0];
%     adjustedPriorCounts = [1/3 1/2 1/3;1/3 1/2 1/3;1/3 1 1/3];
%     priorSumCounts = ones(1,3);
%     clusterCounts = [20 20 10; 180 0 20; 20 30 10];
%     clusterSumCounts = [210 50 40];
%     emissionProbs = calcEmissionProbsForClusterMerging(counts, sumCounts, adjustedPriorCounts, priorSumCounts, clusterCounts, clusterSumCounts);
    
    nLoci = size(counts1,2);
    emissionProbs = zeros(2,nLoci);
    
    logProb1 = computeLocuswisePopulationLogml(counts1, sumCounts1, adjustedPriorCounts, priorSumCounts);
    logProb2 = computeLocuswisePopulationLogml(counts2, sumCounts2, adjustedPriorCounts, priorSumCounts);
    logProbSeparate = logProb1 + logProb2;
    
    logProbMerged = computeLocuswisePopulationLogml(counts1+counts2, sumCounts1+sumCounts2, adjustedPriorCounts, priorSumCounts);
    
    emissionProbs(1,:) = exp(logProbMerged);
    emissionProbs(2,:) = exp(logProbSeparate);
    
    if any(any(emissionProbs==0))
        logProbBoth = [logProbMerged; logProbSeparate];
        smallerLogProb = min(logProbBoth);
        logProbMinimum = -700; % This is the minimum value that avoids numerical problems
        amountTooSmall = logProbMinimum - smallerLogProb;
        amountTooSmall = max(amountTooSmall, 0);
        
        logProbBoth = logProbBoth + amountTooSmall([1 1],:);
        logProbBoth = min(logProbBoth, 0);
        emissionProbs = exp(logProbBoth);
    end
    
end


function popLocLogml = computeLocuswisePopulationLogml(popCounts, popSumCounts, adjustedPriorCounts, priorSumCounts)
    % Returns 1*nLoci table, in which logml values have been calculated for a
    % cluster whose counts are given in "popCounts", and "popSumCounts".

    popLocLogml = ...
        gammaln(priorSumCounts) ...
        - gammaln(popSumCounts+priorSumCounts) ...
        + sum(gammaln(popCounts+adjustedPriorCounts)) ...
        - sum(gammaln(adjustedPriorCounts));
end