function emissionProbs = calcEmissionProbsForClusterMerging(counts, ...
    sumCounts, adjustedPriorCounts, priorSumCounts, clusterCounts, clusterSumCounts)
    % The function calculates emission probabilities for the observed SNPs 
    % in two clusters. The first hidden state corresponds to situation that
    % the two clusters are equal. The second hidden state corresponds to the
    % situation that the clusters are different.
    %
    % "counts" contains the counts of observed SNPs in different clusters.
    %
    % "sumCounts" contains the total count of observed SNPs in different
    % positions in different clusters.
    %
    % "adjustedPriorCounts" and "priorSumCounts" are analogous to the previous two, 
    % except that these contain the prior counts (parameters of the Dirichlet
    % distributions)
    %
    % "dataRow" contains the observed SNP values.
    %
    % USAGE EXAMPLE:
%     counts(:,:,1) = [100 200 200; 400 400 200; 100 0 200];
%     counts(:,:,2) = [0 0 1; 0 1 1; 2 0 0];
%     counts(:,:,3) = [0 0 0; 0 0 0; 0 0 0];
%     sumCounts = [600 600 600;2 1 2;0 0 0];
%     adjustedPriorCounts = [1/3 1/2 1/3;1/3 1/2 1/3;1/3 1 1/3];
%     priorSumCounts = ones(1,3);
%     clusterCounts = [20 20 10; 180 0 20; 20 30 10];
%     clusterSumCounts = [210 50 40];
%     emissionProbs = calcEmissionProbsForClusterMerging(counts, sumCounts, adjustedPriorCounts, priorSumCounts, clusterCounts, clusterSumCounts);


    [maxNoalle, nLoci, npops] = size(counts);
    emissionProbs = zeros(npops,nLoci);

    for popIndex = 1:npops
        % Compute the emission probabilities that the cluster under
        % consideration emreged from the merged cluster with popIndex.
        
        emissionProbs(popIndex,:) = computeLocuswisePopulationLogml( ...
            counts(:,:,popIndex) + clusterCounts, ...
            sumCounts(popIndex,:) + clusterSumCounts, ...
            adjustedPriorCounts, priorSumCounts);
        
        denominator = computeLocuswisePopulationLogml( ...
            counts(:,:,popIndex), sumCounts(popIndex,:), ...
            adjustedPriorCounts, priorSumCounts);
        
        emissionProbs(popIndex,:) = emissionProbs(popIndex,:) - denominator;
    end
    emissionProbs = exp(emissionProbs);
    
end


function popLocLogml = computeLocuswisePopulationLogml(popCounts, popSumCounts, adjustedPriorCounts, priorSumCounts)
    % Returns 1*nLoci table, in which logml values have been calculated for a
    % cluster whose counts are given in "popCounts", and "popSumCounts".

    popLocLogml = ...
        gammaln(priorSumCounts) ...
        - gammaln(popSumCounts+priorSumCounts) ...
        + sum(gammaln(popCounts+adjustedPriorCounts)) ...
        - sum(gammaln(adjustedPriorCounts));
end