function fastGEAR(inputFile, outputFile, inputSpecsFile)
% Running Rafal's data:
% inputFile = 'C:\Work\Brat_all\Brat4\testData\ser23_data.aln';
% outputFile = 'C:\Work\Brat_all\Brat4\testData\ser23_results.mat';
% outputDir = 'C:\Work\Brat_all\Brat4\testData\ser23_output';
% partitionFile = [];
% nIterations = 3;
% fastGEAR(inputFile, outputFile, outputDir, partitionFile, nIterations, priorProbOfNoBreaks)
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Extract input specifications %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if nargin==2
        inputSpecsFile = 'fG_input_specs.txt';
    end
    % File containing a partition for strains
     
    [partitionFile, nIterations, nClustList, runAllUpperBounds, reducedOutput] = getSpecifications(inputSpecsFile);
    
    outputDir = inferOutputDir(outputFile); % <Directory containing the outputFile> \output
    
    possibleInitStates = {'dataLoaded', 'clusteringDone', 'clusterRelationshipsInspected', 'recProfilesCalculated', 'relationshipsReInspected'};
    
    if ~exist(outputDir)
        mkdir(outputDir);
    end
      
    
    %%%%%%%%%%%%%%%
    %% LOAD DATA %%
    %%%%%%%%%%%%%%%
    if isequal(inputFile(end-3:end), '.mat')
        % A result file of a previously crashed analysis is given as input.
        load(inputFile);
        initState = uData.state;
        uData.nIter = nIterations;
    else
        uData = loadFastaData(inputFile, outputFile, nIterations);
        initState = 'dataLoaded';
    end
    
    % Abort if only 0 or 1 SNPs
    if length(uData.snpPositions) <= 1
        uData = setFieldsToDefaultValues(outputFile, uData, outputDir);
        writeOutputs(outputFile, outputDir, uData);
        return;
    end
    
    startingOperationIndex = find(cellfun(@(x)isequal(x,initState), possibleInitStates));
    if ~isequal(initState, '')
        disp(['Starting analysis from state: ' initState]);
    end
    
    
    %%%%%%%%%%%%%%%%%%%%%%
    %% LEARN CLUSTERING %%
    %%%%%%%%%%%%%%%%%%%%%%
    if startingOperationIndex < 2
        
        % Always use as input file (the first argument), the output of the
        % previous step. Further, overwrite the input file with the updated
        % results.
        if isequal(partitionFile, '-')
            % Learn partition using BAPS clustering
            uData = learnClustering(outputFile, outputFile, nClustList, runAllUpperBounds);
        else
            % Load the given partition and use it.
            partition = load(partitionFile);
            uData.partition = partition;
            uData.state = 'clusteringDone';
            save(outputFile, 'uData');
        end
        
    end
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% INSPECT RELATIONSHIPS BETWEEN CLUSTERS %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if startingOperationIndex < 3
        
        if isequal(partitionFile, '-')

            disp('Inferring lineages by combining clusters');

            analyseLineages = false;
            uData = inspectClusterRelationships(outputFile, outputDir, analyseLineages);
            % The function changes partition s.t. the largest cluster has the
            % largest index.
            uData.clusterDist = computeClusterDistances(outputFile, outputDir, analyseLineages);
            uData = createClusterGroups(uData); % Creates lineages and orders them according to their sizes.

            uData.outputDir = outputDir; % This is needed in the following function
            uData = lineageSimilaritiesToLineageStructure(uData, 2); % Adds clusterStructure to uData.

            uData.state = 'clusterRelationshipsInspected';
            save(outputFile, 'uData');
            
        else
            
            % If a partition has been given, use it as such to define
            % the lineages.
            disp(['Starting analysis with ' num2str(length(unique(partition))) ' pre-defined lineages.']);
            uData.groupedPartition = partition;
            uData.clusterDist = [];
            uData.clusterGroups = [];
            uData.clusterStructure = [];
            uData.outputDir = outputDir;
            uData.state = 'clusterRelationshipsInspected';
            save(outputFile, 'uData');
            
        end
        
    end
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% LEARN RECOMBINATIONS IN STRAINS %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %if startingOperationIndex < 4
        uData = learnRecombinationsInClusters(outputFile, outputFile, outputDir, reducedOutput);
    %end
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% INSPECT RELATIONSHIPS BETWEEN LINEAGES %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %if startingOperationIndex < 5
        
        analyseLineages = true;
        uData = inspectClusterRelationships(outputFile, outputDir, analyseLineages);
        % The function changes partition s.t. the largest cluster has the
        % largest index.
        uData = lineageSimilaritiesToLineageStructure(uData, 1); % Adds lineageStructure to uData.
        uData.state = 'relationshipsReInspected';
        save(outputFile, 'uData');
        
    %end
    
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% CLEAN NON-SIGNIFICANT ANCESTRAL RECOMBINATIONS %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %if startingOperationIndex < 6
        
        disp('Checking significance');
        uData = clearNonSignificantAncestralRecombinations(outputFile, 1); % adds lineageStructureCleaned to uData
        if isequal(partitionFile, '-')
            uData = clearNonSignificantAncestralRecombinations(outputFile, 2); % adds clusterStructureCleaned to uData
        else
            uData.clusterStructureCleaned = [];
        end
        uData = clearNonSignificantRecentRecombinations(outputFile); % adds popStructureCleaned to uData
        uData.state = 'significancesComputed';
        save(outputFile, 'uData');
        
    %end
    
    
    writeOutputs(outputFile, outputDir, uData);
    
end



function writeOutputs(outputFile, outputDir, uData)
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% WRITE PARTITION TO FILE %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    partitionOutputFile = [outputDir uData.delimiter 'lineage_information.txt'];
    writePartition(outputFile, partitionOutputFile); % Writes both the lineage and the BAPS cluster
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% WRITE RECENT AND ANCESTRAL RECOMBINATIONS %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    for outputType = 1:2
        if outputType == 1
            fileName = 'recombinations_recent';
        elseif outputType == 2 || outputType == 4
            fileName = 'recombinations_ancestral';
        elseif outputType == 3 || outputType == 5
            fileName = 'recombinations_cluster_level';
        end
        segmentsOutputFile = [outputDir uData.delimiter fileName '.txt'];
        writeRecombinationsToTextFile(outputFile, segmentsOutputFile, outputType);
    end

end
