function plotRecombinations(inputFile, type, orderInDendrogram)
    % type == 1: recent recombinations (popStructureCleaned)
    % type == 2: lineage-level (lineageStructureCleaned)
    % type == 3 cluster-level (clusterStructureCleaned)
    % type == 4 lineage-level, not cleaned (lineageStructure)
    % type == 5 cluster-level, not cleaned (clusterStructure)
    % type == 6: recent recombinations, not cleaned (updatedPopStructure)
    
    % orderInDendrogram == 1: use the original order of the sequences
    % orderInDendrogram == 2: order by the cluster label
    % orderInDendrogram == 'name of a file with strain labels in the order to plot'
    
    outputDir = inferOutputDir(inputFile);
    
    if nargin == 2
        orderInDendrogram = 1;
    end
    
    if ~isnumeric(type)
        type = str2double(type);
    end
    
    load(inputFile); % Loads variable uData
    
    if isequal(orderInDendrogram,1) || isequal(orderInDendrogram,'1')
        orderToShow = length(uData.partition):-1:1;
    elseif isequal(orderInDendrogram,2) || isequal(orderInDendrogram, '2')
        orderToShow = sortStrainsByPartition(uData.partition);
    else
        orderToShow = extractOrderFromFile(orderInDendrogram, uData.strainLabels);
    end
    
    
    if type == 1
        
        popStructureToPlot = uData.popStructureCleaned;
        partition = uData.groupedPartition;
        
    elseif type == 6
        
        popStructureToPlot = uData.updatedPopStructure;
        partition = uData.groupedPartition;
        
    else
        
        popStructureToPlot = derivePopStructureFromLineageStructure(uData, type);
        
        % Mark as missing all recent recombinations
        nSnps = length(uData.snpPositions);
        lineageStructureWithoutRecentRecombinations = uint8(repmat(uData.groupedPartition, [1 nSnps]));
        recentRecombinationInds = find(lineageStructureWithoutRecentRecombinations ~= uData.popStructureCleaned);
        
        if type == 2 || type == 4
            nClusters = max(uData.groupedPartition);
            popStructureToPlot(recentRecombinationInds) = nClusters + 2; % nClusters+1 is black, nClusters+2 is white.
            partition = uData.groupedPartition;
            
        elseif type == 3 || type == 5
            nClusters = max(uData.partition);
            popStructureToPlot(recentRecombinationInds) = nClusters + 2;
            partition = uData.partition;
            
        end        
    end
    
    load(uData.snpDataFileName); % Load 'snpData'
    missingnessTable = snpData<=0;
    clear snpData;
    
    uData.clusterStructure = [];
    uData.clusterMarginals = [];
    uData.c = [];
    minMissingLength = 1;
    
    h = plotPopStructure(partition, ...
        popStructureToPlot, orderToShow, ...
        uData.totalSequenceLength, uData.snpPositions, ...
        determineTickLocations(uData.totalSequenceLength), ...
        missingnessTable, [], minMissingLength);
    
    % partition, popStructure, orderInDendrogram, totalSequenceLength, snpPositions, sequenceTickValues, missingnessTable, cmap, minMissingLength, nClusters
    set(gca,'Box','on');
    set(gcf,'Position',[75 400 960 560]);
    
    writeOrderInPlot(orderInDendrogram, orderToShow, outputDir, uData.partition, uData.groupedPartition, uData.strainLabels);
    
end



function writeOrderInPlot(orderInDendrogram, orderToShow, outputDir, partition, groupedPartition, strainLabels)
    
    sysName = computer;
    if (isequal(sysName(1:2),'PC')) % Windows
        delimiter = '\';
    else
        delimiter = '/';
    end
    
    if isequal(orderInDendrogram,1) || isequal(orderInDendrogram,'1')
        orderingText = 'original_ordering';
    elseif isequal(orderInDendrogram,2) || isequal(orderInDendrogram, '2')
        orderingText = 'by_cluster';
    else
        orderingText = 'given_ordering';
    end
    
    fid = fopen([outputDir delimiter 'order_in_plot_' orderingText '.txt'], 'w');
    
    row = 'PlotIndex Strain  Cluster  Lineage  Name';
    fprintf(fid, '%s\n',row);
    nStrains = length(orderToShow);
    for rowIndex = nStrains:-1:1
        origStrainIndex = orderToShow(rowIndex);
        if origStrainIndex > 0
            cluster = partition(origStrainIndex);
            lineage = groupedPartition(origStrainIndex);
            fprintf(fid,['%-10.0f%-8.0f%-9.0f%-9.0f' strainLabels{origStrainIndex} '\n'], rowIndex, origStrainIndex, cluster, lineage);
        else
            fprintf(fid,['%-10.0f\n'], rowIndex);
        end
    end
    
    fclose(fid);
    
end



function orderToShow = extractOrderFromFile(orderInDendrogram, strainLabels)
    
    if exist(orderInDendrogram) ~= 2
        error('File with strain order does not exist');
    end
    
    orderedLabels = cell(10000,1);
    nOrderedLabels = 0;
    
    fid = fopen(orderInDendrogram,'r');
    lineFromFile = fgetl(fid);
    while ischar(lineFromFile)
        nOrderedLabels = nOrderedLabels + 1;
        orderedLabels{nOrderedLabels} = lineFromFile;
        lineFromFile = fgetl(fid);
    end
    fclose(fid);
    orderedLabels = orderedLabels(1:nOrderedLabels);
    
    orderToShow = zeros(nOrderedLabels,1);
    for labelIndex = 1:nOrderedLabels
        nameHere = orderedLabels{labelIndex};
        strainIndex = find(strcmp(nameHere, strainLabels));
        if ~isempty(strainIndex)
            orderToShow(labelIndex) = strainIndex;
        end
    end
    orderToShow = orderToShow(end:-1:1); % Because plotting starts from bottom.
end
