function uData = loadFastaData(inputFile, outputFile, nIterations)
% "inputFile" contains the FASTA alignment and has .aln extension. The file
% name should have full path.
%
% "outputFile" contains the uData created after loadind the data. The file
% should have .mat extension. The file name should have full path.

% if "priorProbOfNoBreaks" is not given, then the fixed default value=0.5
% will be used. If a negative value is given, then the parameter will be
% learned. Otherwise, the given value will be used.

if nargin == 2
    priorProbOfNoBreaks = [];
end

[strainLabels, snpData, snpPositions, totalSequenceLength] = ...
    readFastaData(inputFile);


snpData = uint8(snpData);

uData = makeInitialUserData;
uData.nIter = nIterations;
uData.nIterCompleted = 0;
uData.strainLabels = strainLabels;
uData.snpDataFileName = [outputFile(1:end-4), '_snpData.mat'];
uData.snpPositions = snpPositions;
uData.totalSequenceLength = totalSequenceLength;
uData.transPars.rho0 = 0.5^(1/totalSequenceLength);
uData.priorProbOfNoBreaks = 0.5; % This will be used when inspecting cluster relationships
uData.state = 'dataLoaded';
uData.priorCounts = [1 1];
uData.ancestralLogBfThreshold = log(10);

sysName = computer;
if (isequal(sysName(1:2),'PC')) % Windows
    uData.delimiter = '\';
else
    uData.delimiter = '/';
end


save(outputFile, 'uData', '-v7.3');
save(uData.snpDataFileName,'snpData', '-v7.3');


function uData = makeInitialUserData
uData.probabilityThreshold = 0.05;
uData.alpha = 1;

uData.transPars.muMax = 20000;
uData.transPars.muMin = 10;
meanTractLength = (uData.transPars.muMax+uData.transPars.muMin)/2;
uData.transPars.rho = (meanTractLength-1)/meanTractLength;
uData.transPars.rho0 = 0.5^(1/1000000);  % Update after reading data.
uData.transPars.aPriorCount = 10;  % These values can't be changed.
uData.transPars.negAPriorCount = 1.5;  % These values can't be changed.
uData.transPars.a = uData.transPars.aPriorCount/(uData.transPars.aPriorCount+uData.transPars.negAPriorCount);
