function uData = learnClustering(inputFile, outputFile, nClustList, runAllUpperBounds)
% "inputFile" is the output file from function "loadFastaData". Full path
% should be given.
%
% "outputFile" contains the uData created after learning the PSA tree and
% learning the clustering by cutting the tree. The file should have .mat
% extension, and full path should be given.
%
% "nClusters" is the number of clusters and it determines where the PSA 
% tree will be cut. If the tree is completely flat, all sequences will be
% clustered as singletons (this has no consequences on subsequent 
% analyses).
%
% "alpha" is the hyperparameter. If it is to be learned, alpha should be
% set to -1. Otherwise a positive value should be given, which will be used
% as the fixed value.

load(inputFile); % Loads variable uData
load(uData.snpDataFileName); % Load variable snpData

disp('Clustering strains');

% Run all in one block. Therefore, blockWidth==totalSequenceLength
if runAllUpperBounds
    
    initialNumPops = nClustList;
    [partitionArray, ~, ~] = ...
        doBapsClusteringsInBlocks(snpData, uData.snpPositions, ...
        uData.totalSequenceLength, 0, ...
        uData.totalSequenceLength, uData.alpha, initialNumPops);
    
else
    
    nUpperBounds = length(nClustList);
    upperBoundIndex = 0;
    
    while (upperBoundIndex == 0) || ...
            ((length(unique(partitionArray)) == nClustList(upperBoundIndex)) && (upperBoundIndex < nUpperBounds))
        % Stop when the first upper bound has been found such that the
        % learned number of clusters is less than the upper bound, or when
        % the last upper bound has already been analysed.
        
        upperBoundIndex = upperBoundIndex + 1;
        
        % If maximum number of populations, run with a larger number.
        initialNumPopsNow = nClustList(upperBoundIndex);
        [partitionArray, ~, ~] = ...
            doBapsClusteringsInBlocks(snpData, uData.snpPositions, ...
            uData.totalSequenceLength, 0, ...
            uData.totalSequenceLength, uData.alpha, initialNumPopsNow);
    end
    
end

uData.partition = partitionArray;
uData.state = 'clusteringDone';

% Reorder cluster labels such that the smallest cluster has label 1 and so on.
uData.partition = reorderClusterLabels(uData.partition);

save(outputFile, 'uData', '-v7.3');

disp([' Number of clusters detected: ' num2str(length(unique(uData.partition)))]);
