function [partitionArray nSnpsArray npopsArray] = ...
    doBapsClusteringsInBlocks(snpData, snpPositions, totalSequenceLength, ...
    nSnpsLowerLimit, blockWidth, alpha, initialNumPops)
% Carries out BAPS clustering at nonoverlapping intervals of length
% "blockWidth". If some block has < "nSnpsLowerLimit" SNPs, this block will 
% be combined with the block coming after it. This will be repeated until 
% "nSnpsLowerLimit" SNPs or more are included in the analysis.
%
% "snpData" nStrains*nSnps data matrix (A=1,C=2,G=3,T=4,miss=0).
%
% "snpPositions", 1*nSnps, the positions of the SNPs in the genome.
%
% "totalSequenceLength", total length of the genomes.
%
% "partitionArray" nStrains*nBlocks matrix, where the i:th column contains
% partition within the i:th block.
%
% "nSnpsArray", 1*nBlocks array, specifies the numbers of SNPs in different
% blocks.
%
% "nPopsArray", 1*nBlocks array, specifies the numbers of clusters in
% partitions in different blocks.
%
% "alpha" is the hyperparameter representing the sum of Dirichlet 
% parameters
% 
% TESTING EXAMPLE:
% cd ..\dataProcessing
% [strainLabels, data, snpPositions, totalSequenceLength] = ...
%   readFastaData('..\data\testData\fastaTest.aln');
% 
% cd ..\parallelBaps
% [partitionArray nSnpsArray npopsArray] = ...
%    doBapsClusteringsInBlocks(data, snpPositions, totalSequenceLength, ...
%    2, 2);
% 
% isequal(partitionArray(:,1),partitionArray(:,2))
% partitionArray(1,4) ~= partitionArray(2,4)
% partitionArray(3,4) ~= partitionArray(4,4)

if nargin==5
    alpha=1;
    initialNumPops = [];
elseif nargin==6
    initialNumPops = [];
end

global PARTITION

nBlocks = ceil(totalSequenceLength./blockWidth);
nSequences = size(snpData,1);
npopsArray = zeros(1,nBlocks);
partitionArray = zeros(nSequences,nBlocks);
nSnpsArray = zeros(1,nBlocks);

indicesOfTooSmallBlocks = [];
cOld = [];

for blockIndex = 1:nBlocks
    
    c = getBapsInputBlockFromSnpData(blockIndex, blockWidth, ...
        snpData, snpPositions, totalSequenceLength, alpha);
    
    nSnpsArray(blockIndex) = length(c.noalle);
    
    if ~isempty(cOld)
        % Previous data set(s) was (were) too small. Concatenate with the
        % current data set.
        if size(c.adjprior,1)==size(cOld.adjprior,1)
            c.adjprior = [cOld.adjprior c.adjprior];
        elseif size(c.adjprior,1)>size(cOld.adjprior,1)
            diff = size(c.adjprior,1)-size(cOld.adjprior,1);
            aux = [cOld.adjprior; ones(diff,length(cOld.noalle))];
            c.adjprior = [aux c.adjprior];
        else
            diff = size(cOld.adjprior,1)-size(c.adjprior,1);
            aux = [c.adjprior; ones(diff,length(c.noalle))];
            c.adjprior = [cOld.adjprior aux];
        end
        c.priorTerm = cOld.priorTerm + c.priorTerm;
        c.data = [cOld.data c.data];
        c.noalle = [cOld.noalle c.noalle];
    end
        
    if length(c.noalle)>=nSnpsLowerLimit || blockIndex==nBlocks
        % Analyze the concatenated data
        
        % Remove sequences without any observed values:
        emptySeqs = find(all(c.data==0,2));
        c.data(emptySeqs,:) = [];
        
        if ~isempty(c.data)
            c = updateStruct(c);
            
            if isempty(initialNumPops)
                % Use 5 as the default upper bound for the number of clusters:
                [logml, npops, partitionSummary] = indMix(c,5,0, alpha);
                if npops==5  % If 5 pops were discovered, try with a higher value.
                    [logml, npops, partitionSummary] = indMix(c,10,0, alpha);
                    if npops==10
                        [logml, npops, partitionSummary] = indMix(c,20,0, alpha);
                        if npops==20
                            [logml, npops, partitionSummary] = indMix(c,40,0, alpha);   % 40 is the maximum allowed number of clusters.
                        end
                    end
                end
                partition = PARTITION;  % The global variable PARTITION was updated by indMix.
            else
                
                logmlList = zeros(1, length(initialNumPops));
                partitionList = zeros(size(c.data,1), length(initialNumPops));
                for initPopIndex = 1:length(initialNumPops)
                    initPopsNow = initialNumPops(initPopIndex);
                    %disp(['Started clustering with ' num2str(initPopsNow) ' clusters']);
                    [logml, npops, partitionSummary] = indMix(c,initPopsNow,0, alpha);
                    partitionList(:,initPopIndex) = PARTITION;  % The global variable PARTITION was updated by indMix.
                    logmlList(initPopIndex) = logml;
                end
                [maxLogml, maxIndex] = max(logmlList);
                partition = partitionList(:, maxIndex);
                clear logmlList partitionList;
            end
            
        else
            % c.data can be empty only when blockIndex==nBlocks
            % Use the last partition calculated from non-empty data.
            lastNonEmptyIndex = min([indicesOfTooSmallBlocks blockIndex])-1;
            partition = partitionArray(:,lastNonEmptyIndex);
            npops = length(unique(partition));
            emptySeqs = [];   % All strains are assigned to a cluster!
        end
        
        % Add sequences without any observed values to a random cluster:
        if ~isempty(emptySeqs)
            nonEmptySeqs = setdiff(1:nSequences,emptySeqs);
            partitionNew = zeros(nSequences,1);
            partitionNew(nonEmptySeqs) = partition;
            partitionNew(emptySeqs) = ceil(rand(length(emptySeqs),1)*npops);
            partition = partitionNew; clear partitionNew;
        end
        
        partitionArray(:,[indicesOfTooSmallBlocks blockIndex]) = repmat(partition,[1 1+length(indicesOfTooSmallBlocks)]);
        npopsArray([indicesOfTooSmallBlocks blockIndex]) = npops*ones(1,1+length(indicesOfTooSmallBlocks));
        %disp([num2str(blockIndex) '/' num2str(nBlocks) ' segments clustered.']);
        %disp([num2str([indicesOfTooSmallBlocks blockIndex]) ' ' num2str(toc)]);
        
        cOld = [];
        indicesOfTooSmallBlocks = [];
    else
        cOld = c;
        if isempty(indicesOfTooSmallBlocks)
            indicesOfTooSmallBlocks = blockIndex;
        else
            indicesOfTooSmallBlocks = [indicesOfTooSmallBlocks blockIndex];
        end
    end
end