#include "graph.h"
#include <assert.h>


graph::graph(uint32_t cnt) : m_nodes(cnt), m_edges(cnt * (cnt - 1)), m_edgecnt(0)
{
	for (uint32_t i = 0 ; i < cnt; i++) {
		m_nodes[i].index = i;
		m_nodes[i].in.degree = 0;
		TAILQ_INIT(&m_nodes[i].in.edges);
		TAILQ_INIT(&m_nodes[i].in.essentials);
		m_nodes[i].out.degree = 0;
		TAILQ_INIT(&m_nodes[i].out.edges);
		TAILQ_INIT(&m_nodes[i].out.essentials);
	}
}

void
graph::reset_edges()
{
	for (uint32_t i = 0 ; i < nodecnt(); i++) {
		m_nodes[i].in.degree = 0;
		TAILQ_INIT(&m_nodes[i].in.edges);
		TAILQ_INIT(&m_nodes[i].in.essentials);
		m_nodes[i].out.degree = 0;
		TAILQ_INIT(&m_nodes[i].out.edges);
		TAILQ_INIT(&m_nodes[i].out.essentials);
	}

	for (uint32_t i = 0 ; i < m_edges.size(); i++)
		m_edges[i].isactive = false;

	m_edgecnt = 0;
}



void
graph::bind_edge(uint32_t from, uint32_t to)
{
	uint32_t ind = edge_index(from, to);
	assert(ind < m_edges.size());
	assert(!m_edges[ind].isactive);

	m_edges[ind].out.target = to;
	m_edges[ind].in.target = from;
	m_edges[ind].isactive = true; // XXX make sure that this is inactivated during unbinding

	m_edgecnt++;

	edge *e = TAILQ_LAST(&m_nodes[from].out.edges, edgelist);
	while (e && e->out.target > to)
		e = TAILQ_PREV(e, edgelist, out.entries);

	if (e)
		TAILQ_INSERT_AFTER(&m_nodes[from].out.edges, e, &m_edges[ind], out.entries);
	else
		TAILQ_INSERT_HEAD(&m_nodes[from].out.edges, &m_edges[ind], out.entries);

	e = TAILQ_LAST(&m_nodes[to].in.edges, edgelist);
	while (e && e->in.target > from)
		e = TAILQ_PREV(e, edgelist, in.entries);

	if (e)
		TAILQ_INSERT_AFTER(&m_nodes[to].in.edges, e, &m_edges[ind], in.entries);
	else
		TAILQ_INSERT_HEAD(&m_nodes[to].in.edges, &m_edges[ind], in.entries);
}


void
graph::unbind_edge(uint32_t from, uint32_t to)
{
	uint32_t ind = edge_index(from, to);
	edge *e = &m_edges[ind];
	assert(e->isactive);
	e->isactive = false;

	//printf("sieving: %d %d %d %p %d\n", from, to, ind, e, e->isactive);

	TAILQ_REMOVE(&m_nodes[from].out.edges, e, out.entries);
	TAILQ_REMOVE(&m_nodes[to].in.edges, e, in.entries);
	m_edgecnt--;

}


void
graph::findchildren(uint32_t n, boolvector & visited, const graph & g, bool count, boolvector & out) const
{
	if (visited[n] && (!count || out[n])) return;
	visited[n] = true;

	if (count) out[n] = true;

	edge *e, *ref;
	ref = TAILQ_FIRST(&g.m_nodes[n].out.edges);

	TAILQ_FOREACH(e, &m_nodes[n].out.edges, out.entries) {
		bool c = count;
		if (!c) {
			while (ref && ref->out.target < e->out.target) ref = TAILQ_NEXT(ref, out.entries);
			if (ref && ref->out.target == e->out.target) c = true;
		}
		findchildren(e->out.target, visited, g, c, out);
	}
}

void
graph::closure(const graph *ref)
{
	if (!ref) ref = this;

	//std::vector<boolvector> children(nodecnt(), nodecnt());
	std::vector<boolvector> children(nodecnt());

	for (uint32_t i = 0; i < nodecnt(); i++) {
		children[i] = boolvector(nodecnt());
		boolvector visited(nodecnt(), false);
		ref->findchildren(i, visited, *this, false, children[i]);
		children[i][i] = false;
	}

	reset_edges();

	for (uint32_t i = 0; i < nodecnt(); i++) {
		for (uint32_t j = 0; j < nodecnt(); j++) {
			if (children[i][j]) { bind_edge(i, j); }
		}
	}

	build_essentials();
}




void
graph::build_essentials()
{
	for (uint32_t i = 0 ; i < m_nodes.size(); i++) {
		m_nodes[i].in.degree = 0;
		m_nodes[i].out.degree = 0;
	}

	for (uint32_t i = 0 ; i < m_nodes.size(); i++) {
		node * n = &m_nodes[i];
		TAILQ_INIT(&n->out.essentials);

		edge *e;

		TAILQ_FOREACH(e, &n->out.edges, out.entries) {
			node *m = &m_nodes[e->out.target];
			n->out.degree++;
			m->in.degree++;
			TAILQ_INSERT_TAIL(&n->out.essentials, e, out.essential);
			e->isess = true;
		}

		TAILQ_FOREACH(e, &n->out.edges, out.entries) {
			node *m = &m_nodes[e->out.target];

			edge *c = TAILQ_FIRST(&m->out.edges);
			edge *ess = TAILQ_FIRST(&n->out.essentials);

			while (ess && c) {
				if (ess->out.target < c->out.target)
					ess = TAILQ_NEXT(ess, out.essential);
				else if (ess->out.target > c->out.target)
					c = TAILQ_NEXT(c, out.entries);
				else {
					edge *t = ess;
					//printf("unessential: %d %d\n", ess->in.target, ess->out.target);
					ess = TAILQ_NEXT(ess, out.essential);
					c = TAILQ_NEXT(c, out.entries);

					t->isess = false;
					TAILQ_REMOVE(&n->out.essentials, t, out.essential);
				}
			}
		}
	}

	for (uint32_t i = 0 ; i < m_nodes.size(); i++) {
		node * n = &m_nodes[i];
		edge *e;
		TAILQ_FOREACH(e, &n->out.essentials, out.essential) {
			node *m = &m_nodes[e->out.target];
			TAILQ_INSERT_TAIL(&m->in.essentials, e, in.essential);
		}
	}
}


bool
graph::adjacent(uint32_t from, uint32_t to) const
{
	return m_edges[edge_index(from, to)].isactive;
}

bool
graph::essential(uint32_t from, uint32_t to) const
{
	return m_edges[edge_index(from, to)].isactive && m_edges[edge_index(from, to)].isess;
}



bool
graph::subset(const graph *g) const
{
	for (uint32_t i = 0; i < nodecnt(); i++) {
		const node *n1 = &m_nodes[i];
		const node *n2 = &g->m_nodes[i];
		const edge *e1;
		const edge *e2 = TAILQ_FIRST(&n2->out.edges);
		TAILQ_FOREACH(e1, &n1->out.edges, out.entries) {
			while (e2 && e2->out.target < e1->out.target)
				e2 = TAILQ_NEXT(e2, out.entries);
			if (!e2 || e2->out.target > e1->out.target)
				return false;
		}
	}
	return true;
}





void
graph::print(FILE *f, const graph & ref) const
{
	uint32_t ecnt = 0;
	for (uint32_t i = 0; i < nodecnt(); i++) {
		edge *e;
		TAILQ_FOREACH(e, &m_nodes[i].out.essentials, out.essential)
			ecnt++;
	}

	fprintf(f, "size: %d %d\n", nodecnt(), ecnt);

	fprintf(f, "edges:");

	for (uint32_t i = 0; i < nodecnt(); i++) {
		edge *e;
		TAILQ_FOREACH(e, &m_nodes[i].out.essentials, out.essential)
			fprintf(f, " %d -> %d", i, e->out.target);
	}

	fprintf(f, "\n");

	/*
	for (uint32_t i = 0; i < nodecnt(); i++) {
		edge *e;
		TAILQ_FOREACH(e, &m_nodes[i].out.edges, out.entries) {
			//if (!e->isloop)
				fprintf(f, " %d -> %d (%d %d)", i, e->out.target, ref.m_representative[i], ref.m_representative[e->out.target]);
		}
	}
	fprintf(f, "\n");
	*/

}



void
graph::init_degree(intvector & degree, intlist & out) const
{
	for (uint32_t i = 0; i < m_nodes.size(); i++) {
		degree[i] = m_nodes[i].in.degree;
		if (degree[i] == 0)
			out.push_back(i);
	}
}


void
graph::lower_degree(uint32_t n, intvector & degree, intlist & out) const
{
	const edge *e;
	TAILQ_FOREACH(e, &m_nodes[n].out.edges, out.entries) {
		uint32_t m = e->out.target;
		assert(degree[m] > 0);
		degree[m]--;
		if (degree[m] == 0) {
			out.push_back(m);
			//printf("Add: %d\n", m);
		}
	}
}

