#include "defines.h"
#include "episode.h"
#include <getopt.h>

#include <sys/time.h>

#include <algorithm>

bool
episodecompare(const episode *ep1, const episode *ep2)
{
	return std::lexicographical_compare(ep1->events().begin(), ep1->events().end(), ep2->events().begin(), ep2->events().end());
}


int
main(int argc, char **argv)
{
	static struct option longopts[] = {
		{"win",             required_argument,  NULL, 'w'},
		{"out",             required_argument,  NULL, 'o'},
		{"in",              required_argument,  NULL, 'i'},
		{"data",            required_argument,  NULL, 'd'},
		{"sparse",          no_argument,        NULL, 's'},
		{"minwin",          no_argument,        NULL, 'c'},
		{"help",            no_argument,        NULL, 'h'},
		{ NULL,             0,                  NULL,  0 }
	};

	char *inname = NULL;
	char *outname = NULL;
	char *timename = NULL;
	char *seqname = NULL;
	params par;

	int ch;
	while ((ch = getopt_long(argc, argv, "d:w:o:i:smcht:", longopts, NULL)) != -1) {
		switch (ch) {
			case 'h':
				printf("Usage: %s -w <window> -i <input file> -d <sequence file> -o <output file> [-scm]\n", argv[0]);
				printf("  -h    print this help\n");
				printf("  -w    size of the window\n");
				printf("  -i    input file\n");
				printf("  -o    output file\n");
				printf("  -d    sequence file\n");
				printf("  -s    sequence file is in sparse format\n");
				printf("  -m    sequence file has multiple sequences (-1 as seprator), currently must be true\n");
				printf("  -c    count support (currenty ignored)\n");

				return 0;
				break;
			case 'w':
				par.winsize = atoi(optarg);
				break;
			case 'i':
				inname = optarg;
				break;
			case 'd':
				seqname = optarg;
				break;
			case 'o':
				outname = optarg;
				break;
			case 'c':
				par.supmode = params::st_sid;
				break;
			case 's':
				par.sparse = true;
				break;
			case 'm':
				par.sid = true;
				break;
			case 't':
				timename = optarg;
				break;
		}
	}

	if (inname == NULL) {
		fprintf(stderr, "Missing input file\n");
		return 1;
	}

	if (outname == NULL) {
		fprintf(stderr, "Missing output file\n");
		return 1;
	}

	if (seqname == NULL) {
		fprintf(stderr, "Missing sequence file\n");
		return 1;
	}


	sequence s(seqname, par);

	episodelist episodes = read(s, inname);

	episodevector ordered(episodes.begin(), episodes.end());

	std::sort(ordered.begin(), ordered.end(), episodecompare);

	FILE *out = fopen(outname, "w");

	doublevector timespent(ordered.size());

	for (uint32_t i = 0; i < ordered.size(); i++) {
		fprintf(stderr, "%i\r", i);
		timeval t1, t2;

		episode *ep = ordered[i];
		ep->support_count(s);
		ep->testind(s);

		gettimeofday(&t1, NULL);

		ep->test(s);

		gettimeofday(&t2, NULL);
	
		timespent[i] = (t2.tv_sec - t1.tv_sec) * 1000.0;
		timespent[i] += (t2.tv_usec - t1.tv_usec) / 1000.0;   
	}

	for (uint32_t i = 0; i < ordered.size(); i++) {
		fprintf(stderr, "%i\r", i);
		timeval t1, t2;

		gettimeofday(&t1, NULL);

		episode *ep1 = ordered[i];
		for (uint32_t j = i + 1; j < ordered.size() && ep1->events() == ordered[j]->events(); j++) {
			episode *ep2 = ordered[j];
			if (ep1->edges().subset(&ep2->edges()))
				ep1->testsuper(ep2, s);
			else if (ep2->edges().subset(&ep1->edges()))
				ep2->testsuper(ep1, s);
		}

		gettimeofday(&t2, NULL);

		timespent[i] += (t2.tv_sec - t1.tv_sec) * 1000.0;
		timespent[i] += (t2.tv_usec - t1.tv_usec) / 1000.0;   

	}


	for (uint32_t i = 0; i < ordered.size(); i++)
		ordered[i]->print(out);

	fclose(out);

	if (timename) {
		FILE *tout = fopen(timename, "w");
		for (uint32_t i = 0; i < ordered.size(); i++) {
			episode *ep = ordered[i];

			fprintf(tout, "%d %d %d %d %f\n", ep->edges().nodecnt(), ep->edges().edgecnt(), ep->machnodecnt(), ep->machedgecnt(), timespent[i]);
		}
		fclose(tout);
	}
}
