#ifndef SEQUENCE_H
#define SEQUENCE_H

#include "defines.h"

#include <vector>
#include <map>
#include <string>
#include "queue.h"

struct params;


class sequence {
	public:

		sequence(const char *name, const params & p);


		struct stamp {
			uint32_t sid;
			index_t index;

			bool operator < (const stamp & a) const {return sid < a.sid || (sid == a.sid && index < a.index);}
			bool operator > (const stamp & a) const {return sid > a.sid || (sid == a.sid && index > a.index);}
			bool operator == (const stamp & a) const {return sid == a.sid && index == a.index;}
		};

		struct event {
			stamp s;
			uint32_t id; // Unique id
			symbol_t label; // Label

			//event* lastsid; // last event with the same sid

			event *prev() {return TAILQ_PREV(this, sequence::eventhead, entries);}
			event *next() {return TAILQ_NEXT(this, entries);}

			const event *prev() const {return TAILQ_PREV(this, sequence::eventhead, entries);}
			const event *next() const {return TAILQ_NEXT(this, entries);}

			event *other; // counter event when dealing with intervals

			TAILQ_ENTRY(event) entries;
			TAILQ_ENTRY(event) similar;

			const event * first() const {return this;} // needed when computing support
			const event * last() const {return this;} // needed when computing support
			bool minimal() const {return true;} // needed when computing support
		};

		TAILQ_HEAD(eventhead, event);

		const eventhead & events(symbol_t s) const {return m_events[s];}

		symbol_t eventcnt() const {return m_events.size();} 

		const std::string & label(uint32_t i) const {return m_labels[i];}
		symbol_t
		index(const std::string & label) const
		{
			symmap::const_iterator it = m_labelmap.find(label);
			assert(it != m_labelmap.end());
			return it->second;
		}


		uint32_t count(uint32_t i) const {return m_counts[i];}
		uint32_t total() const {return m_total;}
		
		uint32_t rowcount() const {return m_sequence.size();}

		const eventhead & row(uint32_t i) const {return m_sequence[i];}
		index_t
		length(uint32_t i) const
		{
			if (TAILQ_EMPTY(&m_sequence[i])) return 0;
			return 1 + TAILQ_LAST(&m_sequence[i], eventhead)->s.index - TAILQ_FIRST(&m_sequence[i])->s.index;
		}

		index_t headlength(const event *e) const {return e->s.index - TAILQ_FIRST(&m_sequence[e->s.sid])->s.index;}
		index_t taillength(const event *e) const {return TAILQ_LAST(&m_sequence[e->s.sid], eventhead)->s.index - e->s.index;}


	protected:
		bool isfrequent(const params & p, symbol_t s) const;


		typedef std::vector<eventhead> eventmap;

		void read_full(const char *name, bool sid);
		void read_sparse(const char *name, bool sid);
		void read_interval(const char *name, bool sid);

		void compute_counts();

		typedef std::map<std::string, uint32_t> symmap;

		eventmap m_events;

		eventmap m_sequence;

		std::vector<event> m_data;
		stringvector m_labels;

		intvector m_counts;
		uint32_t m_total;

		symmap m_labelmap;
};

class substringiterator {
	public:
		substringiterator(const sequence & s, const symbolvector & labels);

		const sequence::event * operator *() const {return m_heap.front();}
		const sequence::event * operator ->() const {return m_heap.front();}

		substringiterator & operator ++();

		bool done() const {return m_size == 0;}

	protected:
		typedef std::vector<const sequence::event *> eventheap; 
		eventheap m_heap;
		uint32_t m_size;
};



struct params  {
	params() : winsize(0), supmode(st_fixwin), sparse(false), sid(false) {}
	index_t winsize;

	enum suptype {st_fixwin, st_sid};
	suptype supmode;

	bool iswithin(index_t f, index_t l) const {return l - f <= winsize - 1;} // XXX remove -1 
	bool iswithin(const sequence::event & f, const sequence::event & l) const {return f.s.sid == l.s.sid && iswithin(f.s.index, l.s.index);} 

	uint32_t maxnodes;
	bool sparse;
	bool sid;
};



#endif
