%% Script for analysis of optimal window lengths in random data

% Copyright (c) 2015, Jefrey Lijffijt
%
% Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
%
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OU
T OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

% Init
K = 3:6;
M = 2.^(10:2:16);

%% Compute optimal window lengths
windows = cell(length(K),length(M));

parfor k = 1:length(K)
    windows_k = cell(1,length(M));
    for m = 1:length(M)
        windows_k{m} = kmedoidsppBernoulli(M(m), K(k), true);
        fprintf('%s\n',['Completed m = ' num2str(m) ', k = ' num2str(k) '.']);
    end
    windows(k,:) = windows_k;
end

clear k windows_k;

%% Save result
save('dami_bernoulli.mat','K','M','windows');

%% Load result
load('dami_bernoulli.mat','K','M','windows');

%% Plot results
figure('Name','Bernoulli Process','Units','centimeters',...
    'OuterPosition',[3 3 26 18],'Resize','off',...
    'PaperOrientation','landscape','PaperType','A4','Renderer','painters');

lm = 0.068;
rm = 0.01;
chm = 0.02;
width = (1-lm-rm-chm)/2;
bm = 0.095;
tm = 0.015;
cvm = 0.06;
height = (1-bm-tm-cvm)/2;

ls = {'o:','o--','o-.','o-'};

for k = 1:4
    switch k
        case 1
            hpos = lm;
            vpos = bm+height+cvm;
        case 2
            hpos = lm+width+chm;
            vpos = bm+height+cvm;
        case 3
            hpos = lm;
            vpos = bm;
        case 4
            hpos = lm+width+chm;
            vpos = bm;
    end
    axes('Position',[hpos vpos width height],'FontSize',14.0,'TickDir','out'); %#ok<LAXES>
    hold on;

    series = sort(reshape(vertcat(windows{k,:}),K(k),length(M)));
    plot(series(:,1),ls{1},'LineWidth',2.0,'MarkerSize',4,'MarkerFaceColor','b');
    hold all;
    plot(series(:,2),ls{2},'LineWidth',2.0,'MarkerSize',4,'MarkerFaceColor','g');
    plot(series(:,3),ls{3},'LineWidth',2.0,'MarkerSize',4,'MarkerFaceColor','r');
    plot(series(:,4),ls{4},'LineWidth',2.0,'MarkerSize',4,'MarkerFaceColor','c');
    
    
    set(gca,'YScale','log');
    xlim([1-(K(k)-1)*0.1 K(k)+(K(k)-1)*0.1]);
    ylim([1 max(M)]);
    set(gca,'XTick',1:K(k));
    set(gca,'YTick',2.^(0:log2(max(M))/4:log2(max(M))));
    set(gca,'TickDir','out');
    box off;
    
    text(0.5+K(k)/2,M(end)*0.4,['k = ' num2str(K(k))],...
        'HorizontalAlignment','center','FontSize',14);
    
    if(k == 1)
        legend({['m = ' num2str(M(1))],['m = ' num2str(M(2))],...
            ['m = ' num2str(M(3))],['m = ' num2str(M(4))]},...
            'Location','NorthWest');
        legend('boxoff');
    end
    if(ismember(k,[3,4]))
        xlabel('Window length (sorted)');
    end
    if(ismember(k,[1,3]))
        ylabel('Selected window lengths (2\^)');
        set(gca,'YTickLabel',0:log2(max(M))/4:log2(max(M)));
    else
        set(gca,'YTickLabel',[]);
    end
end

clear lm rm width bm tm height hpos vpos chm cvm ls k m series ans;