#!/usr/bin/env python

######################################################################
# Copyright (C) 2013 Jaakko Luttinen
#
# This file is licensed under Version 3.0 of the GNU General Public
# License.
######################################################################

######################################################################
# This script contains simple plotting for the paper
# 'Fast Variational Bayesian Linear State-Space Model'
# by Jaakko Luttinen (ECML 2013).
#
# See usage:
# ./plot_experiment.py --help
#
# For instance:
# ./plot_experiment.py --experiment=testbed --rotate
######################################################################

"""
Do some plotting for speed-up rotations demonstration.
"""

import sys, getopt

import h5py
import numpy as np

import run_experiment
import matplotlib.pyplot as plt

def plot_comparison(baseline_filename, rotations_filename):
    """
    Plot comparison of the results: lowerbound, training RMSE and test RMSE.
    """

    def do_plotting(filename, *args):
        f = h5py.File(filename, 'r')
        try:
            iters = np.arange(len(f['L'])) + 1
            plt.subplot(1,3,1)
            plt.semilogx(iters, f['L'], *args)
            plt.title('Lower bound')
            plt.subplot(1,3,2)
            plt.semilogx(iters, f['callback_output'][0], *args)
            plt.title('Training RMSE')
            plt.subplot(1,3,3)
            plt.semilogx(iters, f['callback_output'][1], *args)
            plt.title('Test RMSE')
        finally:
            f.close()

    plt.figure()
    # Baseline with black dashed line
    do_plotting(baseline_filename, 'k--')
    # Rotations with red solid line
    do_plotting(rotations_filename, 'r-')
    plt.subplot(1,3,1)
    plt.legend(['baseline', 'rotations'], loc='lower right')
    plt.subplot(1,3,2)
    plt.legend(['baseline', 'rotations'])
    plt.subplot(1,3,3)
    plt.legend(['baseline', 'rotations'])
        
    # Plot comparison of the lowerbound, training RMSE and test RMSE
    #run_experiment.compare_experiment_results(baseline_filename, rotations_filename)

if __name__ == '__main__':
    """
    This makes it possible to run the file as a script.
    """
    
    try:
        opts, args = getopt.getopt(sys.argv[1:],
                                   None,
                                   ["baseline=","rotations="])
    except getopt.GetoptError:
        print('run_experiment.py --baseline="<filename>" --rotations="<filename>"')
        print('')
        print('Shows a simple comparison plot')
        print('')
        print('--baseline="<filename>"\t The result file for the baseline method (required)')
        print('--rotations="<filename>"\t The result file for the rotations method (required)')
        sys.exit(2)
        
    for opt, arg in opts:
        if opt == "--rotations":
            rotations_filename = arg
        elif opt == "--baseline":
            baseline_filename = arg
            
    # Run experiment
    plot_comparison(baseline_filename, rotations_filename)
    plt.show()
