/* popsim: a simple population simulator
 * Copyright (C) 2004-2006  Jouni K. Seppänen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
*/

#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

#include <string.h>

#include "randfun.h"
#include "util.h"

static gsl_rng * rng = NULL;

static uint rng_seed = 0;
static const char *rng_type = "mt19937";

void 
randfun_set_seed(uint seed)
{
  rng_seed = seed;
}

void randfun_set_type(const char *type)
{
  rng_type = xstrdup(type);
}

void
randfun_init()
{
  const gsl_rng_type **t;
  for (t=gsl_rng_types_setup(); *t != 0; t++)
    if (strcmp(rng_type, (*t)->name) == 0) {
      rng = gsl_rng_alloc(*t);
      gsl_rng_set(rng, rng_seed);
    }
  if (rng == NULL) {
    fprintf(stderr, "error: rng not known: '%s'\n", rng_type);
    abort();
  }
}

void
randfun_cleanup()
{
  gsl_rng_free(rng);
}

void
randfun_write_config()
{
  fprintf(stderr, "rng type '%s'\nrng seed %u\n", rng_type, rng_seed);
}

uint
randrange(uint n)
{
  return gsl_rng_uniform_int(rng, n);
}

uint 
poisson(double mu)
{
  uint r = gsl_ran_poisson(rng, mu);
  DEBUG("Poisson(%f) => %u\n", mu, r);
  return r;
}

unsigned short 
coinflip()
{
  unsigned short r = gsl_rng_uniform_int(rng, 2);
  DEBUG("coinflip() => %u\n", r);
  return r;
}

void 
sample(uint arraysize, uint *array, uint howmany, uint *result)
{
  gsl_ran_choose(rng, result, howmany, array, arraysize, sizeof(uint));
}
