/* popsim: a simple population simulator
 * Copyright (C) 2004-2006  Jouni K. Seppänen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
*/

#define _GNU_SOURCE

#define VERSION "1.01"

#include <assert.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <strings.h>
#include <time.h>
#include <unistd.h>

#include "famtrack.h"
#include "params.h"
#include "randfun.h"
#include "util.h"


const uint CHROMO_END = UINT_MAX;

#define min(a,b)         \
  ({ typeof(a) _a = (a); \
     typeof(b) _b = (b); \
     _a < _b ? _a : _b; })
#define SWAP(a,b)         \
  { typeof(a) _tmp = (a); \
   (a) = (b);             \
   (b) = _tmp;            \
  }

static int ISATTY;

struct population {
  uint npeople;
  uint *family, *nends;
  uint **ends, **ids;
} *population, *children;

struct chromosome {
  uint nends;
  uint *ends;
  uint *ids;
};


/* Called from main() */
static void print_time(void);
static void print_generation(uint gen);
static void init_population(void); /* initialize POPULATION & CHILDREN*/
static void next_generation(uint nchildren); /* replace POPULATION */

/* Subprograms */
static struct chromosome *meiosis(uint person); /* returns pointer to static var! */
static uint *marry(uint *ncouples_p); /* finds spouses for people */
static void wipepop(struct population *which); /* frees some memory */
static void teardown(struct population **which); /* frees the rest */
static void print_percentage(int p);
#define PROGRESS(p,outof) do { if (ISATTY) print_percentage(100.0*(p)/(outof)); } while(0)
static void output_markers(FILE *file, struct population *pop);
void sample_nonsiblings(struct population *pop, uint n, struct population **smp);
void print_chromosome(uint nends, uint *ends, uint *ids);


int
main(int argc, char **argv)
{
  uint i;

  set_debugging();
  ISATTY = isatty(2);
  if (argc != 2) {
    fprintf(stderr, 
	    "Usage: popsim conf-file\n"
	    "\n"
	    "popsim version %s, Copyright (C) 2004-2006 Jouni K. Seppanen\n"
	    "This program comes with ABSOLUTELY NO WARRANTY.\n"
	    "This is free software, and you are welcome to redistribute it\n"
	    "under certain conditions. See the file COPYING for details\n"
	    "on the lack of warranty and your right to redistribute.\n",
	    VERSION);
    return 1;
  }
  if (read_conf(argv[1])) {
    return 1;
  }
  write_conf();
  randfun_init();
  plan_population();
  init_population();
  for (i=0; i<GENERATIONS; i++) {
    print_generation(i);
    next_generation(POPULATION_PLAN[i]);
  }
  print_generation(i);
  teardown(&children);
  free(POPULATION_PLAN);
  print_time(); fprintf(stderr, " Outputting...\n");
  do_output();
  print_time(); fprintf(stderr, " done\n");
  wipepop(population); teardown(&population);
  randfun_cleanup();

  return 0;
}

void 
print_time()
{
  time_t now = time(NULL);
  char *timestr = ctime(&now);
  *rindex(timestr, ' ') = '\0';
  fprintf(stderr, "%s", rindex(timestr, ' ')+1);
}

void
print_generation(uint gen)
{
  print_time();
  fprintf(stderr, " Generation %u: %u people\n",
	  gen, population->npeople);
#ifndef NDEBUG
  if (debug_output) {
    uint i;
    for (i=0; i<population->npeople; i++) {
      DEBUG("person %u (family %u):\n ", i, population->family[i]);
      print_chromosome(population->nends[2*i  ], 
		       population->ends[2*i  ], population->ids[2*i  ]);
      DEBUG("\n ");
      print_chromosome(population->nends[2*i+1], 
		       population->ends[2*i+1], population->ids[2*i+1]);
      DEBUG("\n");
    }
  }
#endif
}

void
print_percentage(int p)
{
  static int last = -1;
  if (p >= 0 && p != last)
    fprintf(stderr, "%3d%%\b\b\b\b", p);
  last = p;
}

void init_population()
{
  uint i;
  size_t size;
  
  population = xmalloc(sizeof(struct population));
  children = xmalloc(sizeof(struct population));

  population->npeople = INITIAL_POPULATION;
  size = FINAL_POPULATION*sizeof(uint);
  population->family = xmalloc(size);
  children->family = xmalloc(size);
  size = 2*FINAL_POPULATION*sizeof(uint);
  population->nends = xmalloc(size);
  children->nends = xmalloc(size);
  size = 2*FINAL_POPULATION*sizeof(uint *);
  population->ends = xmalloc(size);
  population->ids = xmalloc(size);
  children->ends = xmalloc(size);
  children->ids = xmalloc(size);
  
  for (i=0; i < population->npeople; i++) {
    population->family[i] = i;
  }
  for (i=0; i < 2*population->npeople; i++) {
    population->nends[i] = 1;
    population->ends[i] = xmalloc(1*sizeof(uint));
    population->ids[i] = xmalloc(1*sizeof(uint));
    population->ends[i][0] = CHROMO_END;
    population->ids[i][0] = i;
  }
}

void 
next_generation(uint nchildren)
{
  uint *couples;
  uint i, ncouples;

  children->npeople = nchildren;
  couples = marry(&ncouples);

  for (i=0; i < nchildren; i++) {
    uint couple = randrange(ncouples);
    struct chromosome *chr;
    int j;

    DEBUG("*** child %u born to %u and %u ***\n", i, couples[2*couple], couples[2*couple+1]);
    children->family[i] = min(couples[2*couple],couples[2*couple+1]);

    for (j=0; j < 2; j++) {
      chr = meiosis(couples[2*couple + j]);
      children->nends[2*i + j] = chr->nends;
      children->ends[2*i + j] = chr->ends;
      children->ids[2*i + j] = chr->ids;
    }
  }

  xfree(couples);
  wipepop(population);
  SWAP(population, children);
}

uint *
marry(uint *ncouples_p)
{
  uint nsingles = population->npeople, ncouples = 0, i;
  uint *singles = xmalloc(population->npeople*sizeof(uint));
  uint *spouses = xmalloc(population->npeople*sizeof(uint));
  uint *family = population->family;
  struct famtrack *ft;

  /* Initialize */
  for (i=0; i < population->npeople; i++) singles[i] = i;
  ft = ft_init(population->npeople, family);
  if (ft->nfamilies <= 1) {
    fprintf(stderr, "Extinction: all people are siblings\n");
    abort();
  }

  /* Pair everyone up. */
  while(1) {
    int mates[2];
    while(1) {			/* keep sampling until a legal pair is found */
      assert(nsingles >= 2);
      sample(nsingles, singles, 2, mates);
      if (family[mates[0]] != family[mates[1]])
	break;
    }
    DEBUG("Marriage: %u and %u ", mates[0], mates[1]);
    /* Remove mates from pool of singles. */
    ordremv(singles, &nsingles, mates[0]);
    ordremv(singles, &nsingles, mates[1]);
    DEBUG("(now %u singles left) ", nsingles);
    /* Record mates as a couple */
    spouses[2*ncouples] = mates[0];
    spouses[2*ncouples++ +1] = mates[1];
    DEBUG("(now %u married couples)\n", ncouples);
    /* Decrease family sizes. */
    ft_decrease(ft, family[mates[0]]);
    ft_decrease(ft, family[mates[1]]);
    if (ft->nfamilies <= 1) {		/* no legal pairs left */
      DEBUG("only %u famil%s left, stopping marriages\n",
	    ft->nfamilies,
	    ft->nfamilies == 1 ? "y" : "ies");
      break;
    }
    PROGRESS(population->npeople - nsingles, population->npeople);
  }

  xfree(singles);
  ft_delete(ft);
  *ncouples_p = ncouples;
  return spouses;
}

void
output_histogram(int full, unsigned int samplesize, char *filename)
{
  FILE *file;

  file = fopen(filename, "w");
  if (!file) {
    perror(filename);
    return;
  }
  if (full)
    output_markers(file, population);
  else {
    struct population *pop;
    sample_nonsiblings(population, samplesize, &pop);
    
    output_markers(file, pop);
    teardown(&pop);
  }
  if (fclose(file))
    perror(filename);
}

void
sample_nonsiblings(struct population *pop, uint n, struct population **smp)
{
  uint i;
  struct famtrack *ft = ft_init(pop->npeople, pop->family);
  assert(ft->nfamilies >= n);

  *smp = xmalloc(sizeof(struct population));
  (*smp)->npeople = n;
  (*smp)->family = xmalloc(n*sizeof(uint));
  (*smp)->nends = xmalloc(2*n*sizeof(uint));
  (*smp)->ends = xmalloc(2*n*sizeof(uint *));
  (*smp)->ids = xmalloc(2*n*sizeof(uint *));

  for (i=0; i < n; i++) {
    uint who, j;
    /* Be sure to select at most one from each family */
    do {
      who = randrange(population->npeople);
    } while (ft_isempty(ft, population->family[who]));
    ft_clear(ft, population->family[who]);
    (*smp)->family[i] = pop->family[who];
    for (j=0; j < 2; j++) {
      (*smp)->nends[2*i+j] = pop->nends[2*who+j];
      (*smp)->ends [2*i+j] = pop->ends [2*who+j];
      (*smp)->ids  [2*i+j] = pop->ids  [2*who+j];
    }
  }

  ft_delete(ft);
}

void 
output_markers(FILE *file, struct population *pop)
{
  uint **histogram = xmalloc(MARKERS*sizeof(uint*));
  uint *mrkpos = xmalloc(MARKERS*sizeof(uint));
  uint i;

  for (i=0; i<MARKERS; i++) {
    histogram[i] = xcalloc(2*INITIAL_POPULATION, sizeof(uint));
    mrkpos[i] = (uint) rint((double)i/(MARKERS-1) * CHROMO_END);
  }

  for (i=0; i < 2*pop->npeople; i++) {
    uint j, k=0, *ends = pop->ends[i], *ids = pop->ids[i];
    PROGRESS(i, 2*pop->npeople);
    for (j=0; j < MARKERS; j++) {
      while (ends[k] < mrkpos[j])
	k++;
      histogram[j][ids[k]]++;
    }
  }

  for (i=0; i < MARKERS; i++) {
    uint j;
    for (j=0; j < 2*INITIAL_POPULATION; j++)
      fprintf(file, "%u ", histogram[i][j]);
    fprintf(file, "\n");
    free(histogram[i]);
  }
  free(histogram); free(mrkpos);
}

void
print_chromosome(uint nends, uint *ends, uint *ids)
{
  uint i;
  fprintf(stderr, "<");
  for (i=0; i<nends; i++)
    fprintf(stderr, "%u -> %f%s", ids[i], (double)ends[i]/CHROMO_END,
	    i == nends-1 ? "" : ", ");
  fprintf(stderr, "> ");
}

struct chromosome *
meiosis(uint person)
{
# define PARENT(n,member) (population->member[2*person + (n)])
  static struct chromosome result;
  unsigned short source = coinflip();
  uint nchiasmas = poisson(1.0);
  uint *chiasmas = NULL;
  uint i, lastloc=0, pos[2]={0,0};
  
  size_t reserve = 
    (PARENT(0,nends) + PARENT(1,nends) + nchiasmas) 
    * sizeof(uint);
  
  result.ends = xmalloc(reserve);
  result.ids = xmalloc(reserve);
  result.nends = 0;
#ifndef NDEBUG
  if (debug_output) {
    fprintf(stderr, "Meiosis (%u chiasmas) -- parent chromosomes:\n", nchiasmas);
    for (i=0; i<2; i++) {
      print_chromosome(PARENT(i,nends), PARENT(i,ends), PARENT(i,ids));
      fprintf(stderr, "\n");
    }
  }
#endif

  if (!nchiasmas)
    goto fix_end;		/* Electric Fence dislikes malloc(0) */

  chiasmas = xmalloc(nchiasmas*sizeof(uint));
  for (i=0; i<nchiasmas; i++)
    chiasmas[i] = randrange(CHROMO_END);
  qsort(chiasmas, nchiasmas, sizeof(uint), compar_uint);

  for (i=0; i<nchiasmas; i++) {
    uint loc = chiasmas[i];
    while(PARENT(source,ends)[pos[source]] <= lastloc)
      pos[source]++;
    while(PARENT(source,ends)[pos[source]] <= loc) {
      result.ends[result.nends] = PARENT(source,ends)[pos[source]];
      result.ids[result.nends] =  PARENT(source,ids)[pos[source]];
      result.nends++;
      pos[source]++;
    }
    result.ends[result.nends] = loc;
    result.ids[result.nends] = PARENT(source,ids)[pos[source]];
    result.nends++;

    lastloc = loc;
    source = 1-source;
  }

 fix_end:
  while(PARENT(source,ends)[pos[source]] <= lastloc)
    pos[source]++;
  for (; pos[source] < PARENT(source,nends); pos[source]++) {
    result.ends[result.nends] = PARENT(source,ends)[pos[source]];
    result.ids[result.nends] = PARENT(source,ids)[pos[source]];
    result.nends++;
  }

  xfree(chiasmas);
  result.ends = xrealloc(result.ends, result.nends*sizeof(uint));
  result.ids = xrealloc(result.ids, result.nends*sizeof(uint));
#ifndef NDEBUG
  if (debug_output) {
    DEBUG("Resulting chromosome:\n");
    print_chromosome(result.nends, result.ends, result.ids);
    fprintf(stderr, "\n");
  }
#endif
  return &result;
# undef PARENT
}

void
wipepop(struct population * which)
{
  uint i;
  for(i=0; i<2*which->npeople; i++) {
    xfree(which->ends[i]);
    xfree(which->ids[i]);
  }
}

void
teardown(struct population **which)
{
  struct population *ptr = *which;

  free(ptr->family); 
  free(ptr->nends); 
  free(ptr->ends); 
  free(ptr->ids);
  free(ptr);
  *which = NULL;
}
