function [ G] = neighborhood( X, options )
%NEIGHBORHOOD Calculates neighborhoods based on euclidean coordinates for x
% The amount of neighbors for each point can be verified by typing
% sum(G,2).
% Mika Juuti, 2014-09-22

[n, d] = size(X);

if nargin <= 1
    options.Neighborhood.Type = 'knn';
    options.Neighborhood.k = round(log(size(X,1)));
end

if(~isfield(options,'Neighborhood'))
    error('options.Neighborhood required. Example: n.Type = "nn"')
end

if(~isfield(options.Neighborhood,'Type'))
    error('Neighborhood.Type required. Example: n.Type = "nn"')
end

if(~isfield(options,'Verbose'))
    options.Verbose = 0;
end

verbose = options.Verbose;
NeighborhoodNames = {'knn','mnn','epsilon','nn','nc'};

if ~ismember(options.Neighborhood.Type,NeighborhoodNames)
    error(['Built-in neighborhood types include: ' evalc('NeighborhoodNames')])
end

if(~isfield(options,'Large'))
    if n>10000
        large = 1;
        if verbose
            disp('Creating sparse neighborhood matrix.')
        end
        G = sparse(n,n,0);
    else
        large = 0;
        G = zeros(n,n);
    end
end

if(isfield(options.Neighborhood,'useKernel') && options.Neighborhood.useKernel)% supplied x and y are kernel matrices
    Kx = X;
    clear X; % throws error if using neighborhoods that do not support kernels
    Dx = diag(Kx)*ones(1,n) + ones(n,1)*diag(Kx)' - 2*Kx;
else
    if ~large
        Dx = distSqrdSelf(X);
    end
end

if verbose
    disp('Calculating the neighborhoods..')
end
switch options.Neighborhood.Type
    case {'knn','mnn'}  % Kernel ok
        if(~isfield(options.Neighborhood,'k'))
            error('Neighborhood.k required. Example: n.k = 7')
        end
        if ~large
            [~,Ix] = sort(Dx);
            knn = Ix(2:(options.Neighborhood.k+1),:); 
            for i = 1:n
                G(i,knn(:,i)) = 1;
            end
        else
            G = fast_approx_knn(X,options.Neighborhood.k);
        end
        if(~isfield(options.Neighborhood,'symmetric'))
            options.Neighborhood.symmetric = 0;
        end
        if strcmpi(options.Neighborhood.Type,'mnn')
            G = floor((G + G')/2);
        end
    case 'epsilon' % Kernel ok
        if(~isfield(options.Neighborhood,'epsilon'))
            error('Neighborhood.epsilon required. Example: n.epsilon = 0.05')
        end
        for i = 1:n
            G(i,find(Dx(i,:)<=options.Neighborhood.epsilon)) = 1;
        end
    case 'nn' % Kernel NA
        if d < 7
            G = naturalNeighbor(X);
        else
            G = naturalNeighborLP(X);
        end
    case 'nc' % Kernel NA
        if(~isfield(options.Neighborhood,'angle'))
            options.Neighborhood.angle = 5;
        end
        if(~isfield(options.Neighborhood,'tolerance'))
            options.Neighborhood.tolerance = 1e-3;
        end
        G = nearestCorrelation(X,options.Neighborhood);
end

end

