function [ G, H ] = nearestCorrelation( x, options)
%NEARESTCORRELATION Find classify correlating points as neighbors
%   The nearest correlation neighborhood finds points that
%align well with each other, i.e. fall on the same line. The tolerance can
%be calculated in two ways: The default mode is to use angles to determine
%the collinearity. The difference can also be determined using the
%difference between the length of the catheti and the hypotenuse. All
%ways are different ways of determining if the triangle determined by three
%points is sufficiently flat to be called a line. The methods try to find
%three linearly aligned points.
%Tolerance: the percentual difference between the length of the shorter
%lines (cateti) and the longest line (hypotenuse) in the triangle. tol=0.05
%means that differences up to 5% are allowed.
%Angle: given in angles, not radians

% H is the acceptance limit of each node. The tolerance needs to be larger
% than H(i,j) in order to make j a neighbor of i. After one run with
% nearestCorrelation, a new 0-1 neighborhood matrix can be constructed by
% using H. E.g. G2 = H < 0.02;

[n,d]=size(x);

if nargin < 2
    options.angle = 5;
end

if isfield(options,'useTolerance')
    useTol = options.useTolerance;
else
    useTol = 0;
end

if isfield(options,'tolerance') % minimum relative difference between the sum of the catheti and the hypotenuse
    tol = options.tolerance;
else
    tol = 1e-3;
end

if isfield(options,'angle') % minimum difference in the angles between points
    angle = options.angle;
else
    angle = 0.5;
end

Dx = distSqrdSelf(x);
H = inf*ones(n);
G = zeros(n);

if useTol %symmetric
    for i=1:n
        i
        for j=1:n
            if i~=j
                for k=(j+1):n
                    if k~=j && k~=i
                        %abs(Dx(i,j) + Dx(i,k) - Dx(j,k));
                        ij = sqrt(Dx(i,j));
                        ik = sqrt(Dx(i,k));
                        jk = sqrt(Dx(j,k));
                        ijk = sum([ij ik jk]);
                        dist = min([ ...
                            abs(ij + ik - jk) / ijk, ...
                            abs(ik + jk - ij) / ijk, ...
                            abs(jk + ij - ik) / ijk]);
                        
                        if dist < tol
                            G(i,j) = 1; %exp(-Dx(j,k)); %*exp(-Dy(j,k));
                            G(j,i) = 1; %exp(-Dx(j,k)); %*exp(-Dy(j,k));
                            G(i,k) = 1; %exp(-Dx(j,k)); %*exp(-Dy(j,k));
                            G(k,i) = 1; %exp(-Dx(j,k)); %*exp(-Dy(j,k));
                        end
                        if dist < H(i,j)
                            H(i,j) = dist;
                            H(j,i) = dist;
                        end
                    end
                end
            end
        end
    end
else % use angles, asymmetrical: each angle is different
    
    tol = deg2rad(angle); %change the tolerance to radians
    for i=1:n
        i
        xx = x- repmat(x(i,:),n,1);
        
        intprods = xx*xx';
        N = sqrt(diag(intprods));
        Dc = intprods./(N*N');
        dist = min(acos(abs(Dc)),abs(pi-acos(Dc))); % the angular difference between points
        dist(logical(eye(size(dist)))) = inf;
        
        % dist is symmetric
        [H(i,:), where] = min(dist'); %the minimum angle that makes any j a neighbor of i (in radians here; conversion to angles later)
        g = H(i,:) < tol; %seen from i: which points fall on the same line
        G(i,g) = 1;
        
        Dx = distSqrdSelf(xx);
        G(i,g) = exp(-Dx(i,g));
        
    end
    H = rad2deg(H);
end

