%% Demo for SDPP (Classification)
% Run each snippet by pressing CTRL+ENTER (Windows and Emacs defaults)
% Mika Juuti, 2015-02-20

%% Use the iris data for classification
% Using default settings
clear all; close all;
load iris_dataset;
x = irisInputs';
y = irisTargets';

[w, z, g, eigs]  = sdpp(x,y,'Classification',1);

% The third and fourth eigenvalue are zero, most of the information is in
% the first two vectors: z(:,1) and z(:,2)
showEigs(eigs); 
figure; scatter(z(:,1),z(:,2),[],y,'.');

%% Use the iris data for classification
% Use the nearest correlation neighborhood. Give the tolerance for the
% equation h - k1 - k2 = 0. (The hypotenuse equals the length of the
% catheti). The equation is normalized with the length of h.
% if tolerance=1 --> everyone is everyone's neighbor.

neighStruct.Type = 'nc'; neighStruct.tolerance = 1e-5; neighStruct.useTolerance = true;
options.Neighborhood = neighStruct; G = neighborhood(x,options);

[w, z, g, eigs]  = sdpp(x,y,[],G,'Classification',1,'Algorithm', 'sqlp');

showEigs(eigs);
figure; scatter(z(:,1),z(:,2),[],y,'.');

% show the neighborhood graph
figure; imagesc(g)
%% Use the iris data for classification
% Use the nearest correlation neighborhood. Give the angle in degrees. The
% default angle is 5. Give a different angle.
neighStruct.Type = 'nc'; neighStruct.angle = 0.5; neighStruct.useTolerance = false;
options.Neighborhood = neighStruct; G = neighborhood(x,options);

[w, z, g, eigs]  = sdpp(x,y,[],G,'Classification',1,'Algorithm', 'sqlp');

showEigs(eigs);
figure; scatter(z(:,1),z(:,2),[],y,'.');

% show the neighborhood graph
figure; imagesc(g)
%% Use the iris data for classification
% Custom neighborhood size.

neighStruct.Type = 'knn'; neighStruct.k = 15;
options.Neighborhood = neighStruct; G = neighborhood(x,options);
[w, z, g, eigs]  = sdpp(x,y,[],G,'Classification',1,'Algorithm', 'sqlp');

showEigs(eigs);
showNeighborhood(z,G,y);
figure; scatter(z(:,1),z(:,2),[],y,'.')
figure; imagesc(G)

%% Use the iris data for classification
% mnn (mutual nearest neighbors) includes only those neighbors of knn that
% are mutual. It is more conservative and more robust than knn in certain
% problems.

neighStruct.Type = 'mnn'; neighStruct.k = 15; 
options.Neighborhood = neighStruct; G = neighborhood(x,options);

[w, z, g, eigs]  = sdpp(x,y,[],G,'Classification',1,'Algorithm', 'sqlp');

showEigs(eigs);
showNeighborhood(z,g,y);
figure; scatter(z(:,1),z(:,2),[],y,'.');

% show the neighborhood graph
figure; imagesc(g)

%% Classification with kernels
% A kernel needs to fullfill Mercer's conditions, mostly being positive
% semidefinite. There are a number of frequently used kernels included in
% the SDPP.
%% Use the iris data for classification, using Gaussian kernel

neighStruct.Type = 'knn'; neighStruct.k = 15; options.Neighborhood = neighStruct;
G = neighborhood(x,options);
% the same kernel type is used for y by default, however it can be
K.x.KernelType = 'gaussian';
K.x.t = silverman(x); % specify a width of the gaussian kernels

[wk, zk, gk, eigsk]  = sdpp(x,y,[],...
    [],'Neighborhood',neighStruct,'Algorithm','cg','Classification',1,'Kernel',K);

showEigs(eigsk);
figure; scatter(zk(:,1),zk(:,2),[],y,'.');

%% Use the iris data for classification, using polynomial kernel

neighStruct.Type = 'knn'; neighStruct.k = 15; options.Neighborhood = neighStruct;

K.x.KernelType = 'polyplus';
K.x.d = 2; % specify the degree of the polynomial

[wk, zk, gk, eigsk]  = sdpp(x,y,[],...
    [],'Neighborhood',neighStruct,'Algorithm','cg','Classification',1,'Kernel',K);
showEigs(eigsk);
figure; scatter(zk(:,1),zk(:,2),[],y,'.');

%% Use the iris data for classification, the kernels can be precomputed too.
% If you want to use your own exotic kernel, you can specify it beforehand
% and pass it to SDPP. 

neighStruct.Type = 'knn'; neighStruct.k = 15; options.Neighborhood = neighStruct;

K.x.KernelType = 'gaussian';
K.x.t = silverman(x); % specify a width of the gaussian kernels
K.compute = 1;

n = size(x,1);
Kx = constructKernel(x,[],K.x);
H = eye(n)- ones(n,1)*ones(1,n)/n; % the x values should be centered around 0 (required for KPCA).
Kx = H*Kx*H;

Ky = constructKernel(y,[],K.x);

% Calculate the kernel pca solution. Kw0 could also be any other initial
% solution for the kernel.
[n,d] = size(x);
K.w = K.x;
if d > n
    K.w.m = n; %options for the kernel pca
end
[~, Kw0] = kernel_pca(x,K.w);

K.precomputed = 0;
[wk, zk, gk, eigsk]  = sdpp(Kx,Ky,Kw0,...
    [],'Neighborhood',neighStruct,'Algorithm','cg','Classification',1,'Kernel',K);

figure; scatter(zk(:,1),zk(:,2),[],y,'.');

% The same z can now be obtained by a matrix multiplication with the
% precalculated x kernel.
zz = Kx*wk;
figure; scatter(zz(:,1),zz(:,2),[],y,'.')