#include <limits.h>
#include "alloc.h"
#include "dassert.h"
#include "prefix.h"
#include "ulstack.h"
#include "darray.h"
#include "ptnet.h"
#include "set.h"
#include "mset.h"
#include "dphash.h"


Prefix::Prefix(PTNet *in_net, BOOL in_do_ltl)
{
  unsigned long i;
  unsigned long size;

  net = in_net;
  do_ltl = in_do_ltl;

  num_conditions = 0;
  num_events = 0;
  num_edges = 0;

  curr_condition_id = 0;
  curr_event_id = 0;

  conditions = (Condition **)xmalloc(PREFIX_MINALLOC *
				     sizeof(Condition *));

  events = (Event **)xmalloc(PREFIX_MINALLOC *
			     sizeof(Event *));

  edges = (Edge **)xmalloc(PREFIX_MINALLOC *
			   sizeof(Edge *));

  num_alloc_conditions = PREFIX_MINALLOC;
  num_alloc_events = PREFIX_MINALLOC;
  num_alloc_edges = PREFIX_MINALLOC;


  // Initialize the caches as empty

  cache_event = ULONG_MAX;
  cache_config = CreateMonSet(PREFIX_MINALLOC);
  cache_stack = CreateULStack();
  cache_parikh = CreateMSet(net->getNumTransitions());

  num_foata_levels = 0;
  num_alloc_foata_levels = PREFIX_MIN_LEVEL_ALLOC;
  cache_foata_levels = (DARRAY **)xmalloc(PREFIX_MIN_LEVEL_ALLOC *
					  sizeof(DARRAY *));

  for(i = 0; i < num_alloc_foata_levels; i++) {

    cache_foata_levels[i] = CreateDArray();

  }

  size = net->getNumPlaces();
  initial_marking = CreateMSet(size);
  for(i = 0; i < size; i++) {
    if(net->isInInitialState(i+1)) {
      AddOnetoMSet(initial_marking, i);
    }
  }

  markings = CreateDynPHash(1);

  event_queue = (EventQueue *)NULL;

  initial_conditions = CreateDArray();

  in_co_counterexample = FALSE;

  event_index = (unsigned long *)NULL;
  condition_index = (unsigned long *)NULL;
  num_added_cutoffs = 0; // Initialized by InitIndexes()

}


Prefix::~Prefix()
{
  unsigned long i;

  if(event_index != NULL) {
    free(event_index);
  }
  if(condition_index != NULL) {
    free(condition_index);
  }

  for(i = 0; i < num_edges; i++) {
    free(edges[i]);
  }
  free(edges);

  for(i = 0; i < num_events; i++) {
    if(events[i]->info != NULL) {

      // Deallocate memory used in comparisons
      if(events[i]->info->foata != NULL) {
	free(events[i]->info->foata);
      }
      if(events[i]->info->parikh != NULL) {
	free(events[i]->info->parikh);
      }
      free(events[i]->info);

    }
    if(events[i]->marking != NULL) {
      free(events[i]->marking);
    }
    free(events[i]);
  }
  free(events);

  for(i = 0; i < num_conditions; i++) {
    free(conditions[i]);
  }
  free(conditions);

  DeleteDArray(initial_conditions);
  DeleteDynPHash(markings);
  DeleteMSet(initial_marking);
  for(i = 0; i < num_alloc_foata_levels; i++) {
    DeleteDArray(cache_foata_levels[i]);
  }
  free(cache_foata_levels);
  DeleteMSet(cache_parikh);
  DeleteULStack(cache_stack);
  DeleteMonSet(cache_config);

}


unsigned long
Prefix::AddInitialCondition(unsigned long pl_id)
{
  Condition *c;

  if(num_conditions == num_alloc_conditions) {
    num_alloc_conditions *= 2;

    conditions = (Condition **)xrealloc(conditions,
					(num_alloc_conditions *
					 sizeof(Condition *)));
  }

  c = (Condition *)xmalloc(sizeof(Condition));
  conditions[num_conditions] = c;
  num_conditions++;
  curr_condition_id++;

  c->preset = (Edge *)NULL;
  c->postset = (Edge *)NULL;
  c->id = curr_condition_id;
  c->gen_idx = (curr_condition_id -1);
  c->pl_id = pl_id;

  AppendDArray(initial_conditions, (num_conditions -1));

  return(num_conditions -1); // From which index can this one be found?

}

unsigned long
Prefix::AddCondition(unsigned long pl_id)
{
  Condition *c;

  if(num_conditions == num_alloc_conditions) {
    num_alloc_conditions *= 2;

    conditions = (Condition **)xrealloc(conditions,
					(num_alloc_conditions *
					 sizeof(Condition *)));
  }

  c = (Condition *)xmalloc(sizeof(Condition));
  conditions[num_conditions] = c;
  num_conditions++;

  c->preset = (Edge *)NULL;
  c->postset = (Edge *)NULL;
  c->id = 0; // Will be fixed only later
  c->gen_idx = (num_conditions -1);
  c->pl_id = pl_id;

  return(num_conditions -1); // From which index can this one be found?

}


Edge *
Prefix::newEdge(void)
{
    Edge *eg;

    if(num_edges == num_alloc_edges) {
      num_alloc_edges *= 2;
      
      edges = (Edge **)xrealloc(edges,
				(num_alloc_edges *
				 sizeof(Edge *)));
    }

    eg = (Edge *)xmalloc(sizeof(Edge));
    edges[num_edges] = eg;
    num_edges++;

    return(eg);
}


// Preset and postset are represented as a list of <gc_index>

unsigned long
Prefix::AddEvent(unsigned long tr_id,
		 unsigned long preset_size,
		 unsigned long *preset,
		 unsigned long postset_size,
		 unsigned long *postset,
		 BOOL can_stutter)
{
  unsigned long i;
  Event *e;
  Edge *eg;

  if(num_events == num_alloc_events) {
    num_alloc_events *= 2;

    events = (Event **)xrealloc(events,
				(num_alloc_events *
				 sizeof(Event *)));

    cache_event = ULONG_MAX;
    DeleteMonSet(cache_config);
    cache_config = CreateMonSet(num_alloc_events);

  }

  e = (Event *)xmalloc(sizeof(Event));
  e->preset = (Edge *)NULL;
  e->postset = (Edge *)NULL;
  e->id = 0; // The event id is assigned only later
  e->tr_id = tr_id;
  e->gen_idx = num_events; // These start from 0!
  e->corr_event_idx = ULONG_MAX; // This will be initialized lazily
  e->config_size = 0; // This will be initialized lazily
  e->foata_level = 0; // This will be initialized lazily
  e->marking = (unsigned long *)NULL; // This will be initialized later
  e->prefix_cut_off = FALSE;
  e->omega_prefix_cut_off = FALSE;
  e->L_prefix_cut_off = FALSE;
  e->success = FALSE; // Was this a successful cut-off
  e->can_stutter = can_stutter;
  e->history_L_event = ULONG_MAX;

  e->next_marking_event = (Event *)NULL;
  e->info = (EVENT_QUEUE_INFO *)NULL;

  events[num_events] = e;
  num_events++;



  // First add the preset edges

  for(i = 0; i < preset_size; i++) {

    // Create the preset edge
    eg = newEdge();

    // Add the pointers to both the event and the condition

    eg->condition = conditions[preset[i]];
    eg->event = e;

    // Event preset extension

    eg->nextcondition = eg->event->preset;
    eg->event->preset = eg;

    // Condition postset extension

    eg->nextevent = eg->condition->postset;
    eg->condition->postset = eg;

  }


  // Then add the postset edges

  for(i = 0; i < postset_size; i++) {

    // Create the preset edge
    eg = newEdge();

    // Add the pointers to both the event and the condition

    eg->condition = conditions[postset[i]];
    eg->event = e;

    // Event postset extension

    eg->nextcondition = eg->event->postset;
    eg->event->postset = eg;

    // Condition preset extension

    eg->nextevent = eg->condition->preset;
    eg->condition->preset = eg;

  }


  if(do_ltl != FALSE) {

    if(IsLEvent((num_events -1))) {

      // Hey, a new L event found, mark all of the successors
      // with the number of this event number

      e->history_L_event = (num_events -1);

    } else {

      // Inherit the belonging to an L prefix from the first
      // parent

      if(e->preset->condition->preset != NULL) {

	e->history_L_event = e->preset->condition->preset->event->history_L_event;

      }
    
    }

  }

  return (num_events -1); // From which index can this one be found?

}


void
Prefix::AssignIDs(unsigned long event_idx)
{
  Condition *cond;
  Edge *eg;

  ASSERT(events[event_idx]->id == 0);
  curr_event_id++;
  events[event_idx]->id = curr_event_id;

  for(eg = events[event_idx]->postset;
      eg != NULL;
      eg = eg->nextcondition) {
    
    cond = eg->condition;

    ASSERT(cond->id == 0);
    curr_condition_id++;
    cond->id = curr_condition_id;

  }

}


unsigned long
Prefix::GetLocalConfigSize(unsigned long event_idx)
{
  unsigned long size;

  if(cache_event != event_idx) {
    CacheLocalConfig(event_idx);
  }

  size = GetMonSetSize(cache_config);
  return (size);

}

unsigned long
Prefix::GetNumIEvents(unsigned long event_idx)
{
  unsigned long i;
  unsigned long size;
  unsigned long idx;
  unsigned long num_I_events;
  MONSET *c_config;

  if(cache_event != event_idx) {
    CacheLocalConfig(event_idx);
  }

  c_config = cache_config;
  num_I_events = 0;
  size = GetMonSetSize(c_config);
  for(i = 0; i < size; i++) {
    idx = GetMonSetItem(c_config, i);
    if(IsIEvent(idx)) {
      num_I_events++;
    }
  }

  return (num_I_events);

}

unsigned long *
Prefix::GetParikh(unsigned long event_idx)
{

  unsigned long i;
  unsigned long size;
  unsigned long idx;
  unsigned long *parikh;
  unsigned long *t;
  MSET *c_parikh;
  MONSET *c_config;

  // TODO: Speed this up by creating a monotone multiset
  //       and using that for cache_parikh!

  if(cache_event != event_idx) {
    CacheLocalConfig(event_idx);
  }

  c_parikh = cache_parikh;
  ASSERT(EmptyMSet(c_parikh));

  c_config = cache_config;
  for(i = 0, size = GetMonSetSize(c_config);
      i < size;
      i++) {

    idx = GetMonSetItem(c_config, i);
    AddOnetoMSet(c_parikh,
		 (events[idx]->tr_id -1));

  }

  // Sort the items of the multiset in increasing transition order

  SortMSet(c_parikh);


  // Allocate the memory needed to store the Parikh vector

  size = GetMSetSize(c_parikh);
  ASSERT(size > 0);
  parikh = (unsigned long *)xmalloc(((2 * size) +1) *
				    sizeof(unsigned long));

  // Store the parikh is memory as: <num, <t1,c1>, ..., <tn,cn>>
  t = parikh;
  *t++ = size;
  for(i = 0; i < size; i++) {
    *t++ = GetMSetItem(c_parikh, i);
    *t++ = GetMSetItemCount(c_parikh, i);
    ASSERT(GetMSetItemCount(c_parikh, i) > 0);
  }

  // Clear the used multiset here, while the cache is still hot

  ClearMSet(c_parikh);

  return (parikh);
}



#define MAX(a,b) (((a) >= (b)) ? (a) : (b))

unsigned long
Prefix::GetFoataLevel(unsigned long event_idx)
{
  unsigned long level;
  unsigned long parent_level;
  Edge *eg;

  level = events[event_idx]->foata_level;
  if(level != 0) {
    // The level has been calculated before, return the cached value
    return (level);
  }

  // Calculate the maximum parent level using recursion
  // TODO: Make a non-recursive version!

  for(level = 0, eg = events[event_idx]->preset;
      eg != NULL;
      eg = eg->nextcondition) {

    if(eg->condition->preset != NULL) {
      parent_level = GetFoataLevel(eg->condition->preset->event->gen_idx);
      level = MAX(level, parent_level);
    }
    
  }

  // Add one to account for this event

  level++;


  // Cache the level

  events[event_idx]->foata_level = level;

  return (level);

}

unsigned long *
Prefix::GetFoata(unsigned long event_idx)
{
  unsigned long level;
  unsigned long old_size;
  unsigned long idx;
  unsigned long i;
  unsigned long j;
  unsigned long l;
  unsigned long size;
  unsigned long tr_idx;
  unsigned long i1;
  unsigned long i2;
  unsigned long *foata;
  unsigned long *t;
  MONSET *c_config;
  DARRAY *da;
  Event **ev;

  if(cache_event != event_idx) {
    CacheLocalConfig(event_idx);
  }

  level = GetFoataLevel(event_idx);

  if(level > num_alloc_foata_levels) {

    // Not enough levels allocated, round up to the next power of two

    for(old_size = num_alloc_foata_levels;
	num_alloc_foata_levels < level;
	num_alloc_foata_levels *= 2) {
      ;
    }

    cache_foata_levels = (DARRAY **)xrealloc(cache_foata_levels,
					     (num_alloc_foata_levels *
					      sizeof(DARRAY*)));

    // Create the new levels

    for(i = old_size; i < num_alloc_foata_levels; i++) {

      cache_foata_levels[i] = CreateDArray();

    }

  }


  // Append each transition to its own level

  c_config = cache_config;
  ev = events;
  for(i = 0, size = GetMonSetSize(c_config);
      i < size;
      i++) {

    idx = GetMonSetItem(c_config, i);
    l = GetFoataLevel(idx);
    da = cache_foata_levels[l -1];
    tr_idx = (ev[idx]->tr_id -1);

    AppendDArray(da, tr_idx);

  }


  for(i = 0 ; i < level; i++) {
    SortDArray(cache_foata_levels[i]);
  }

  // Check that each level is smaller than the number of transitions

  for(i = 0 ; i < level; i++) {

    da = cache_foata_levels[i];
    ASSERT(GetDArraySize(da) <= net->getNumTransitions());

  }

  // Check that no transition appears twice in any level, and that
  // they are sorted in increasing order within a level

  for(i = 0 ; i < level; i++) {
    
    da = cache_foata_levels[i];

    for(j = 1, l = GetDArraySize(da); j < l; j++) {

      i1 = GetDArrayItem(da, j-1);
      i2 = GetDArrayItem(da, j);
     
      ASSERT(i1 < i2);

    }

  }

  // Check that the sizes before and after Foata calculation match

  old_size = size; // The size of the configuration
  for(i = 0, size = 0; i < level; i++) {
    da = cache_foata_levels[i];
    size += GetDArraySize(da);
  }
  ASSERT(size == old_size);


  // Allocate the memory to store the levels

  foata = (unsigned long *)xmalloc((size + level + 1) *
				   sizeof(unsigned long));


  // The Foata levels as: <numlevels,
  //                      <l1size, <t1,t2, ..., tn>>,
  //                      <l2size, ...>, ..., <lmsize, ...>>

  t = foata;
  *t++ = level;
  for(i = 0; i < level; i++) {

    da = cache_foata_levels[i];
    size = GetDArraySize(da);

    *t++ = size;

    for(j = 0; j < size; j++) {

      idx = GetDArrayItem(da, j);
      *t++ = idx;

    }
    
  }


  // Cleanup the cached foata levels

  for(i = 0; i < level; i++) {
    ClearDArray(cache_foata_levels[i]);
  }

  return (foata);

}


unsigned long *
Prefix::GetMarking(unsigned long event_idx)
{
  unsigned long i;
  unsigned long size;
  unsigned long idx;
  unsigned long count;
  MONSET *c_config;
  Event **ev;
  MSET *marking_mset;
  Edge *eg;
  unsigned long *marking;
  unsigned long *t;
  

  if(cache_event != event_idx) {
    CacheLocalConfig(event_idx);
  }

  // First add the initial marking

  marking_mset = CopyMSet(initial_marking);

  // The add the postsets

  c_config = cache_config;
  ev = events;
  for(i = 0, size = GetMonSetSize(c_config);
      i < size;
      i++) {

    idx = GetMonSetItem(c_config, i);
    for(eg = ev[idx]->postset;
	eg != NULL;
	eg = eg->nextcondition) {

      AddOnetoMSet(marking_mset, eg->condition->pl_id -1);

    }

  }


  // Then remove the presets

  for(i = 0, size = GetMonSetSize(c_config);
      i < size;
      i++) {

    idx = GetMonSetItem(c_config, i);
    for(eg = ev[idx]->preset;
	eg != NULL;
	eg = eg->nextcondition) {

      RemoveOnefromMSet(marking_mset, eg->condition->pl_id -1);

    }

  }

  // Sort the items of the multiset in increasing place order

  SortMSet(marking_mset);


  // Allocate memory to store the marking

  size = GetMSetSize(marking_mset);
  marking = (unsigned long *)xmalloc((size +1) *
				     sizeof(unsigned long));

  // Markings are stored as: <size, p1, p2, ... , pn>,
  // where p1 ... pn are sorted

  t = marking;
  *t++ = size;

  for(i = 0; i < size; i++) {
    idx = GetMSetItem(marking_mset, i);
    count = GetMSetCount(marking_mset, idx);
    // Check that the marking is 1-safe
    ASSERT(count == 1);
    *t++ = idx;
  }

  DeleteMSet(marking_mset);
  
  return (marking);

}


void
Prefix::PrintMarking(unsigned long *marking)
{
  unsigned long i;
  unsigned long size;
  unsigned long *t;
  BOOL first;
  
  fprintf(stdout, "Marking: < ");

  t = marking;
  size = *t++;
  for(i = 0, first = TRUE; i < size; i++) {
    fprintf(stdout, ((first != FALSE) ? ("p%lu") : (", p%lu")), ((*t++) +1));
    first = FALSE;
  }

  fprintf(stdout, " >\n");

}


void
Prefix::CacheLocalConfig(unsigned long event_idx)
{
  unsigned long t;
  unsigned long t2;
  Edge *eg;
  MONSET *c_config;
  UL_STACK *c_stack;
  Event **ev;

  ASSERT(event_idx != cache_event);

  // First clear old cached config
  c_config = cache_config;
  ClearMonSet(c_config);
  ASSERT(EmptyMonSet(c_config));


  // Then add everything in the local config of event
  // to the cached config.

  c_stack = cache_stack;
  ASSERT(ULStackEmpty(c_stack));
  PushULStack(c_stack, event_idx);
  AddtoMonSet(c_config, event_idx);

  ev = events;
  while(ULStackEmpty(c_stack) == FALSE) {

    t = RemoveULStack(c_stack);
    for(eg = ev[t]->preset;
	eg != NULL;
	eg = eg->nextcondition) {

      if(eg->condition->preset != NULL) {
	t2 = eg->condition->preset->event->gen_idx;

	if(AddtoMonSet(c_config, t2) == FALSE) {

	  // The event t2 was new, check also its' preset

	  PushULStack(c_stack, t2);

	}

      }

    }

  }

  cache_event = event_idx;

}


BOOL
Prefix::CheckCutOff(unsigned long event_idx)
{
  BOOL cut;
#if 0
  unsigned long i;
  unsigned long size;
  unsigned long tmp_idx;
#endif
  unsigned long *marking;
  Event *old_event;
  Event *event;
  Event *e;
  unsigned long e1_idx;
  unsigned long e2_idx;
  MONSET *c_config;
  EVENT_QUEUE_INFO *i1;
  EVENT_QUEUE_INFO *i2;

  marking = GetMarking(event_idx);

//PrintMarking(marking);
  SetMarking(event_idx, marking);

  event = events[event_idx];

  old_event = (Event *)DynPInsertDataIfNew(markings,
					   marking,
					   event);


  if(do_ltl == FALSE) {

    // The non-LTL case is simply handled here

    cut = FALSE;
    if(old_event != NULL) {
      // There was an earlier event with this marking
      // so this one is a cut-off event

      cut = TRUE;

      event->corr_event_idx = old_event->gen_idx;
      
    }

    events[event_idx]->prefix_cut_off = cut;
    return (cut);

  }

  // The LTL case is handled here

  
  if(events[event_idx]->history_L_event == ULONG_MAX) {

    // First handle the normal prefix & the omega-prefix

    // These are always marked as L-event cut-offs
    events[event_idx]->L_prefix_cut_off = TRUE;

    events[event_idx]->prefix_cut_off = FALSE;
    cut = FALSE;
    if(old_event != NULL) {

      // The normal prefix is always cut here

      events[event_idx]->prefix_cut_off = TRUE;


      // However the omega-prefix might not be, we have to check
      // each event with the same marking

      // First check whether any of those events are in the
      // local configuration of this event. If so, this event
      // is a cut-off also in the omega-prefix.

      // Get the local configuration

      if(cache_event != event_idx) {
	CacheLocalConfig(event_idx);
      }

      c_config = cache_config;

      for(e = old_event;
	  e != NULL;
	  e = e->next_marking_event) {

	// All events on this list should have an id set up!
	ASSERT(e->id != 0);

	if(InMonSet(c_config, e->gen_idx)) {

	  // This event e' < e, and thus e is a cut-off in the omega prefix
	  cut=TRUE;
	  event->corr_event_idx = e->gen_idx;

	  // Check is the cut-off is also successfull
	  
	  e1_idx = e->gen_idx;
	  e2_idx = event_idx;
	  i1 = e->info;
	  i2 = events[event_idx]->info;

	  // Note: The code below changes cache_config, but that's OK, as
	  // we will no longer use it!

	  // ULONG_MAX = not intialized yet

	  if(i1->num_I_events == ULONG_MAX) {
	    i1->num_I_events = GetNumIEvents(e1_idx);
	  }
	  if(i2->num_I_events == ULONG_MAX) {
	    i2->num_I_events = GetNumIEvents(e2_idx);
	  }

	  if(i2->num_I_events > i1->num_I_events) {

	    // At least one I event happens between [e1] and [e2], be have
	    // an accepting loop!

	    events[event_idx]->success = TRUE;

	  }

	  break;

	}

      }

      c_config = NULL; // This might change under us, so init it to NULL to catch
                       // some bugs

      if(cut == FALSE) {

	// None of the events were in the local config of e

	// Check is there one which is smaller in the adequate order,
	// and has more I-events

	for(e = old_event;
	    e != NULL;
	    e = e->next_marking_event) {

	  e1_idx = e->gen_idx;
	  e2_idx = event_idx;
	  i1 = e->info;
	  i2 = events[event_idx]->info;

	  if(event_queue->CompareInfos(i1, i2) == -1) {

	    // The old event is smaller in the adaquate order, does it
	    // also have more I-events

	    // ULONG_MAX = not intialized yet

	    if(i1->num_I_events == ULONG_MAX) {
	      i1->num_I_events = GetNumIEvents(e1_idx);
	    }
	    if(i2->num_I_events == ULONG_MAX) {
	      i2->num_I_events = GetNumIEvents(e2_idx);
	    }

	    if(i1->num_I_events >= i2->num_I_events) {

	      // The smaller one has more or equal amount of
	      // I-events, we found a cut-off

	      cut = TRUE;
	      event->corr_event_idx = e->gen_idx;
	      break;

	    } else {

	      // The smaller one has less I-events, we can't cut here!

	      ;

	    }

	  }

	}

      }

    }

    events[event_idx]->omega_prefix_cut_off = cut;

    if(old_event != NULL) {

      // Append this new event at the end of the list if events
      // with the same marking

      for(e = old_event;
	  e->next_marking_event != NULL;
	  e = e->next_marking_event) {
	;
      }

      e->next_marking_event = events[event_idx];

    }

    return (cut);


  } else {



    // Handle the L-prefixes


    // These events are always marked as cut-offs in the normal prefix
    // and the omega prefix

    events[event_idx]->prefix_cut_off = TRUE;
    events[event_idx]->omega_prefix_cut_off = TRUE;



    // Handling of L-events is done next

    if(IsLEvent(event_idx)) {

      cut = FALSE;
      if(events[event_idx]->can_stutter == FALSE) {

	// This should never happen due to code in unfold.cc!
	
	ASSERT(FALSE);
#if 0	
	// If this L-event cannot stutter (see paper), we mark it as a cut-off
	cut = TRUE;
	event->corr_event_idx = event->gen_idx; // We make the corresponding event the same event!
#endif
	
      }

      if(cut == FALSE) {

	if(old_event != NULL) {
	  // There was an earlier L-event with this marking
	  // so this one is a cut-off event

	  cut = TRUE;
	  event->corr_event_idx = old_event->gen_idx;

	}
	
      }

// The following code is I believe correct, but it has been removed in order
// for the implemantation to correspond to the algorithm described in the paper.
      
#if 0
      if(cut == FALSE) {

	// Check in the local configuration of this L-event contains a cut-off of the
	// normal prefix. If so, this L-event is also a cut-off.

	// Get the local configuration

	if(cache_event != event_idx) {
	  CacheLocalConfig(event_idx);
	}

	c_config = cache_config;

	for(i = 0, size = GetMonSetSize(c_config); i < size; i++) {

	  tmp_idx = GetMonSetItem(c_config, i);

	  // Ignore this event, handle everything else
	  if(tmp_idx != event_idx) {

	    if(events[tmp_idx]->prefix_cut_off != FALSE) {

	      // There is an event which is a cut-off in the normal prefix
	      // in the local configuration of this event. Therefore this
	      // L-event can be marked as a cut-off event.
	      
	      cut = TRUE;
	      // Well, this does not have the same marking, but it is the reason
	      // why this event is a cut-off, so I'm not lying too much.
	      event->corr_event_idx = events[tmp_idx]->gen_idx;
	      break;

	    }

	  }
	  
	}

	c_config = NULL; // This might change under us, so init it to NULL to catch
	                 // some bugs
	
	
      }

#endif

      
      // Note: We don't append this one to the list!
      
      events[event_idx]->L_prefix_cut_off = cut;
      return (cut);

    }


    // Handling of non-L-events is a bit more complicated

    cut = FALSE;
    if(old_event != NULL) {


      // First check that there is no event with a smaller
      // history_L_event with the same marking


      for(e = old_event;
	  e != NULL;
	  e = e->next_marking_event) {

	// All events on this list should have an id set up!
	ASSERT(e->id != 0);

	// Also the history_L_event should be set up!
	ASSERT(e->history_L_event != ULONG_MAX);
	ASSERT(events[event_idx]->history_L_event != ULONG_MAX);


	// History L-events are compared with respect to the
	// real id of the corresponding event

	e1_idx = events[e->history_L_event]->id;
	e2_idx = events[events[event_idx]->history_L_event]->id;

	ASSERT(e1_idx != 0);
	ASSERT(e2_idx != 0);

	if(e1_idx < e2_idx) {

	  // A (non-succesful) cut-off found!

	  cut = TRUE;
	  event->corr_event_idx = e->gen_idx;
	  break;
	  
	}

      }


      if(cut == FALSE) {

	// Next check whether any of those events are in the
	// local configuration of this event. If so, this event
	// is a cut-off also in the L-prefix.

	// Get the local configuration

	if(cache_event != event_idx) {
	  CacheLocalConfig(event_idx);
	}

	c_config = cache_config;

	for(e = old_event;
	    e != NULL;
	    e = e->next_marking_event) {

	  // All events on this list should have an id set up!
	  ASSERT(e->id != 0);

	  if(InMonSet(c_config, e->gen_idx)) {

	    // This event e' < e, and thus e is a cut-off in the L-prefix
	    cut=TRUE;
	    event->corr_event_idx = e->gen_idx;
	    
	    // These cut-offs are always successful

	    events[event_idx]->success = TRUE;

	    break;

	  }

	}

	c_config = NULL; // This might change under us, so init it to NULL to catch
	                 // some bugs
      }



      if(cut == FALSE) {

	// None of the events were in the local config of e

	for(e = old_event;
	    e != NULL;
	    e = e->next_marking_event) {

	  e1_idx = e->gen_idx;
	  e2_idx = event_idx;

	  // Check that the history_L_events match

	  ASSERT(events[e1_idx]->history_L_event != ULONG_MAX);
	  ASSERT(events[e2_idx]->history_L_event != ULONG_MAX);
	  ASSERT(events[e1_idx]->history_L_event ==
		 events[e2_idx]->history_L_event);


	  // Check if the events are in coflict

	  if(InConflict(e1_idx, e2_idx) == FALSE) {

	    // The events are not in co, this is a successful cut-off

	    cut=TRUE;
	    event->corr_event_idx = e->gen_idx;
	    
	    // These cut-offs are always successful

	    events[event_idx]->success = TRUE;

	    // These events need special handling in the counterexample generation

	    in_co_counterexample = TRUE;
	    
	    break;

	  } 

	  // The events are in conflict

	  i1 = e->info;
	  i2 = events[event_idx]->info;

	  if(event_queue->CompareInfos(i1, i2) == -1) {

	    // The old event is smaller in the adaquate order, does it
	    // also have more events

	    if(i1->local_config_size == 0) {
	      i1->local_config_size = GetLocalConfigSize(e1_idx);
	    }
	    if(i2->local_config_size == 0) {
	      i2->local_config_size = GetLocalConfigSize(e2_idx);
	    }
	    
	    if(i1->local_config_size >= i2->local_config_size) {

	      // The smaller one has more events,
	      // we found a cut-off

	      cut = TRUE;
	      event->corr_event_idx = e->gen_idx;
	      break;

	    } else {

	      // The smaller one has less events, we can't cut here!

	      ;

	    }

	  }

	}

      }

    }

  }

  events[event_idx]->L_prefix_cut_off = cut;

  if(old_event != NULL) {

    // Append this new event at the end of the list if events
    // with the same marking

    for(e = old_event;
	e->next_marking_event != NULL;
	e = e->next_marking_event) {
      ;
    }

    e->next_marking_event = events[event_idx];

  }

  return (cut);

}


BOOL
Prefix::InConflict(unsigned long event_idx1, unsigned long event_idx2)
{
  unsigned long i;
  unsigned long size;
  unsigned long idx;
  BOOL conflict;
  MONSET *c1;
  MONSET *c2;
  MONSET *conds;
  Edge *eg;
  Event **ev;

  if(cache_event != event_idx1) {
    CacheLocalConfig(event_idx1);
  }

  c1 = CopyMonSet(cache_config);

  if(cache_event != event_idx2) {
    CacheLocalConfig(event_idx2);
  }

  c2 = cache_config;


  // This might be quite slow
  // TODO: Rewrite!

  conds = CreateMonSet(num_conditions);


  ev = events;
  conflict = FALSE;
  for(i = 0, size = GetMonSetSize(c1); i < size; i++) {
    
    idx = GetMonSetItem(c1, i);

    for(eg = ev[idx]->preset;
	eg != NULL;
	eg = eg->nextcondition) {

      // This condition is removed by the configuration C1

      AddtoMonSet(conds, eg->condition->gen_idx);

    }

  }

  for(i = 0, size = GetMonSetSize(c2);
      ((conflict == FALSE) && (i < size));
      i++) {
    
    idx = GetMonSetItem(c2, i);

    if(InMonSet(c1, idx)) {

      // This event is common to both configs, there's no
      // conflict on it

      continue;

    }

    // This event belongs to C2 only, is it a spoiler?

    for(eg = ev[idx]->preset;
	eg != NULL;
	eg = eg->nextcondition) {

      // This condition is removed by the configuration C2

      // Was the condition already removed by C1?
      if(InMonSet(conds, eg->condition->gen_idx)) {

	// Yes, two different events in the postset
	// of the condition -> conflict!

	conflict = TRUE;
	break;

      }

    }

  }


  DeleteMonSet(conds);
  DeleteMonSet(c1);

  return (conflict);

}


BOOL
Prefix::ExpandLPreset(unsigned long l_trans_idx, SARRAY *tr_preset)
{
  unsigned long i;
  unsigned long size;
  unsigned long idx;
  unsigned long t;
  unsigned long t2;
  BOOL result;
  SARRAY *preset_places;
  Edge *eg;
  MONSET *c_config;
  MSET *cut; // TODO: Replace me with a set!
  UL_STACK *c_stack;
  Event **ev;
  Condition **cd;

  // First clear old cached config
  c_config = cache_config;
  ClearMonSet(c_config);
  ASSERT(EmptyMonSet(c_config));


  // Then add everything in the local config of event
  // to the cached config.

  ev = events;
  cd = conditions;
  c_stack = cache_stack;
  ASSERT(ULStackEmpty(c_stack));
  for(i = 0, size = GetSArraySize(tr_preset); i < size; i++) {
    idx = GetSArrayItem(tr_preset, i);
    if(cd[idx]->preset != NULL) {
      t=cd[idx]->preset->event->gen_idx;
      if(AddtoMonSet(c_config, t) == FALSE) {
	PushULStack(c_stack, t);
      }
    }
  }

  // Then close the configuration backward

  while(ULStackEmpty(c_stack) == FALSE) {

    t = RemoveULStack(c_stack);
    for(eg = ev[t]->preset;
	eg != NULL;
	eg = eg->nextcondition) {

      if(eg->condition->preset != NULL) {
	t2 = eg->condition->preset->event->gen_idx;

	if(AddtoMonSet(c_config, t2) == FALSE) {

	  // The event t2 was new, check also its' preset

	  PushULStack(c_stack, t2);

	}

      }

    }

  }


  // We have a configuration, create the corresponding cut

  cut = CreateMSet(num_conditions);

  // First add initial conditions


  for(i = 0, size = GetDArraySize(initial_conditions);
      i < size;
      i++) {
    idx = GetDArrayItem(initial_conditions, i);

    ASSERT(GetMSetCount(cut, idx) == 0);
    AddOnetoMSet(cut, idx);

  }

  // Then add postsets

  for(i = 0, size = GetMonSetSize(c_config);
      i < size;
      i++) {

    idx = GetMonSetItem(c_config, i);
    for(eg = ev[idx]->postset;
	eg != NULL;
	eg = eg->nextcondition) {

      AddOnetoMSet(cut, eg->condition->gen_idx);

    }

  }


  // Then remove the presets

  for(i = 0, size = GetMonSetSize(c_config);
      i < size;
      i++) {

    idx = GetMonSetItem(c_config, i);
    for(eg = ev[idx]->preset;
	eg != NULL;
	eg = eg->nextcondition) {

      ASSERT(GetMSetCount(cut, eg->condition->gen_idx) != 0);
      RemoveOnefromMSet(cut, eg->condition->gen_idx);

    }

  }


  // Then remove all the stuff that is already in the preset

  for(i = 0, size = GetSArraySize(tr_preset); i < size; i++) {
    idx = GetSArrayItem(tr_preset, i);
    ASSERT(GetMSetCount(cut, idx) == 1);
    RemoveOnefromMSet(cut, idx);
  }

  // All which is left to the cut is the new stuff to be added
  // to the preset

  for(i = 0, size = GetMSetSize(cut); i < size; i++) {
    idx = GetMSetItem(cut, i);
    ASSERT(GetMSetItemCount(cut, i) == 1);
    AppendSArray(tr_preset, idx);
  }

  DeleteMSet(cut);

  cache_event = ULONG_MAX;
  ClearMonSet(c_config);
  ASSERT(EmptyMonSet(c_config));

  // Check whether this one can stutter?
  // First map the conditions back to places

  size = GetSArraySize(tr_preset);
  preset_places = CreateSArray(size);
  for(i = 0; i < size; i++) {
    AppendSArray(preset_places,
		 cd[GetSArrayItem(tr_preset, i)]->pl_id -1);
  }

  result = net->CanStutter(l_trans_idx, size, GetSArray(preset_places));

  DeleteSArray(preset_places);
  
  return (result);

}

void
Prefix::WriteNum(unsigned long num, FILE *out)
{
  fwrite(((void *)&num), sizeof(unsigned long), 1, out);
}

void
Prefix::WriteString(char *str, FILE *out)
{
  // Write a string + '\0'

  fwrite(((void *)str), sizeof(char), (strlen(str) +1), out);
}


void
Prefix::InitIndexes(void)
{
  unsigned long i;

  if(event_index != NULL) {

    // Initialization already done!

    return;

  }
  
  event_index = (unsigned long *)xmalloc(curr_event_id *
					 sizeof(unsigned long));

  condition_index = (unsigned long *)xmalloc(curr_condition_id *
					     sizeof(unsigned long));

  num_added_cutoffs = 0;
  for(i = 0; i < num_events; i++) {
    if(events[i]->id != 0) {
      event_index[events[i]->id -1] = events[i]->gen_idx;
      if(events[i]->corr_event_idx != ULONG_MAX) {
	num_added_cutoffs++;
      }
    }
  }
  for(i = 0; i < num_conditions; i++) {
    if(conditions[i]->id != 0) {
      condition_index[conditions[i]->id -1] = conditions[i]->gen_idx;
    }
  }
  
}


BOOL
Prefix::WriteMciFile(FILE *out)
{
  unsigned long i;
  unsigned long id;
  unsigned long size;
  unsigned long name_len;
  Edge *eg;
  Event *e;
  UL_STACK *stack;
  DARRAY *da;
  Place **places;
  Transition **transitions;
  
  InitIndexes();
  stack = CreateULStack();
  da = CreateDArray();
  
  WriteNum(curr_condition_id, out);
  WriteNum(curr_event_id, out);

  for(i = 0; i < curr_event_id; i++) {
    WriteNum(events[event_index[i]]->tr_id,
	     out);
  }

  for(i = 0; i < curr_condition_id; i++) {

    // Output the place label
    
    WriteNum(conditions[condition_index[i]]->pl_id,
	     out);

    // Output the preset
    
    if(conditions[condition_index[i]]->preset == NULL) {

      // This condition is in the set of initial conditions
      
      WriteNum(0, out);

    } else {

      WriteNum(conditions[condition_index[i]]->preset->event->id,
	       out);
      
    }


    // Output the postset

    for(eg = conditions[condition_index[i]]->postset;
	eg != NULL;
	eg = eg->nextevent) {

      e = eg->event;
      PushULStack(stack, e->id);
      
    }

    // Reverse stuff so that the postset is written
    // in increasing event order
    
    while(ULStackEmpty(stack) == FALSE) {

      id = RemoveULStack(stack);
      WriteNum(id, out);

    }
      
    // Zero terminate the postset

    WriteNum(0, out);
    
  }


  // Output the cutoff events and their corresponding events
  
  for(i = 0; i < curr_event_id; i++) {

    // Cutoffs have their corresponding event set!
    
    if(events[event_index[i]]->corr_event_idx != ULONG_MAX) {
      AppendDArray(da, events[event_index[i]]->id);
    }
    
  }

  SortDArray(da);
  
  for(i = 0, size = GetDArraySize(da); i < size; i++) {

    id = GetDArrayItem(da, i);
    
    WriteNum(id,out);
    
    WriteNum(events[events[event_index[id -1]]->corr_event_idx]->id,
	     out);
    
  }
  WriteNum(0, out);


  // No maximal configs from us
  
  WriteNum(0, out);


  // Write the number of Places

  WriteNum(net->getNumPlaces(), out);
  

  // Write the number of Transitions

  WriteNum(net->getNumTransitions(), out);


  // Write the maximum name len

  places = net->getPlaces();
  for(i = 0, size = net->getNumPlaces(), name_len = 0; i < size; i++) {
    name_len = MAX(name_len, strlen(places[i]->name));
  }
  transitions = net->getTransitions();
  for(i = 0, size = net->getNumTransitions(); i < size; i++) {
    name_len = MAX(name_len, strlen(transitions[i]->name));
  }
  WriteNum(name_len, out);
  

  // Write the Place names

  for(i = 0, size = net->getNumPlaces(), name_len = 0; i < size; i++) {
    WriteString(places[i]->name, out);
  }
  WriteString("", out);
  
  // Write the Transition names

  for(i = 0, size = net->getNumTransitions(); i < size; i++) {
    WriteString(transitions[i]->name, out);
  }
  WriteString("", out);
  
  DeleteDArray(da);
  DeleteULStack(stack);
  
  return (TRUE);

}

#undef WRITENUM
#undef MAX


void Prefix::PrintCounterexample(void)
{
  unsigned long i;
  unsigned long num_printed_events;
  unsigned long tr_id;
  unsigned long event_idx;
  unsigned long size;
  unsigned long idx;
  unsigned long id;
  const char *format;
  MONSET *c_config;
  SET *s;
  SET *s2;
  
  num_printed_events = 0;
  format = (const char *)NULL;

  InitIndexes();
  
  // Find the index of the successful event

  event_idx = ULONG_MAX;
  for(i = 0; i < num_events; i++) {

    if(events[i]->success != FALSE) {
      event_idx = i;
      break;
    }
    
  }
  ASSERT(event_idx != ULONG_MAX);

  if(cache_event != event_idx) {
    CacheLocalConfig(event_idx);
  }
  c_config = cache_config;
  size = GetMonSetSize(c_config);
  s = CreateSet(curr_event_id);

  // Get the local config and sort it
  
  for(i = 0 ; i < size; i++) {
    idx = GetMonSetItem(c_config, i);
    AddtoSet(s, (events[idx]->id -1));
  }
  SortSet(s);

  if(in_co_counterexample != FALSE) {
    
    // Handle co-counterexamples differently

    s2 = CreateSet(curr_event_id);
    
    if(cache_event != events[event_idx]->corr_event_idx) {
      CacheLocalConfig(events[event_idx]->corr_event_idx);
    }

    c_config = cache_config;
    size = GetMonSetSize(c_config);

    for(i = 0 ; i < size; i++) {
      idx = GetMonSetItem(c_config, i);
      if(InSet(s, (events[idx]->id -1)) != FALSE) {

	// This event is in both configs

	AddtoSet(s2, (events[idx]->id -1));

      }

    }
    SortSet(s2);
    c_config = NULL;
    
    for(i = 0, size = GetSetSize(s2); i < size; i++) {

      id = GetSetItem(s2, i);
      RemovefromSet(s, id);
      
      if(num_printed_events == 0) {
	format = "E%lu(%s)";
      } else if (num_printed_events & 0x1) {
	format = ", E%lu(%s)";
      } else {
	format = ",\n  E%lu(%s)";
      }

      tr_id = events[event_index[id]]->tr_id;
      fprintf(stdout, format, (id +1), net->getTrName(tr_id));

      num_printed_events++;

    }

    DeleteSet(s2);
    
  }

  for(i = 0, size = GetSetSize(s); i < size; i++) {

    id = GetSetItem(s, i);

    if(num_printed_events == 0) {
      format = "E%lu(%s)";
    } else if (num_printed_events & 0x1) {
      format = ", E%lu(%s)";
    } else {
      format = ",\n  E%lu(%s)";
    }

    tr_id = events[event_index[id]]->tr_id;
    fprintf(stdout, format, (id +1), net->getTrName(tr_id));

    num_printed_events++;

  }
  
  if(num_printed_events > 0) {
    fprintf(stdout, ".");
  }

  fprintf(stdout, "\n");
  fflush(stdout);

  DeleteSet(s);
  
}

unsigned long
Prefix::getNumAddedCutOffs(void)
{
  InitIndexes();

  return (num_added_cutoffs);
  
}


unsigned long
Prefix::GetLEventIdx(unsigned long event_idx)
{
  unsigned long L_idx;

  // L-events have no L-events in their preset

  if(IsLEvent(event_idx)) {

    // There is no L-event in the history, return ULONG_MAX
    
    return (ULONG_MAX);

  }

  // All non-L-events have the L-event idx in their
  // event structure

  L_idx = events[event_idx]->history_L_event;
  
  return (L_idx);

}


void
Prefix::PrintLTLStats(void)
{
  unsigned long id;
  unsigned long event_idx;
  unsigned long i;
  unsigned long size;
  unsigned long tmp_idx;
  BOOL cut;
  Edge *eg;
  MONSET *c_config;
  
  unsigned long conventional_B;
  unsigned long conventional_E;
  unsigned long conventional_cutoffs;
  unsigned long infinite_B;
  unsigned long infinite_E;
  unsigned long infinite_cutoffs;
  unsigned long divergence_B;
  unsigned long divergence_E;
  unsigned long divergence_cutoffs;
  unsigned long L_conditions;
  unsigned long L_events;
  unsigned long L_cutoffs;
  
  conventional_B = GetDArraySize(initial_conditions);
  conventional_E = 0;
  conventional_cutoffs = 0;
  infinite_B = 0;
  infinite_E = 0;
  infinite_cutoffs = 0;
  divergence_B = 0;
  divergence_E = 0;
  divergence_cutoffs = 0;
  L_conditions = 0;
  L_events = 0;
  L_cutoffs = 0;

  // Loop over only those which have been added to the prefix
  
  for(id = 0; id < curr_event_id; id++) {

    event_idx = event_index[id];
    
    if(events[event_idx]->history_L_event == ULONG_MAX) {

      // First handle the normal prefix & the omega-prefix

      // Get the local configuration

      if(cache_event != event_idx) {
	CacheLocalConfig(event_idx);
      }

      c_config = cache_config;

      cut = FALSE;
      for(i = 0, size = GetMonSetSize(c_config); i < size; i++) {

	tmp_idx = GetMonSetItem(c_config, i);

	if(tmp_idx != event_idx) {
	
	  if(events[tmp_idx]->prefix_cut_off != FALSE) {

	    // There is an event which is a cut-off in the normal prefix
	    // in the local configuration of this event.
	    // Therefore this event does not belong to the conventional prefix.
	  	      
	    cut = TRUE;
	    break;

	  }

	}

      }

      c_config = NULL; // This might change under us, so init it to NULL to catch
                       // some bugs

      if(cut == FALSE) {


	// This event belongs to the conventional prefix

	conventional_E++;

	if(events[event_idx]->prefix_cut_off != FALSE) {

	  conventional_cutoffs++;
	  
	}

	for(eg = events[event_idx]->postset;
	    eg != NULL;
	    eg = eg->nextcondition) {

	  conventional_B++;

	}

	
      } else {

	
	// This event belongs to the infinite loop prefix only

	infinite_E++;
	
	if(events[event_idx]->omega_prefix_cut_off != FALSE) {

	  infinite_cutoffs++;
	  
	}

	for(eg = events[event_idx]->postset;
	    eg != NULL;
	    eg = eg->nextcondition) {

	  infinite_B++;

	}

      }

      
    } else {


      // Handling of L-events is done next

      if(IsLEvent(event_idx)) {

	L_events++;

	if(events[event_idx]->L_prefix_cut_off != FALSE) {

	  L_cutoffs++;

	}
	
	for(eg = events[event_idx]->postset;
	    eg != NULL;
	    eg = eg->nextcondition) {

	  L_conditions++;

	}

      }

      // All L-events are also counted to the divergence stuff
      
      divergence_E++;

      if(events[event_idx]->L_prefix_cut_off != FALSE) {

	divergence_cutoffs++;

      }
      
      for(eg = events[event_idx]->postset;
	  eg != NULL;
	  eg = eg->nextcondition) {

	divergence_B++;

      }
      
    }
    
  }
  
  fprintf(stdout, "LTL statistics:\n");
  fprintf(stdout, "Conventional prefix:\n");
  fprintf(stdout, "|B|= %lu, |E|= %lu, Cutoffs= %lu\n",
	  conventional_B,conventional_E,conventional_cutoffs);
  fprintf(stdout, "Additional infinite path checking part:\n");
  fprintf(stdout, "|B|= %lu, |E|= %lu, Cutoffs= %lu\n",
	  infinite_B,infinite_E,infinite_cutoffs);
  fprintf(stdout, "Additional system divergence checking part:\n");
  fprintf(stdout, "|B|= %lu, |E|= %lu, Cutoffs= %lu, out of which: \n",
	  divergence_B,divergence_E,divergence_cutoffs);
  fprintf(stdout, "|LB|= %lu, |LE|= %lu, and LCutoffs= %lu\n",
	  L_conditions, L_events,L_cutoffs);
  
  fprintf(stdout, "\n");

}
