#include <string.h>
#include <strings.h>
#include <limits.h>
#include "alloc.h"
#include "dassert.h"
#include "eventq.h"
#include "prefix.h"


EventQueue::EventQueue(Prefix *in_prefix, BOOL in_do_ltl)
{
  prefix = in_prefix;
  do_ltl = in_do_ltl;
  size = 0;
  alloc_size = EVENTQUEUE_MINALLOC;
  infos = (EVENT_QUEUE_INFO **)xmalloc(EVENTQUEUE_MINALLOC *
				       sizeof(EVENT_QUEUE_INFO *));

}


EventQueue::~EventQueue()
{
  unsigned long i;
  EVENT_QUEUE_INFO *info;

  for(i = 0; i < size; i++) {
    info = infos[i];

    if(do_ltl != FALSE) {
      prefix->SetEventInfo(info->idx, NULL);
    }

    // Deallocate memory used in comparisons
    if(info->foata != NULL) {
      free(info->foata);
    }
    if(info->parikh != NULL) {
      free(info->parikh);
    }
    free(info);

  }

  free(infos);
}


void
EventQueue::Add(unsigned long event)
{
  unsigned long i;
  EVENT_QUEUE_INFO *info;
  int result;

  if(size == alloc_size) {
    alloc_size *= 2;
    infos = (EVENT_QUEUE_INFO **)xrealloc(infos,
					  (alloc_size *
					   sizeof(EVENT_QUEUE_INFO *)));
  }


  // Initialize all info fields

  info = (EVENT_QUEUE_INFO *)xmalloc(sizeof(EVENT_QUEUE_INFO));
  info->idx = event;
  if(do_ltl == FALSE) {
    info->L_event_idx = ULONG_MAX;
  } else {
    info->L_event_idx = prefix->GetLEventIdx(event);
  }
  info->local_config_size = 0;
  info->num_I_events = ULONG_MAX; // This can be zero, therefore init to this
  info->parikh = (unsigned long *)NULL;
  info->foata = (unsigned long *)NULL;

  // This is simple insertion sort (append to the end)
  // NOTE: This assumes a total adequate order!

  // TODO: This is damn slow, use a real data structure instead!

  for(i = size; i > 0; i--) {

    result = CompareInfos(infos[i-1], info);
    ASSERT(result != 0); // Totality check
    if(result == -1) {
      break;
    }

    infos[i] = infos[i-1];

  }

  infos[i] = info;

  if(do_ltl != FALSE) {
    prefix->SetEventInfo(event, info);
  }

  size++;

}


// This is really slow!
// TODO: Rewrite!


unsigned long
EventQueue::RemoveSmallest(void)
{
  unsigned long i;
  unsigned long item;
  EVENT_QUEUE_INFO *info;

  ASSERT(size != 0);

  // Get the first item in queue
  info = infos[0];
  item = info->idx;

  if(do_ltl == FALSE) {
    // Deallocate memory used in comparisons
    if(info->foata != NULL) {
      free(info->foata);
    }
    if(info->parikh != NULL) {
      free(info->parikh);
    }
    free(info);
  } else {
    // Can't do that in the LTL model checking version, sorry!
    ;
  }


  // Move all the remaining items one step
  for(i = 1; i < size; i++) {
    infos[i-1] = infos[i];
  }

  // Make the queue smaller
  size--;

#if 0
  fprintf(stdout, "Model %ld removed\n", (item +1));
#endif

  return (item);

}


#define MIN(a,b) (((a) <= (b)) ? (a) : (b))

int
EventQueue::CompareInfos(EVENT_QUEUE_INFO *in_i1,
			 EVENT_QUEUE_INFO *in_i2)
{
  EVENT_QUEUE_INFO *i1;
  EVENT_QUEUE_INFO *i2;
  int res1;
  
  if(do_ltl != FALSE) {

    i1 = in_i1;
    i2 = in_i2;

    // For LTL model checking use use another adequate order

    if(in_i1->L_event_idx != ULONG_MAX) {

      // If C1 contains an L-event, use the local config of it first

      i1 = prefix->GetEventInfo(in_i1->L_event_idx);

    }

    if(in_i2->L_event_idx != ULONG_MAX) {

      // If C2 contains an L-event, use the local config of it first

      i2 = prefix->GetEventInfo(in_i2->L_event_idx);

    }


    // Compare the parts before L-events against each other

    ASSERT(i1->L_event_idx == ULONG_MAX);
    ASSERT(i2->L_event_idx == ULONG_MAX);
    
    res1 = CompareERV(i1, i2);

    if(res1 != 0) {

      // The parts before L-events differed, return the result

      return (res1);

    }

    // The parts before L-events agreed, fall though to compare
    // full configurations
    
  }
  
  // OK, resort to the ERV comparison of full configurations.

  return (CompareERV(in_i1, in_i2));

}



// This (hopefully) implements the ERV adequate order as
// implemented by the ERVunfold tool. (Modulo bugs.)
//
// See the undeffed code for the variant in the ERV paper!

//#define DEBUG_PARIKH

int
EventQueue::CompareERV(EVENT_QUEUE_INFO *i1,
		       EVENT_QUEUE_INFO *i2)
{
  unsigned long e1_idx;
  unsigned long e2_idx;
  unsigned long i;
  unsigned long j;
  unsigned long p_min;
  unsigned long *t1;
  unsigned long *t2;
  unsigned long t1_size;
  unsigned long t2_size;
  unsigned long e1_levels;
  unsigned long e2_levels;
  unsigned long l_min;

#ifdef DEBUG_PARIKH

  int compare;
  unsigned long k;
  unsigned long l;
  unsigned long p1_size;
  unsigned long p2_size;
  unsigned long p1_len;
  unsigned long p2_len;
  unsigned long *p1;
  unsigned long *p2;
  char *ps1;
  char *ps2;
  char *buf;
  BOOL first;

#endif /* DEBUG_PARIKH */

  e1_idx = i1->idx;
  e2_idx = i2->idx;

  // First calculate the local config sizes

  if(i1->local_config_size == 0) {
    i1->local_config_size = prefix->GetLocalConfigSize(e1_idx);
  }
  if(i2->local_config_size == 0) {
    i2->local_config_size = prefix->GetLocalConfigSize(e2_idx);
  }

  // Return if the sizes do not match

  if(i1->local_config_size < i2->local_config_size) {
    return (-1);
  } else if(i1->local_config_size > i2->local_config_size) {
    return (1);
  }

  // The sizes match, have to check the Parikh vextors
  // (Try to maximize cache usage by doing the gets in reverse order.)

  if(i2->parikh == NULL) {
    i2->parikh = prefix->GetParikh(e2_idx);
  }
  if(i1->parikh == NULL) {
    i1->parikh = prefix->GetParikh(e1_idx);
  }


  // Parikh vector comparison is made in lexicographical order.
  // This seems to be one of the most difficult parts to get right!
  // (I've found bugs in my own & other people's code right here.)
  //
  // The idea is the following: (i) Write down the set of transitions
  // in the local configuration of the event. (ii) Sort this string
  // in increasing transition order. (iii) Compare these string in
  // normal lexicographic order.
  // 
  // The hard part seems to be to implement (iii) when starting from
  // a multiset representation. You really have to get all the corner
  // cases right.
  //
  // Examples:
  // <t1>        <  <t2>
  // <t1,t1>     <  <t2>
  // <t1,t1>     <  <t1,t1,t1>
  // <t1,t2>     <  <t1,t2,t3>
  // <t1,t1,t5>  <  <t1,t3>
  // <t1>        <  <t1,t1,t2>
  //

#ifdef DEBUG_PARIKH

  p1 = i1->parikh;
  p2 = i2->parikh;

  p1_size = *p1++;
  p2_size = *p2++;

  // Calculate the length of the string to represent P1

  for(k = 0, p1_len = 0; k < p1_size; k++) {
    p1++;
    p1_len += *p1++;
  }

  // Calculate the length of the string to represent P2

  for(k = 0, p2_len = 0; k < p2_size; k++) {
    p2++;
    p2_len += *p2++;
  }

  // Allocate memory for the strings

  buf = (char *)xmalloc(14 * sizeof(char));
  ps1 = (char *)xmalloc(((p1_len * 13)-1) *
			sizeof(char));
  ps2 = (char *)xmalloc(((p2_len * 13)-1) *
			sizeof(char));
  ps1[0] = '\0';
  ps2[0] = '\0';

  p1 = i1->parikh;
  p2 = i2->parikh;
  p1++;
  p2++;

  for(k = 0, first = TRUE; k < p1_size; k++, p1++, p1++) {
    for(l = 0, p1_len = p1[1]; l < p1_len; l++) {
      sprintf(buf,
	      ((first != FALSE) ? ("t%.10lu") : (", t%.10lu")),
	      ((*p1) +1));
      strcat(ps1, buf);
      first = FALSE;
    }
  }

  for(k = 0, first = TRUE; k < p2_size; k++, p2++, p2++) {
    for(l = 0, p2_len = p2[1]; l < p2_len; l++) {
      sprintf(buf,
	      ((first != FALSE) ? ("t%.10lu") : (", t%.10lu")),
	      ((*p2) +1));
      strcat(ps2, buf);
      first = FALSE;
    }
  }


#if 0
  fprintf(stdout, "P1: < %s >\n", ps1);
  fprintf(stdout, "P2: < %s >\n", ps2);
#endif


  // Compare the two strings lexicographically
  // Normalize the results to -1,0, and 1

  compare = strcmp(ps1, ps2);
  if(compare < 0) {
    compare = -1;
  } else if(compare > 0) {
    compare = 1;
  }


#if 0
  if(compare == -1) {
    fprintf(stdout, "P1 < P2\n");
  } else if(compare == 1) {
    fprintf(stdout, "P1 > P2\n");
  } else {
    fprintf(stdout, "P1 = P2\n");
  }
#endif

  free(ps2);
  free(ps1);
  free(buf);

#endif DEBUG_PARIKH



  // First get the Parikh multiset lengths

  t1 = i1->parikh;
  t2 = i2->parikh;
  t1_size = *t1;
  t2_size = *t2;

  // Handle the common prefix

  p_min = MIN(t1_size, t2_size);

  for(i = 0; i < p_min; ) {

    i++;
    t1++;
    t2++;

    if(*t1 < *t2) {
      // The smaller transition exists in C1 but not in C2, C1 is smaller
#ifdef DEBUG_PARIKH
      ASSERT(compare == -1);
#endif /* DEBUG_PARIKH */
      return (-1);
    } else if(*t1 > *t2) {
      // The smaller transition exists in C2 but not in C1, C2 is smaller
#ifdef DEBUG_PARIKH
      ASSERT(compare == 1);
#endif /* DEBUG_PARIKH */
      return (1);
    }

    // The transition exists in both, have to check the counts then!
    t1++;
    t2++;

    // The string end is handled differently than the string prefix
    if(i != p_min) {

      // Handle the normal case first
      // This is not the end of either of the strings
      // The string with more small letters is smaller!

      if(*t1 > *t2) {
	// C1 has more instances of this transition, C1 is smaller
#ifdef DEBUG_PARIKH
	ASSERT(compare == -1);
#endif /* DEBUG_PARIKH */
	return (-1);
      } else if (*t1 < *t2) {
	// C2 has more instances of this transition, C2 is smaller
#ifdef DEBUG_PARIKH
	ASSERT(compare == 1);
#endif /* DEBUG_PARIKH */
	return (1);
      }

      // The counts match. Well, try the next transition then.

    } else {

      // This is the last index. (One of the strings end here.)
      // (And both strings have the same transition at this index.)
      //
      // I made a tedious case analysis to come up with the code
      // in here. Hopefully it's now OK, I got it mixed up the first
      // time around.


      // Actually the sizes need to be compared first in this case!

      if(t1_size < t2_size) {
	// C1 ends first, C1 is smaller
#ifdef DEBUG_PARIKH
	ASSERT(compare == -1);
#endif /* DEBUG_PARIKH */
	return (-1);
      } else if(t1_size > t2_size) {
	// C2 ends first, C2 is smaller
#ifdef DEBUG_PARIKH
	ASSERT(compare == 1);
#endif /* DEBUG_PARIKH */
	return (1);
      }


      // Both strings end here.
      // The logic is reversed, and the shorter string is now smaller!

      if(*t1 < *t2) {
	// C1 is the shorter one, C1 is smaller
#ifdef DEBUG_PARIKH
	ASSERT(compare == -1);
#endif /* DEBUG_PARIKH */
	return (-1);
      } else if (*t1 > *t2) {
	// C2 is the shorter one, C2 is smaller
#ifdef DEBUG_PARIKH
	ASSERT(compare == 1);
#endif /* DEBUG_PARIKH */
	return (1);
      }

      // The prefixes of the Parikh vectors are identical!

    }

  }


  // The common prefixes of the vectors match, have to compare sizes

  if(t1_size < t2_size) {
    // C1 ends first, C1 is smaller
#ifdef DEBUG_PARIKH
    ASSERT(compare == -1);
#endif /* DEBUG_PARIKH */
    return (-1);
  } else if(t1_size > t2_size) {
    // C2 ends first, C2 is smaller
#ifdef DEBUG_PARIKH
    ASSERT(compare == 1);
#endif /* DEBUG_PARIKH */
    return (1);
  }

#ifdef DEBUG_PARIKH
  ASSERT(compare == 0);
#endif /* DEBUG_PARIKH */


  // The Parikh vectors match, have to compare Foata normal forms

  if(i1->foata == NULL) {
    i1->foata = prefix->GetFoata(e1_idx);
  }
  if(i2->foata == NULL) {
    i2->foata = prefix->GetFoata(e2_idx);
  }

  t1 = i1->foata;
  t2 = i2->foata;

  // Get the number of levels in both configs
  
  e1_levels = *t1++;
  e2_levels = *t2++;
  l_min = MIN(e1_levels, e2_levels);

  // Thankfully Foata levels are sets, and therefore a little bit
  // easier to handle than Parikh vectors above.

  for(i = 0; i < l_min; i++) {

    t1_size = *t1++;
    t2_size = *t2++;

#define PEP_FOATA_COMPARATOR
#ifdef PEP_FOATA_COMPARATOR

    // PEP uses silex order of Foata levels instead of lexicographical order!

    if(t1_size < t2_size) {
      // This level is narrower in C1, C1 is smaller
      return (-1);
    } else if(t1_size > t2_size) {
      // This level is narrower in C2, C2 is smaller
      return (1);
    }

    // Levels are as wide, compare the levels

#endif /* PEP_FOATA_COMPARATOR */

    // Compare the prefixes

    p_min = MIN(t1_size, t2_size);

    for(j = 0; j < p_min; j++, t1++, t2++) {

      if(*t1 < *t2) {
	// This transition is smaller in C1, C1 is smaller
	return (-1);
      } else if(*t1 > *t2) {
	// This transition is smaller in C2, C2 is smaller
	return (1);
      }

    }

#ifndef PEP_FOATA_COMPARATOR

    // Prefixes match!
    // The ERV paper uses the lexicographic order,
    // compare sizes after prefixes match.

    if(t1_size < t2_size) {
      // This level is narrower in C1, C1 is smaller
      return (-1);
    } else if(t1_size > t2_size) {
      // This level is narrower in C2, C2 is smaller
      return (1);
    }

    // Levels are identical, try next level!

#endif /* PEP_FOATA_COMPARATOR */

  }

  // We should only end up here if the configurations are the same.
  // Sanity check that this really is the case, and no levels were
  // left unprocessed.

  ASSERT(e1_levels == e2_levels);

  return (0);

}

#undef MIN
