#lang racket
;; Copyright © 2013 Tuomas Kuismin

;; Permission is hereby granted, free of charge, to any person obtaining a copy
;; of this software and associated documentation files (the "Software"), to deal
;; in the Software without restriction, including without limitation the rights
;; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
;; copies of the Software, and to permit persons to whom the Software is
;; furnished to do so, subject to the following conditions:

;; The above copyright notice and this permission notice shall be included in
;; all copies or substantial portions of the Software.

;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
;; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
;; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
;; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
;; THE SOFTWARE.

(require
	srfi/13
	"aiger.rkt")

(define (count-to bit n)
	(let ([latches (a-latches n "counter-bit-~a")])
		(foldl
			(λ (latch carry)
				(a-set-latch-next latch (a-xor latch carry))
				(a-and latch carry))
			bit
			latches)))

(module+ main
	(command-line #:args (input)
		(when (string-suffix? ".smv" input)
			(system* (find-executable-path "smv2aig") "-f" input)
			(set! input (path->string (path-replace-suffix input ".aig"))))
		(for ([n (in-naturals 1)])
			(printf "Bounding acceptance with ~a-bit counter\n" n)
			(aiger-read input)
			(a-bad
				(count-to
					(let*
						([watch-bits (flatten (map (λ (kind) (aiger-map kind (curry symbol-lit-map identity))) '(justice fairness)))]
							[latches (a-latches (length watch-bits) "justice-watcher-~a")]
							[increment (apply a-and latches)])
						(map
							(λ (l b)
								(a-set-latch-next l
									(a-and (a-not increment) (a-or l b))))
							latches watch-bits)
						increment)
					n)
				"Never")
			(map aiger-clear '(fairness justice))
			(let ([out-file (make-temporary-file (string-append "live2safe-round-" (number->string n) "-~a.aig"))])
				(aiger-write out-file)
				(let
					([result
						 (call-with-output-string
							 (λ (out)
								 (let ([proc (fifth (process*/ports out #f #f (find-executable-path "bip") "-dump-invar=1" ",treb" out-file))])
									 (with-handlers
										 ([exn:break?
											  (λ (e) (proc 'interrupt) (sleep 0.2) (proc 'kill) (exit 1))])
										 (proc 'wait)))))])
					(if (string-contains-ci result "invariant found")
						(let ([invar-file (make-temporary-file "live2safe-~a.invar")])
							(display result)
							(display-lines-to-file (takef (rest (dropf (string-split result "\n") (negate (curry equal? "Invariant")))) (negate (curry equal? ""))) invar-file #:exists 'truncate/replace)
							(displayln (string-append "To verify result, run \"bip -invar=" (path->string invar-file) " ,check-invar " (path->string out-file) "\""))
							(exit 0))
						(let ([fails-line (filter (λ (l) (string-contains l "fails at depth")) (string-split result "\n"))])
							(if (null? fails-line)
								(displayln "bip didn't report fail-depth!")
								(displayln (first fails-line)))))))
			(aiger-reset))))
