#lang racket
;; Copyright © 2013 Tuomas Kuismin

;; Permission is hereby granted, free of charge, to any person obtaining a copy
;; of this software and associated documentation files (the "Software"), to deal
;; in the Software without restriction, including without limitation the rights
;; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
;; copies of the Software, and to permit persons to whom the Software is
;; furnished to do so, subject to the following conditions:

;; The above copyright notice and this permission notice shall be included in
;; all copies or substantial portions of the Software.

;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
;; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
;; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
;; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
;; THE SOFTWARE.

(require
	ffi/unsafe
	ffi/unsafe/define)

;;; C-types
(define test? (equal? (path->string (file-name-from-path (find-system-path 'run-file)))  "racket"))
(define-ffi-definer define-aiger
	(ffi-lib
		(build-path
			(if test?
				(build-path (find-system-path 'home-dir) "koodi" "live2safe")
				(path-only
					(find-system-path 'run-file))) "libaiger")))
(define-cstruct _aiger_symbol
	([lit _uint]
		[next _uint]
		[reset _uint]
		[size _uint]
		[lits _pointer]
		[name _string]))
(define-cstruct _aiger_and
	([lhs _uint]
		[rhs0 _uint]
		[rhs1 _uint]))
(define-cstruct _aiger
	([maxvar _uint]
		[num_inputs _uint]
		[num_latches _uint]
		[num_outputs _uint]
		[num_ands _uint]
		[num_bad _uint]
		[num_constraints _uint]
		[num_justice _uint]
		[num_fairness _uint]
		[inputs (_or-null _aiger_symbol-pointer)]
		[latches (_or-null _aiger_symbol-pointer)]
		[outputs (_or-null _aiger_symbol-pointer)]
		[bad (_or-null _aiger_symbol-pointer)]
		[constraints (_or-null _aiger_symbol-pointer)]
		[justice (_or-null _aiger_symbol-pointer)]
		[fairness (_or-null _aiger_symbol-pointer)]
		[ands (_or-null _aiger_and-pointer)]
		[comments _pointer]))
(define _aiger_mode (_enum '(aiger_binary_mode aiger_ascii_mode aiger_stripped_mode)))
(define-aiger aiger_version (_fun -> _string))
(define-aiger aiger_init (_fun -> _aiger-pointer))
(define-aiger aiger_reset (_fun _aiger-pointer -> _void))
(define-aiger aiger_write_to_string (_fun _aiger-pointer _aiger_mode _bytes _size -> _int))
(define-aiger aiger_open_and_write_to_file (_fun _aiger-pointer _path -> _int))
(define-aiger aiger_open_and_read_from_file (_fun _aiger-pointer _path -> _string))
(define-aiger aiger_add_input (_fun _aiger-pointer _uint _string -> _void))
(define-aiger aiger_add_latch (_fun _aiger-pointer _uint _uint _string -> _void))
(define-aiger aiger_add_output (_fun _aiger-pointer _uint _string -> _void))
(define-aiger aiger_add_bad (_fun _aiger-pointer _uint _string -> _void))
(define-aiger aiger_add_constraint (_fun _aiger-pointer _uint _string -> _void))
(define-aiger aiger_add_justice (_fun _aiger-pointer _uint _uintptr _string -> _void))
(define-aiger aiger_add_fairness (_fun _aiger-pointer _uint _string -> _void))
(define-aiger aiger_add_and (_fun _aiger-pointer _uint _uint _uint -> _void))
(define-aiger aiger_is_latch (_fun _aiger-pointer _uint -> _aiger_symbol-pointer))

;;; Util

(define (symbol-append s . ss)
	(string->symbol (apply string-append (symbol->string s) (map symbol->string ss))))
(define (binary->n-ary f op ops)
	(if (null? ops)
		op
		(f op (binary->n-ary f (first ops) (rest ops)))))
(define-namespace-anchor aiger-module)

;;; Interface

(define (aiger-new a)
	(let ([var (add1 (aiger-maxvar a))])
		(set-aiger-maxvar! a var)
		var))
(define (aiger-lit var)
	(* 2 var))
(define (aiger-var lit)
	(/ (if (odd? lit) (sub1 lit) lit) 2))
(define (a-not lit)
	(if (even? lit)
		(add1 lit)
		(sub1 lit)))

(define (aiger-clear kind)
	(apply (eval (symbol-append 'set-aiger-num_ kind '!) (namespace-anchor->namespace aiger-module)) (list *aiger* 0)))
(define (aiger-ref kind i)
	(ptr-ref (apply (eval (symbol-append 'aiger- kind) (namespace-anchor->namespace aiger-module)) (list *aiger*)) _aiger_symbol i))
(define (aiger-count kind)
	(apply (eval (symbol-append 'aiger-num_ kind) (namespace-anchor->namespace aiger-module)) (list *aiger*)))
(define (aiger-for kind f)
	(for ([i (aiger-count kind)])
		(f (aiger-ref kind i))))
(define (aiger-map kind f)
	(build-list (aiger-count kind) (λ (i) (f (aiger-ref kind i)))))
(define (symbol-lit-ref s i)
	(ptr-ref (aiger_symbol-lits s) _uint i))
(define (symbol-name s)
	(aiger_symbol-name s))
(define (symbol-size s)
	(aiger_symbol-size s))
(define (symbol-for-lits f s)
	(for ([i (aiger_symbol-size s)])
		(f (symbol-lit-ref s i))))
(define (symbol-lit-map f s)
	(build-list (aiger_symbol-size s) (λ (i) (f (symbol-lit-ref s i)))))

(define *aiger* (aiger_init))
(define (aiger-read file)
	(aiger_open_and_read_from_file *aiger* file))
(define (aiger-write file)
	(aiger_open_and_write_to_file *aiger* file))
(define (aiger-reset)
	(aiger_reset *aiger*)
	(set! *aiger* (aiger_init)))
(define (aiger-visualize [keep #f])
	(let ([temp (make-temporary-file "racket-aig-~a.aag")])
		(aiger_open_and_write_to_file *aiger* temp)
		(system* (find-executable-path "aigvis") temp)
		(if keep
			(displayln (format "Keeping temporary AIGER file ~a" temp))
			(delete-file temp))))
(define (aiger-add kind . args)
	(let ([new-lit (aiger-lit (aiger-new *aiger*))])
		(apply (eval (symbol-append 'aiger_add_ kind) (namespace-anchor->namespace aiger-module)) *aiger* new-lit args)
		new-lit))

(define (a-latch [name #f])
	(let ([new-latch (aiger-lit (aiger-new *aiger*))])
		(aiger_add_latch *aiger* new-latch new-latch name)
		new-latch))
(define (a-latches n [template "latch~a"])
	(build-list n (λ (i) (a-latch (format template i)))))
(define (a-and op . ops)
	(binary->n-ary (λ (o1 o2) (aiger-add 'and o1 o2)) op ops))
(define (a-xor op . ops)
	(binary->n-ary
		(λ (o1 o2)
			(let ([both (a-and o1 o2)]
					 [neither (a-and (a-not o1) (a-not o2))])
				(a-and (a-not both) (a-not neither))))
		op ops))
(define (a-or op . ops)
	(binary->n-ary (λ (o1 o2) (a-not (a-and (a-not o1) (a-not o2)))) op ops))
(define (a-set-latch-next latch next)
	(set-aiger_symbol-next! (aiger_is_latch *aiger* latch) next))
(define (a-bad lit name)
	(aiger_add_bad *aiger* lit name))
(define (a-output lit name)
	(aiger_add_output *aiger* lit name))

(define test-input (build-path (find-system-path 'home-dir) "Projektit" "aiger-malleja" "counter.aag"))
(define (test)
	(aiger-read test-input))

(provide aiger-reset a-not aiger-clear aiger-ref aiger-count aiger-for aiger-map symbol-lit-ref symbol-lit-map symbol-for-lits symbol-name symbol-size aiger-read aiger-write aiger-visualize a-latch a-latches a-and a-xor a-or a-set-latch-next a-output a-bad (struct-out aiger_symbol))

;;; Footer
;; Local Variables:
;; eval: (orgstruct-mode)
;; End:
