/**CFile***********************************************************************

  FileName    [mcCmd.c]

  PackageName [mc]

  Synopsis    [Model checking commands.]

  Description [This file contains all the shell command to deal with
  model checking and for counterexample navigation.]

  SeeAlso     [cmdCmd.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``mc'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include "mc.h"
#include "mcInt.h" 
#include "parser/symbols.h"
#include "utils/error.h" /* for CATCH */
#include "prop/prop.h"
#include "cmd/cmd.h"
#include "utils/utils_io.h"
#include "trace/Trace.h"
#include "trace/TraceManager.h"
#include "enc/enc.h"


static char rcsid[] UTIL_UNUSED = "$Id: mcCmd.c,v 1.23.2.18.2.1 2004/08/04 13:18:35 nusmv Exp $";

/* prototypes of the command functions */
int CommandCheckSpec ARGS((int argc, char **argv));
int CommandCheckInvar ARGS((int argc, char **argv));
int CommandCheckCompute ARGS((int argc, char **argv));

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int UsageCheckSpec ARGS((void));
static int UsageCheckInvar ARGS((void));
static int UsageCheckCompute ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the mc package.]

  Description        [Initializes the mc package.]

  SideEffects        []

******************************************************************************/
void Mc_Init(void){
  Cmd_CommandAdd("check_spec", CommandCheckSpec, 0);
  Cmd_CommandAdd("check_invar", CommandCheckInvar, 0);
  Cmd_CommandAdd("compute", CommandCheckCompute, 0);
}

/**Function********************************************************************

  Synopsis           [Quit the mc package]

  Description        [Quit the mc package]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Mc_End(void)
{ }

/**Function********************************************************************

  Synopsis           [Performs fair CTL model checking.]

  CommandName        [check_spec]          

  CommandSynopsis    [Performs fair CTL model checking.]  

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[-n number | -p "ctl-expr \[IN context\]"\]]  

  CommandDescription [Performs fair CTL model checking.<p>

  A <tt>ctl-expr</tt> to be checked can be specified at command line 
  using option <tt>-p</tt>. Alternatively, option <tt>-n</tt> can be used
  for checking a particular formula in the property database. If 
  neither <tt>-n</tt> nor <tt>-p</tt> are used, all the SPEC formulas in
  the database are checked.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>SPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the <tt>UNIX</tt> command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>SPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>-p "ctl-expr \[IN context\]"</tt>
       <dd> A CTL formula to be checked. <tt>context</tt> is the module 
       instance name which the variables in <tt>ctl-expr</tt> must 
       be evaluated in.       
    <dt> <tt>-n number</tt>
       <dd> Checks the CTL property with index <tt>number</tt> in the property
            database. 
  </dl><p>

  If the <tt>ag_only_search</tt> environment variable has been set, and
  the set of reachable states has already been computed, then a
  specialized algorithm to check AG formulas is used instead of the
  standard model checking algorithms.]  

  SideEffects        []

******************************************************************************/
int CommandCheckSpec(int argc, char **argv)
{
  int c;
  int prop_no = -1;
  char * formula = NIL(char);
  int status = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:n:p:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckSpec());
    case 'n':
      {
        if (formula != NIL(char)) return(UsageCheckSpec());
        if (prop_no != -1) return(UsageCheckSpec());

        prop_no = PropDb_get_prop_index_from_string(util_optarg);
        if (prop_no == -1) return(1);

        break;
      }
    case 'p':
      {
        if (prop_no != -1) return(UsageCheckSpec());
        if (formula != NIL(char)) return(UsageCheckSpec());

        formula = util_strsav(util_optarg);
        break;
      }
    case 'o':
      if (useMore == 1) return(UsageCheckSpec());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckSpec());
      useMore = 1;
      break;
    default:  return(UsageCheckSpec());
    }
  }
  if (argc != util_optind) return(UsageCheckSpec());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, 
            "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if ( (!cmp_struct_get_build_model(cmps)) 
       && (opt_cone_of_influence(options) == false) ) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n",
            TransType_to_string(get_partition_method(options)));
    fprintf(nusmv_stderr, 
            "Use \t \"build_model -f -m %s\"\nto build the transition relation.\n",
            TransType_to_string(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
      nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  
  if (global_fsm_builder == FSM_BUILDER(NULL)) {
    /* to create global_fsm_builder instance */
    Compile_InitializeBuildModel(); 
  }


  if (formula != NIL(char)) {
    prop_no = PropDb_prop_parse_and_add(Enc_get_symb_encoding(),
					formula, Prop_Ctl);
    if (prop_no == -1) return(1); /* error */
    CATCH {
      PropDb_verify_prop_at_index(prop_no);
    }
    FAIL {
      status = 1;
    }
  }
  else if (prop_no != -1) {
    if (Prop_check_type(PropDb_get_prop_at_index(prop_no), Prop_Ctl) != 0) {
      status = 1;
    }
    else {
      CATCH {
        PropDb_verify_prop_at_index(prop_no);
      }
      FAIL {
        status = 1;
      }
    }
  }
  else {
    CATCH {
      PropDb_verify_all_type(Prop_Ctl);
    }
    FAIL {
      status = 1;
    }
  }

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }

  return(status);
}

static int UsageCheckSpec()
{
  fprintf(nusmv_stderr, "usage: check_spec [-h] [-m | -o file] [-n number | -p \"ctl-expr\"]\n");
  fprintf(nusmv_stderr, "   -h \t\t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\t\tPipes output through the program specified\n");
  fprintf(nusmv_stderr, "      \t\t\tby the \"PAGER\" environment variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\t\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "   -n number\t\tChecks only the SPEC with the given index number.\n");
  fprintf(nusmv_stderr, "   -p \"ctl-expr\"\tChecks only the given CTL formula.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Performs model checking of invariants]

  CommandName        [check_invar]         

  CommandSynopsis    [Performs model checking of invariants]  

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[-n number | -p "invar-expr \[IN context\]"\]]  

  CommandDescription [Performs invariant checking on the given
  model. An invariant is a set of states. Checking the invariant is
  the process of determining that all states reachable from the
  initial states lie in the invariant.

  Invariants to be verified can be provided as simple formulas
  (without any temporal operators) in the input file via the
  <tt>INVARSPEC</tt> keyword or directly at command line, using the option 
  <tt>-p</tt>.<p>

  Option <tt>-n</tt> can be used for checking a particular invariant
  of the model. If neither <tt>-n</tt> nor <tt>-p</tt> are
  used, all the invariants are checked.<p>

  During checking of invariant all the fairness conditions associated 
  with the model are ignored.<p>

  If an invariant does not hold, a proof of failure is demonstrated.
  This consists of a path starting from an initial state to a state
  lying outside the invariant. This path has the property that it is the
  shortest path leading to a state outside the invariant.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the program in processing
           <tt>INVARSPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the UNIX command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>INVARSPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>-p "invar-expr \[IN context\]"</tt>
       <dd> The command line specified invariant formula to be verified.
       <tt>context</tt> is the module instance name which the variables 
       in <tt>invar-expr</tt> must be evaluated in.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckInvar(int argc, char **argv)
{
  int c;
  int prop_no = -1;
  char * formula = NIL(char);
  int status = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:n:p:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckInvar());
    case 'n':
      {
        if (formula != NIL(char)) return(UsageCheckInvar());
        if (prop_no != -1) return(UsageCheckInvar());

        prop_no = PropDb_get_prop_index_from_string(util_optarg);
        if (prop_no == -1) return(1);

        break;
      }
    case 'p':
      {
        if (prop_no != -1) return(UsageCheckInvar());
        if (formula != NIL(char)) return(UsageCheckInvar());

        formula = util_strsav(util_optarg);
        break;
      }
    case 'o':
      if (useMore == 1) return(UsageCheckInvar());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckInvar());
      useMore = 1;
      break;
    default:  return(UsageCheckInvar());
    }
  }
  if (argc != util_optind) return(UsageCheckInvar());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, 
            "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if ( (!cmp_struct_get_build_model(cmps)) 
       && (opt_cone_of_influence(options) == false) ) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", 
	    TransType_to_string(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the transition relation.\n", 
	    TransType_to_string(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  if (global_fsm_builder == FSM_BUILDER(NULL)) {
    /* to create global_fsm_builder instance */
    Compile_InitializeBuildModel(); 
  }

  if (formula != NIL(char)) {
    prop_no = PropDb_prop_parse_and_add(Enc_get_symb_encoding(), 
					formula, Prop_Invar);
    if (prop_no == -1) return(1); /* error */
    CATCH {
      PropDb_verify_prop_at_index(prop_no);
    }
    FAIL {
      status = 1;
    }
  }
  else if (prop_no != -1) {
    if (Prop_check_type(PropDb_get_prop_at_index(prop_no), Prop_Invar) != 0) {
      status = 1;
    }
    else {
      CATCH {
        PropDb_verify_prop_at_index(prop_no);
      }
      FAIL {
        status = 1;
      }
    }
  }
  else {
    CATCH {
      PropDb_verify_all_type(Prop_Invar);
    }
    FAIL {
      status = 1;
    }
  }

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }

  return(status);
}

static int UsageCheckInvar()
{
  fprintf(nusmv_stderr, "usage: check_invar [-h] [-m| -o file] [-n number | -p \"invar-expr\"]\n");
  fprintf(nusmv_stderr, "   -h \t\t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\t\tPipes output through the program specified by\n");
  fprintf(nusmv_stderr, "      \t\t\tthe \"PAGER\" shell variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\t\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "   -n number\t\tchecks only the INVARSPEC with the given index number.\n");
  fprintf(nusmv_stderr, "   -p \"invar-expr\"\tchecks only the given invariant formula.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Performs computation of quantitative characteristics]

  CommandName        [compute]     

  CommandSynopsis    [Performs computation of quantitative characteristics]

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[-n number | 
  -p "compute-expr \[IN context\]"\]]  

  CommandDescription [This command deals with the computation of
  quantitative characteristics of real time systems. It is able to
  compute the length of the shortest (longest) path from two given set
  of states.<p>
  
  <center><code>MAX \[ alpha , beta \]</code></center><br>
  <center><code>MIN \[ alpha , beta \]</code></center><p>
  
  Properties of the above form can be specified in the input file via
  the keyword <code>COMPUTE</code> or directly at command line,
  using option <tt>-p</tt>.<p>

  Option <tt>-n</tt> can be used for computing a particular expression
  in the model. If neither <tt>-n</tt> nor <tt>-p</tt> are
  used, all the COMPUTE specifications are computed.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>COMPUTE</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the UNIX command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>COMPUTE</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>-p "compute-expr \[IN context\]"</tt>
       <dd> A COMPUTE formula to be checked. <tt>context</tt> is the module 
       instance name which the variables in <tt>compute-expr</tt> must 
       be evaluated in. 
    <dt> <tt>-n number</tt>
       <dd> Computes only the property with index <tt>number</tt>
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckCompute(int argc, char **argv)
{
  int c;
  int prop_no = -1;
  char * formula = NIL(char);
  int status = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:n:p:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckCompute());
    case 'n':
      {
        if (formula != NIL(char)) return(UsageCheckCompute());
        if (prop_no != -1) return(UsageCheckCompute());

        prop_no = PropDb_get_prop_index_from_string(util_optarg);
        if (prop_no == -1) return(1);

        break;
      }
    case 'p':
      {
        if (prop_no != -1) return(UsageCheckCompute());
        if (formula != NIL(char)) return(UsageCheckCompute());

        formula = util_strsav(util_optarg);
        break;
      }
    case 'o':
      if (useMore == 1) return(UsageCheckCompute());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckCompute());
      useMore = 1;
      break;
    default:  return(UsageCheckCompute());
    }
  }
  if (argc != util_optind) return(UsageCheckCompute());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, 
            "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if ( (!cmp_struct_get_build_model(cmps)) 
       && (opt_cone_of_influence(options) == false) ) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", 
	    TransType_to_string(get_partition_method(options)));
    fprintf(nusmv_stderr, 
	    "Use \t \"build_model -f -m %s\"\nto build the transition relation.\n", 
	    TransType_to_string(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  if (global_fsm_builder == FSM_BUILDER(NULL)) {
    /* to create global_fsm_builder instance */
    Compile_InitializeBuildModel(); 
  }

  if (formula != NIL(char)) {
    prop_no = PropDb_prop_parse_and_add(Enc_get_symb_encoding(), 
					formula, Prop_Compute);
    if (prop_no == -1) return(1); /* error */
    CATCH {
      PropDb_verify_prop_at_index(prop_no);
    }
    FAIL {
      status = 1;
    }
  }
  else if (prop_no != -1) {
    if (Prop_check_type(PropDb_get_prop_at_index(prop_no), Prop_Compute) != 0) {
      status = 1;
    }
    else {
      CATCH {
        PropDb_verify_prop_at_index(prop_no);
      }
      FAIL {
        status = 1;
      }
    }
  }
  else {
    CATCH {
      PropDb_verify_all_type(Prop_Compute);
    }
    FAIL {
      status = 1;
    }
  }

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }

  return(status);
}

static int UsageCheckCompute()
{
  fprintf(nusmv_stderr, "usage: compute [-h] [-m | -o file] [-n number | -p \"compute-expr\"]\n");
  fprintf(nusmv_stderr, "   -h \t\t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\t\tPipes output through the program specified by\n");
  fprintf(nusmv_stderr, "      \t\t\tthe \"PAGER\" shell variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\t\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "   -n number\t\tConsiders only the compute expression with the given index number.\n");
  fprintf(nusmv_stderr, "   -p \"compute-expr\"\tComputes the given expression.\n");
  return(1);
}

