/*
 * RecordIndex.java
 *
 * Copyright 2001 by BRiSK Software,
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package makedocjr;

import java.io.*;

/**
 * <code>RecordIndex</code> class is used to read/write Palm database record indexes
 * NOTE: Information about the various fields was taken from <a href="www.roadcoders.com/pdb.html">Road Coders</a>
 * @author Jeffrey A. Krzysztow
 * @author Pat Beirne
 * @version 1.0
 */
public class RecordIndex {
	/**
	 * the offset to the record
	 */
	public int fileOffset = -1;	// 4 DWord

	/**
	 * the attribute of the record
	 *  0x10 Private record.
	 *  0x20 Record in use.
	 *  0x40 Dirty (archive) record.
	 *  0x80 Delete record.
	 *  The least significant four bits are used to represent the category values.
	 */
	public byte attribute = 0;	// 1 Byte

	/**
	 * Indicates the record is private
	 */
	public final static byte PRIVATE = 0x10;

	/**
	 * unique ID required for the RecordIndex
	 */
	public int uniqueID = 0;	// 3 Bytes used

	/**
	 * The size of the RecordIndex in bytes
	 * @return the number of bytes in the RecordIndex
	 */
	public static int getSize() {
		return(8);
	}

	/**
	 * Reads the RecordIndex from the DataInput
	 * @param the DataInput to read from
	 */
	public void read(DataInput di) throws IOException {
		fileOffset = di.readInt();
		attribute = di.readByte();
		uniqueID = ((0xff & di.readByte()) << 16) + ((0xff & di.readByte()) << 8) + (di.readByte() & 0xff);
	}

	/**
	 * Writes the RecordIndex to the DataInput
	 * @param the DataInput to write to
	 */
	public void write(DataOutput out) throws IOException {
		out.writeInt(fileOffset);
		out.write(attribute);
		out.write((byte)(uniqueID >> 16));
		out.write((byte)(uniqueID >> 8));
		out.write((byte) uniqueID);
	}

	/**
	 * override Object.toString()
	 */
	public String toString() {
		return ">> Record Index <<"
			+ "\nfileOffset = " + fileOffset
			+ "\nattribute = " + attribute
			+ "\nuniqueID = " + uniqueID;
	}

}

