/*
 * PilotDocRecord.java
 *
 * Copyright 2001 Steve Held
 * Copyright 2001 by BRiSK Software,
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package makedocjr;

/**
 * <code>PilotDocRecord</code> class handles PilotDoc records. It can
 * compress, decompress, remove binary codes, duplicate \n codes
 *
 * @author Jeffrey A. Krzysztow
 * @author Steve Held
 * @author Pat Beirne
 * @version 1.0
 */

import java.io.*;

public class PilotDocRecord {
	final static int DISP_BITS = 11;
	final static int COUNT_BITS = 3;
	final static int DEF_LEN = 4096;	// 4096 is the magic number of bytes

	/**
	 * represents a linefeed at the end of each line
	 */
	public final static int EOLLF = 1;

	/**
	 * represents a carriage return at the end of each line
	 */
	public final static int EOLCR = 2;

	/**
	 * represents a carriage return and a linefeed at the end of each line
	 */
	public final static int EOLCRLF = 3;

	/**
	 * internal buffer holding PilotDocRecord
	 */
	byte [] buf = null;

	// during translations, this it the destination buffer
	private byte [] destBuf = null;
	private int destIndex = 0;
	private boolean space = false;

	/**
	 * Constructs a PilotDocRecord of default size
	 */
	PilotDocRecord() {
		this(DEF_LEN);
	}

	/**
	 * Constructs a PilotDocRecord of the specified size
	 * @param size size to create PilotDocRecord
	 */
	PilotDocRecord(int size) {
		buf = new byte[size];
	}

	/**
	 * Returns the number of bytes in the PilotDocRecord
	 * @return the number of bytes in the PilotDocRecord
	 */
	public int length() {
		return buf.length;
	}

	/**
	 * Remove the binary codes throw away really low ASCII or above '~' from the buffer
	 * @return the resulting buffer length
	 */
	public int removeBinary() {
		destBuf = new byte[(buf.length * 3) / 2];
		int k, j;
		for(j = k = 0; j < buf.length; j++, k++) {
			// copy each byte
			destBuf[k] = buf[j];

			// throw away really low ASCII
			if((destBuf[k] >= 0x00 && destBuf[k] < 0x09)) {
				k--;
			}
		}
		assign(destBuf, k);
		return k;
	}

	/**
	 * Change all 0x0a into 0x0d, 0x0a pairs or 0x0a into 0x0d
	 * @return the resulting buffer length
	 */
	public int convertEOL(final int eolType) {
		destBuf = new byte[(buf.length * 3) / 2];
		int k, j;
		for(j = k = 0; j < buf.length; j++, k++) {
			destBuf[k] = buf[j];
			if(destBuf[k] == 0x0a) {
				if(eolType == EOLCRLF) {
					destBuf[k++] = 0x0d;
					destBuf[k] = 0x0a;
				}
				else {	// EOLCR
					destBuf[k] = 0x0d;
				}
			}
		}
		assign(destBuf, k);
		return k;
	}

	/**
	 * if a CR appears before a LF, then remove the CR
	 * if a CR appears in isolation, change to a LF
	 * @return the resulting buffer length
	 */
	public int convertEOL() {
		destBuf = new byte[(buf.length * 3) / 2];
		int k, j;
		for(j = k = 0; j < buf.length; j++, k++) {
			// copy each byte
			destBuf[k] = buf[j];

			if(destBuf[k] == 0x0d) {
				// if next is LF, then drop it
				if(j < buf.length - 1 && buf[j + 1] == 0x0a) {
					k--;
				}
				else { // turn it into a LF
					destBuf[k] = 0x0a;
				}
			}
		}
		assign(destBuf, k);
		return k;
	}

	/**
	 * unwrap single LFs are removed
	 * @param the minimum line length to wrap
	 * @return the resulting buffer length
	 */
	public int unwrap(int lineLength) {
		destBuf = new byte[(buf.length * 3) / 2];
		int k, j, lastLFpos = -1, x, y;
		boolean patternFound;

		for(j = k = 0; j < buf.length; j++, k++) {
			destBuf[k] = buf[j];

			if(destBuf[k] == 0x0a) {
				// Test for some special text patterns
				patternFound = false;

				if(patternFound == false) { // have not found a pattern, so look for one
					patternFound = true; // be optimistic
					// test to see if mirror of each other
					// test to see if all bytes are the same
					for(x = lastLFpos + 1, y = k - 1; x < y; x++, y--) {
						if(destBuf[x] != destBuf[y]){
							patternFound = false;
							break;
						}
					}
					if(patternFound) {
						if(lastLFpos > -1) {
							destBuf[lastLFpos] = 0x0a; // put the LF back so we have a "complete" line
						}
					}
				}

				if(patternFound == false) { // unwrap
					if ((k - lastLFpos) > lineLength) {
						if(destBuf[k - 1] == 0x20 || destBuf[k - 1] == 0x0a) {
							k--;
						}
						else {
							destBuf[k] = 0x20;
						}
					}
				}
				lastLFpos = k;
			}
		}
		assign(destBuf, k);
		return k;
	}

	/**
	 * compressEOL compresses double LF into single LF
	 * @return the resulting buffer length
	 */
	public int compressEOL() {
		destBuf = new byte[(buf.length * 3) / 2];
		int k, j;
		for(j = k = 0; j < buf.length; j++, k++) {
			// copy each byte
			destBuf[k] = buf[j];

			if(k > 0 && destBuf[k] == 0x0a && destBuf[k - 1] == 0x0a) {
				k--;
			}
		}
		assign(destBuf, k);
		return k;
	}

	/**
	 * Decompress the buffer
	 * action: make a new buffer
	 *				run through the source data
	 *				check the 4 cases:
	 *					0,9...7F represent self
	 *					1...8		escape n chars
	 *					80...bf reference earlier run
	 *					c0...ff	space+ASCII
	 * @return the resulting buffer length
	 */
	public int decompress() {
		// we "know" that all decompresses fit within DocumentHeader.textRecordSize, right?
		destBuf = new byte[(DocumentHeader.textRecordSize * 2)];
		destIndex = 0;

		int i,j;
		for(j = i = 0; j < buf.length;) {
			int c;

			// take a char from the input buffer
			c = buf[j++] & 0xff;

			// separate the char into zones: 0, 1...8, 9...0x7F, 0x80...0xBF, 0xC0...0xFF

			// codes 1...8 mean copy that many bytes; for accented chars & binary
			if(c > 0 && c < 9) {
				while(c-- > 0) {
					destBuf[destIndex++] = buf[j++];
				}
			}

			// codes 0, 9...0x7F represent themselves
			else if(c < 0x80) {
				destBuf[destIndex++] = (byte) c;
			}

			// codes 0xC0...0xFF represent "space + ascii char"
			else if(c >= 0xC0) {
				destBuf[destIndex++] = (byte) ' ';
				destBuf[destIndex++] = (byte) (c ^ 0x80);
			}

			// codes 0x80...0xBf represent sequences
			else {
				int m,n;
				c <<= 8;
				c += buf[j++] & 0xff;
				m = (c & 0x3fff) >> COUNT_BITS;
				n = c & ((1 << COUNT_BITS) - 1);
				n += 3;
				while(n-- > 0) {
					destBuf[destIndex] = destBuf[destIndex - m];
					destIndex++;
				}
			}
		}
		assign(destBuf, destIndex);
		return destIndex;
	}

	/**
	 * compress the buffer
	 * @return the resulting buffer length
	 */
	public int compress() {
		int hit = 0;			// points to a walking test hit; works upwards on successive matches
		int prevHit = 0; 		// previous value of hit
		int testHead = 0;		// current test string
		int testTail = 1;		// current walking pointer; one past the current test buffer
		int end = buf.length;	// 1 past the end of the input buffer

		space = false;

		// make a destination buffer
		destBuf = new byte[(buf.length*3)/2];
		destIndex = 0;	// used to walk through the output buffer

		// make a source string, so we can search it quickly
		String src = new String(buf);
		String testStr = "";

		// loop, absorbing one more char from the input buffer on each pass
		for(; testHead != end; testTail++) {
			if(testHead - prevHit > ((1 << DISP_BITS) - 1)) {
				prevHit = testHead - ((1 << DISP_BITS) - 1);
			}

			// scan in the previous data for a match
			testStr = src.substring(testHead, testTail);
			hit = src.indexOf(testStr, prevHit);

			// on a mismatch or end of buffer, issue codes
			if(hit == -1
				|| hit == testHead
				|| testTail - testHead > (1 << COUNT_BITS) + 2
				|| testTail == end) {

				// first, check for short runs
				if(testTail - testHead < 4) {
					issue(buf[testHead]);
					testHead++;
				}

				// for inter runs, issue a run-code
				else {
					// issue space char if required
					if(space) {
						destBuf[destIndex++] = (byte) ' ';
						space = false;
					}

					int dist = testHead - prevHit;
					int compound = (dist << COUNT_BITS) + testTail - testHead - 4;

					destBuf[destIndex++] = (byte) (0x80 + (compound >> 8));
					destBuf[destIndex++] = (byte) (compound & 0xff);
					testHead = testTail - 1;
				}
				// start the search again
				prevHit = 0;
			}
			// got a match
			else {
				prevHit = hit;
			}
			// when we get to the end of the buffer, don't inc past the end
			// this forces the residue chars out one at a time
			if(testTail == end) {
				testTail--;
			}
		}

		// clean up any dangling spaces
		if(space) {
			destBuf[destIndex++] = (byte) ' ';
		}
		// final scan to merge consecutive high chars together
		int k, i, destTemp;
		for(i = k = 0; i < destIndex; i++, k++) {
			destBuf[k] = destBuf[i];
			destTemp = destBuf[k] & 0xff;
			// skip the run-length codes
			if(destTemp >= 0x80 && destTemp < 0xc0) {
				destBuf[++k] = destBuf[++i];
			}
			// if we hit a high char marker, look ahead for another
			else if(destBuf[k] == 1) {
				destBuf[k + 1] = destBuf[i + 1];
				while(i + 2 < destIndex && destBuf[i + 2] == 1 && destBuf[k] < 8) {
					destBuf[k]++;
					destBuf[k + destBuf[k]] = destBuf[i + 3];
					i += 2;
				}
				k += destBuf[k];
				i++;
			}
		}

		assign(destBuf, k);
		return k;
	}

	/*
	 * Handle the details of writing a single character to the compressed stream
	 * used by compress()
	 */
	private void issue(byte src) {
		// if there is an outstanding space char, see if
		// we can squeeze it in with an ASCII char
		int iSrc = src;
		if(space) {
			if(iSrc >= 0x40 && iSrc <= 0x7F) {
				destBuf[destIndex++] = (byte)(iSrc ^ 0x80);
			}
			else {
				// couldn't squeeze it in, so issue the space char by itself
				// most chars go out simple, except the range 1...8,0x80...0xFF
				destBuf[destIndex++] = 0x20;
				if(iSrc < 0x80 && (iSrc == 0 || iSrc > 8)) {
					destBuf[destIndex++] = src;
				}
				else {
					destBuf[destIndex++] = 1;
					destBuf[destIndex++] = src;
				}
			}
			// knock down the space flag
			space = false;
		}
		else {
			// check for a space char
			if(src == 0x20) {
				space = true;
			}
			else {
				if(iSrc < 0x80 && (iSrc == 0 || iSrc > 8)) {
					destBuf[destIndex++] = src;
				}
				else {
					destBuf[destIndex++] = 1;
					destBuf[destIndex++] = src;
				}
			}
		}
	}

	/**
	 * sets the internal buffer to the byte array
	 * @param b byte array to to assign to the internal buffer
	 * @param n number of byte in b to use
	 */
	public void assign(byte[] b, int n) {
		buf = new byte[n];
		System.arraycopy(b, 0, buf, 0, n);
	}

	/**
	 * sets the internal buffer to the byte array
	 * @param b      byte array to to assign to the internal buffer
	 * @param n      number of byte in b to use
	 * @param offset starting location in b
	 */
	public void assign(byte[] b, int n, int offset) {
		buf = new byte[n];
		System.arraycopy(b, offset, buf, 0, n);
	}

}

