/*
 * MakeDocJr.java
 *
 * Copyright 2001 Steve Held,
 * Copyright 2001 by BRiSK Software,
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package makedocjr;

import java.io.*;
import java.util.*;
import java.util.zip.*;

/**
 * <code>MakeDocJr</code> is used to encode and decode DOC files.
 *
 * <p>MakeDocJr is not thread safe.
 *
 * @author Steve Held
 * @author Jeffrey A. Krzysztow
 * @author Reinhold F. Kainhofer
 * @author Pat Beirne
 * @version 1.1.1
 */

public class MakeDocJr implements Cloneable {

	// how to convert end of line
	static int eolType;

	static final byte[] CRLF = new byte[] {0x0D, 0x0A};//CRLF-->DOS
	static final byte[] CR = new byte[] {0x0D};//CR-->Mac
	static final byte[] LF = new byte[] {0x0A};//LF-->Unix

	private EncodeOptions encodeOptions;

	private String title = null;
	private byte[] textArray = null;
	private ArrayList bookmarks = new ArrayList(); // Must be makedocjr.Bookmark objects

	/**
	 * Constructs a new MakeDocJr with a new <code>EncodeOptions</code>
	 */
	public MakeDocJr() {
		// use system end-of-line byte code
		String eol = System.getProperty("line.separator");
		if (Arrays.equals(eol.getBytes(), MakeDocJr.CRLF)) {
			MakeDocJr.eolType = PilotDocRecord.EOLCRLF;
		} else if (Arrays.equals(eol.getBytes(), MakeDocJr.CR)) {
			MakeDocJr.eolType = PilotDocRecord.EOLCR;
		} else { //(Arrays.equals(eol.getBytes(), MakeDocJr.LF))
			MakeDocJr.eolType = PilotDocRecord.EOLLF;
		}
		encodeOptions = new EncodeOptions();
	}

	/**
	 * Constructs a new MakeDocJr with a specified <code>EncodeOptions</code>
	 *
	 * @param encodeOptions    the options used to encode the text to a DOC file
	 */
	public MakeDocJr(EncodeOptions encodeOptions) {
		// use system end-of-line byte code
		String eol = System.getProperty("line.separator");
		if (Arrays.equals(eol.getBytes(), MakeDocJr.CRLF)) {
			MakeDocJr.eolType = PilotDocRecord.EOLCRLF;
		} else if (Arrays.equals(eol.getBytes(), MakeDocJr.CR)) {
			MakeDocJr.eolType = PilotDocRecord.EOLCR;
		} else { //(Arrays.equals(eol.getBytes(), MakeDocJr.LF))
			MakeDocJr.eolType = PilotDocRecord.EOLLF;
		}
		this.encodeOptions = encodeOptions;
	}

	/**
	 * Decodes the DOC File source and sets this object's text and title.
	 *
	 * @param source    the source of the DOC File to be decoded
	 * @throws java.io.FileNotFoundException
	 * @throws java.io.IOException
	 * @throws java.util.zip.DataFormatException if DOC file does not appear to
	 * be valid.
	 */
	public void decode(File source)
			throws IOException, FileNotFoundException, DataFormatException {

		RandomAccessFile fin = new RandomAccessFile(source, "r");
		ByteArrayOutputStream bout = new ByteArrayOutputStream();

		DatabaseHeader dbHeader = new DatabaseHeader();
		try {
			dbHeader.read(fin);
		} catch (DataFormatException dfe) {
			throw new DataFormatException(source.getName() + " is not a " +
						"supported file format.");
		}
		if (dbHeader.creatorID != DatabaseHeader.ReaderID
			&& dbHeader.creatorID != DatabaseHeader.TealDocID
			|| dbHeader.typeID != DatabaseHeader.TEXt) {
			try {
				fin.close();
				bout.close();
			} catch ( IOException ioe ) {}
			throw new DataFormatException(source.getName() + " is not a " +
						"supported file format.");
		}
		encodeOptions.docID = dbHeader.creatorID;

		RecordIndex[] ri = new RecordIndex[dbHeader.numRecords];
		for (int i = 0; i < dbHeader.numRecords; i++) {
			ri[i] = new RecordIndex();
			ri[i].read(fin);
		}

		byte privateAttribute = (byte)(ri[0].attribute & RecordIndex.PRIVATE);
		encodeOptions.makePrivate = (privateAttribute == RecordIndex.PRIVATE) ? true : false;

		encodeOptions.category = (byte)(ri[0].attribute & 0xF);

		DocumentHeader docHeader = new DocumentHeader();
		fin.seek(ri[0].fileOffset);
		docHeader.read(fin);

//		if (docHeader.version != DocumentHeader.UNCOMPRESSED && docHeader.version != DocumentHeader.COMPRESSED) {
//			System.err.println("WARNING: unknown file compression type: " + docHeader.version);
//		}

		boolean compressed = (docHeader.version == DocumentHeader.COMPRESSED);
		encodeOptions.compress = compressed;

		/////////////////////////////////////////////////////////////
		// Handle the text (compressed or not)
		PilotDocRecord textBuffer;
		int dwRecLen;
		for (int i = 1; i <= docHeader.numRecords; i++) {
			// System.out.println("Converting `" + dbHeader.name + "`: text record " + i + " of " + docHeader.numRecords);
			if (i == ri.length - 1) {
				// for the last, use the file len
				dwRecLen = (int) fin.length() - ri[i].fileOffset;
			}
			else {
				dwRecLen = ri[i + 1].fileOffset - ri[i].fileOffset;
			}
			fin.seek(ri[i].fileOffset);
			textBuffer = new PilotDocRecord(dwRecLen);
			fin.read(textBuffer.buf);
			if (compressed) {
				textBuffer.decompress();
			}
			if (MakeDocJr.eolType != PilotDocRecord.EOLLF) {
				textBuffer.convertEOL(MakeDocJr.eolType);
			}
			bout.write(textBuffer.buf);
		}

		textArray = bout.toByteArray();
		try {
			bout.close();
		} catch (IOException ioe) {}

		title = dbHeader.name;

		/////////////////////////////////////////////////////////////
		// Handle Bookmarks if any
		// remove old ones first
		bookmarks.clear();
		if (docHeader.numRecords + 1 < dbHeader.numRecords) {
			for (int i = docHeader.numRecords + 1; i < dbHeader.numRecords; i++) {
				Bookmark bm = new Bookmark();
				fin.seek(ri[i].fileOffset);
				bm.read(fin);
				bookmarks.add(bm);
			}
		}
		try {
			fin.close();
		} catch (IOException ioe) {}
	}

	/**
	 * Encodes a PilotDoc file based on this object's text, title, and, if not
	 * null, bookmarks using this object's <code>EncodeOptions</code>.
	 * The text and title must be set first,
	 * or a NullPointerException will be thrown.
	 *
	 * @param destinationFile   the destination File to be encoded.
	 * @throws java.io.FileNotFoundException if bookmarks are handled but the
	 * bookmark file is not found
	 * @throws java.io.IOException
	 * @throws java.lang.NullPointerException if text and title are not set
	 */
	public void encode(File destinationFile)
				throws IOException {

		boolean processBookmarks = false;
		boolean deletePilotDoc = false;
		PilotDocRecord textBuffer = null;
		RandomAccessFile fout = null;
		DocumentHeader docHeader = new DocumentHeader();
		byte attribute = 0;
		if (encodeOptions.makePrivate) {
			attribute |= RecordIndex.PRIVATE;
		}
		docHeader.storyLen = textArray.length;
		textBuffer = new PilotDocRecord(docHeader.storyLen);	// allocate enough to hold entire file
		textBuffer.buf = textArray;
		destinationFile.delete();
		fout = new RandomAccessFile(destinationFile, "rw");

		docHeader.storyLen = textBuffer.convertEOL();
		if (encodeOptions.unwrap) {
			docHeader.storyLen = textBuffer.unwrap(encodeOptions.lineLength);
		}
		if (encodeOptions.compressEOL) {
			docHeader.storyLen = textBuffer.compressEOL();
		}

		byte[] uncompressedBuffer = new byte[textBuffer.length()];
		System.arraycopy(textBuffer.buf, 0, uncompressedBuffer, 0, textBuffer.length());
		String uncompressedText = new String(uncompressedBuffer);

		///////////////////////////////////////////////////
		// deal with bookmarks if necessary
		if (bookmarks.size() > 0) {
			processBookmarks = true;
		}
		///////////////////////////////////////////////////
		// auto-bookmark if specified
		if (encodeOptions.autoBookmark != null) {
			processBookmarks = true;
			int found = 0;
			while ((found = uncompressedText.indexOf(
									encodeOptions.autoBookmark, found)) > 0) {
				if (uncompressedText.charAt(found - 1) == '\n') {
					Bookmark bm = new Bookmark();
					bm.setFileOffset(found);
					int curCharPos = found += encodeOptions.autoBookmark.length();
					for (; curCharPos < uncompressedText.length(); curCharPos++) {
						if (uncompressedText.charAt(curCharPos) == '\n') {
							while (Character.isWhitespace(uncompressedText.charAt(found))) {
								found++;
							}
							bm.setName(uncompressedText.substring(found, curCharPos));
							bookmarks.add(bm);
							break;
						}
					}
					found = curCharPos;
				}
				else {
					found++;
				}
			}
		}

		DatabaseHeader dbHeader = new DatabaseHeader();
		dbHeader.name = title;
		dbHeader.setModificationDate();
		dbHeader.creatorID = encodeOptions.docID;
		dbHeader.typeID = DatabaseHeader.TEXt;

		docHeader.version = (encodeOptions.compress ? DocumentHeader.COMPRESSED : DocumentHeader.UNCOMPRESSED);
		docHeader.recordSize = DocumentHeader.textRecordSize;

		docHeader.numRecords = (short)(docHeader.storyLen / DocumentHeader.textRecordSize);
		if (docHeader.numRecords * DocumentHeader.textRecordSize < docHeader.storyLen) {
			docHeader.numRecords++;
		}

		if (processBookmarks) {
			dbHeader.numRecords = (short)(docHeader.numRecords + 1 + bookmarks.size());
		}
		else {
			dbHeader.numRecords = (short)(docHeader.numRecords + 1);
		}
		dbHeader.write(fout);

		int bookMarkIndex = docHeader.numRecords + 1;	// this will only be used if we are bookmarking

		RecordIndex[] ri = new RecordIndex[dbHeader.numRecords];
		ri[0] = new RecordIndex();
		ri[0].attribute = (byte)(attribute | encodeOptions.category);
		ri[0].write(fout);
		for (int i = 1; i < dbHeader.numRecords; i++) {
			ri[i] = new RecordIndex();
			ri[i].fileOffset = 0;
			ri[i].write(fout);
		}

		ri[0].fileOffset = (int) fout.getFilePointer();

		docHeader.write(fout);

		int totalCompressedBytes = 0;
		int processed = 0;
		int compressed = 0;
		int original = 0;

		for (int recNum = 1; recNum <= docHeader.numRecords; recNum++) {
			ri[recNum].fileOffset = (int)fout.getFilePointer();
			original = docHeader.recordSize;
			if (original + processed > uncompressedBuffer.length) {
				original = uncompressedBuffer.length - processed;
			}
			textBuffer.assign(uncompressedBuffer, original, processed);
			processed += original;
			if (encodeOptions.compress) {
				compressed = textBuffer.compress();
				totalCompressedBytes += compressed;
			}
			fout.write(textBuffer.buf);
		}

		// process the bookmarks
		if (processBookmarks) {
			if (encodeOptions.sortBookmarks == 1) { // by location
				sortBookmarksByLocation();
			}
			else if (encodeOptions.sortBookmarks == 2) { // alphabetically
				sortBookmarksByName();
			}
			for (int bm = 0; bm < bookmarks.size(); bm++) {
				if (((Bookmark)bookmarks.get(bm)).getFileOffset() == -1) {
					deletePilotDoc = true;
				}
				else {
					ri[bookMarkIndex + bm].fileOffset = (int)fout.getFilePointer();
					((Bookmark)bookmarks.get(bm)).write(fout);
				}
			}
		}

		// re-write the record index array, now with the correct file offsets
		fout.seek(DatabaseHeader.getSize());
		for (int i = 0; i < ri.length; i++) {
			ri[i].write(fout);
		}
		fout.close();

		if (deletePilotDoc) {
			destinationFile.delete();
		}

	}

	/**
	 * Gets the <code>EncodeOptions</code> for this MakeDocJr.
	 *
	 * @return EncodeOptions used by this MakeDocJr
	 */
	public EncodeOptions getEncodeOptions() {
		return encodeOptions;
	}

	/**
	 * Sets the <code>EncodeOptions</code> for this MakeDocJr.
	 *
	 * @param encodeOptions     options for encoding text into a DOC file.
	 */
	public void setEncodeOptions(EncodeOptions encodeOptions) {
		this.encodeOptions = encodeOptions;
	}

	/**
	 * Gets the title of the DOC file. This is the title as appears in a PDA,
	 * not the filename.
	 *
	 * @return the title of the DOC
	 */
	public String getTitle() {
		return title;
	}

	/**
	 * A title is required to make a valid DOC. Titles longer than 31 characters
	 * will be truncated.
	 *
	 * @param title     the title of the DOC to appear in the PDA
	 */
	public void setTitle(String title) {
		if (title.length() > 31)
			this.title = title.substring(0, 31);
		else this.title = title;
	}

	/**
	 * Gets the text that was decoded or that will be encoded.
	 *
	 * @return the text of this MakeDocJr
	 */
	public String getText() {
		return new String(textArray);
	}

	/**
	 * Sets the text to encode. Uses the default character encoding
	 * of the Java runtime environment.
	 *
	 * @param text  the text to encode
	 */
	public void setText(String text) {
		textArray = text.getBytes();
	}

	/**
	 * Sets the text to encode based on the specified character encoding.
	 * Don't confuse encoding a DOC file with the character encoding of unicode.
	 *
	 * @param text      the text of this MakeDocJr
	 * @param encoding  a String representing the character encoding to use. For example, Cp1252.
	 * @throws java.io.UnsupportedEncodingException     if the encoding is not supported.
	 */
	public void setText(String text, String encoding)
			throws UnsupportedEncodingException {
		textArray = text.getBytes(encoding);
	}

	/**
	 * Gets the text that was decoded or that will be encoded as an array of bytes.
	 *
	 * @return the text of this MakeDocJr as an array of bytes
	 */
	public byte[] getTextArray() {
		return textArray;
	}

	/**
	 * Sets the text that will be encoded.
	 *
	 * @param textArray   the text of this MakeDocJr as an array of bytes
	 */
	public void setText(byte[] textArray) {
		this.textArray = textArray;
	}

	/**
	 * Adds a Bookmark to this MakeDocJr.
	 *
	 * @param bm  a Bookmark to be included when encoded
	 */
	public void addBookmark(Bookmark bm) {
		bookmarks.add(bm);
	}

	/**
	 * Deletes a Bookmark from this MakeDocJr.
	 *
	 * @param bm  a Bookmark to be deleted before encoding
	 */
	public void removeBookmark(Bookmark bm) {
		bookmarks.remove(bm);
	}

	/**
	 * Gets all of the Bookmarks associated with this MakeDocJr.
	 *
	 * @return an ArrayList of all Bookmark objects, empty if there are none
	 */
	public ArrayList getBookmarks() {
		return bookmarks;
	}

	/**
	 * Sets all of the Bookmarks associated with this MakeDocJr.
	 *
	 * @param bookmarks    an ArrayList of Bookmark objects
	 */
	public void setBookmarks(ArrayList bookmarks) {
		if ( bookmarks == null ) {
			this.bookmarks.clear();
		} else {
			this.bookmarks = bookmarks;
		}
	}

	/**
	 * Specifies that bookmarks should be sorted by location, not the name of the
	 * bookmark.
	 */
	public void sortBookmarksByLocation() {
		for (int i = 0; i < bookmarks.size(); i++) {
			((Bookmark)bookmarks.get(i)).setCompareNamesNotLocation(false);
		}
		Collections.sort(bookmarks);
	}

	/**
	 * Specifies that bookmarks should be sorted by the name of the
	 * bookmark, not its location.
	 */
	public void sortBookmarksByName() {
		for (int i = 0; i < bookmarks.size(); i++) {
			((Bookmark)bookmarks.get(i)).setCompareNamesNotLocation(true);
		}
		Collections.sort(bookmarks);
	}

	/**
	 * Checks if a bookmark exists with the specified name.
	 *
	 * @param bookmarkName  a String representing the name of the bookmark
	 * @return whether a bookmark with the specified name exists or not
	 */
	public boolean bookmarkNameExists(String bookmarkName) {
		if (bookmarks == null)
			return false;
		for (int i = 0; i < bookmarks.size(); i++) {
			if (bookmarkName.equals(((Bookmark)bookmarks.get(i)).getName()))
				return true;
		}
		return false;
	}

	/**
	 * Gets the type of end of line character(s) this MakeDocJr looks for.
	 *
	 * @return a PilotDocRecord.EOLLF, PilotDocRecord.EOLCR, or PilotDocRecord.EOLCRLF
	 */
	public static int getEolType() {
		return eolType;
	}

	/**
	 * Overrides Object.clone
	 */
	public Object clone() {
		// this should really be a copy constructor instead of clone

		// super.clone causes errors (in editor) -> make new instead
		MakeDocJr mdj = new MakeDocJr();

		if (textArray != null)
			mdj.textArray = (byte[]) textArray.clone();
		for (int i = 0; i < bookmarks.size(); i++) {
			mdj.addBookmark(new Bookmark((Bookmark) bookmarks.get(i)));
		}
		mdj.encodeOptions = (EncodeOptions) encodeOptions.clone();
		return mdj;
	}

	/*
	public static void main(String[] args) throws Exception {
		//Here's the most basic usage:

		// the file to be encoded and decoded
		File file = new File("c:\\temp\\encoded.pdb");

		// create a MakeDocJr object
		MakeDocJr makeDocJr = new MakeDocJr();

		// your code must make sure the title and text are set -
		// give appropriate messages if user did not supply
		makeDocJr.setTitle("The Title");
		makeDocJr.setText("This is the body of the DOC file.");

		// default options create a compressed DOC file
		// modify the EncodeOptions object for other options
		makeDocJr.encode(file);

		// clear text to verify it is really being decoded
		makeDocJr.setTitle("");
		makeDocJr.setText("");

		// read a DOC file
		makeDocJr.decode(file);

		System.out.println(makeDocJr.getTitle());
		System.out.println(makeDocJr.getText());
		System.out.println(file.delete());

	}
	*/
}


